#include "ColorDetector.h"
#include "math.h"


ColorDetector::ColorDetector(GroveColourSensor *sensor, uint16_t threshold, uint16_t varianceThreshold, uint16_t diffThreshold, uint16_t samplesPerReading, uint16_t readingsPerEvent)
    :sensor(sensor), threshold(threshold), varianceThreshold(varianceThreshold), diffThreshold(diffThreshold), samplesPerReading(samplesPerReading), readingsPerEvent(readingsPerEvent),
    changeDetected(false), curBuffPointer(0) {
        // Empty
}



RGBC ColorDetector::takeSamples(uint16_t numSamples) {
    uint32_t tot_r = 0;
    uint32_t tot_g = 0;
    uint32_t tot_b = 0;
    uint32_t tot_c = 0;
    
    __disable_irq();
    for (uint32_t i = curBuffPointer; (i < curBuffPointer + numSamples) && (i < CD_BUFF_LEN); i++) {
        sensor->readBlock(sample_buf + i);
        tot_r += sample_buf[i].ch.red;
        tot_g += sample_buf[i].ch.green;
        tot_b += sample_buf[i].ch.blue;
        tot_c += sample_buf[i].ch.clear;
    }
    __enable_irq();
    curBuffPointer += numSamples;
    
    RGBC ret;
    ret.ch.red = tot_r / numSamples;
    ret.ch.green = tot_g / numSamples;
    ret.ch.blue = tot_b / numSamples;
    ret.ch.clear = tot_c / numSamples;
    
    return ret;
}

void ColorDetector::calculateVariance(RGBC *variance, RGBC *buffer, int numSamples, RGBC mean) {
    double diff_r = 0;
    double diff_g = 0;
    double diff_b = 0;
    double diff_c = 0;
    for (int i = 0; i < numSamples; i++) {
        double tmp = (double)sample_buf[i].ch.red - (double)mean.ch.red;
        diff_r += tmp * tmp;
        
        tmp = (double)sample_buf[i].ch.green - (double)mean.ch.green;
        diff_g += tmp * tmp;
        
        tmp = (double)sample_buf[i].ch.blue - (double)mean.ch.blue;
        diff_b += tmp * tmp;
        
        tmp = (double)sample_buf[i].ch.clear - (double)mean.ch.clear;
        diff_c += tmp * tmp;
    }
    
    
    diff_r = sqrt(diff_r/numSamples);
    diff_g = sqrt(diff_g/numSamples);
    diff_b = sqrt(diff_b/numSamples);
    diff_c = sqrt(diff_c/numSamples);
    
    
    variance->ch.red = (uint16_t)diff_r;
    variance->ch.green = (uint16_t)diff_g;
    variance->ch.blue = (uint16_t)diff_b;
    variance->ch.clear = (uint16_t)diff_c;
}


void ColorDetector::setBaseline() {
    RGBC variance;
    bool cont = true;
    
    while(cont) {
        baselineAvg = takeSamples(samplesPerReading);
        calculateVariance(&variance, sample_buf, curBuffPointer, baselineAvg);
        cont = !isSteady(&variance);
    }
}

bool ColorDetector::isSteady(RGBC *variance) {
    bool steady = true;
    for (int i = 0; i < 4; i++) {
        if (variance->data[i] > varianceThreshold) {
            steady = false;
        }
    }
    return steady;
}

int ColorDetector::sample() {
    RGBC variance;
    int retVal = 0;
    
    lastAvg = takeSamples(samplesPerReading);    
    calculateVariance(&variance, sample_buf, curBuffPointer, lastAvg);
    
    if (isSteady(&variance)) {
        // Calculate diff
        int32_t totalDiff = 0;
        
        for (int i = 0; i < 4; i++) {
            int diff = abs(((int)lastAvg.data[i]) - ((int)baselineAvg.data[i]));
            
            if (diff > 128) {
                diff = 255 - diff;
            }
            
            totalDiff += diff;
        }
        
        for (int i = 0; i < readingsPerEvent - 1; i++) {
            diffBuff[i] = diffBuff[i + 1];
        }
        
        diffBuff[readingsPerEvent - 1] = totalDiff;
        
        printf("totalDiff: %d\r\n", totalDiff);
        if (totalDiff > threshold) {
             if (getTotalPastDiff() < diffThreshold && !changeDetected) {
                 changeDetected = true;
                 retVal = curBuffPointer;
             }
        } else {
            if (changeDetected) {
                changeDetected = false;
            }
        }
    }
    
    curBuffPointer = 0;
    return retVal;
}

int32_t ColorDetector::getTotalPastDiff() {
    int32_t totalDiff = 0;
    for (int i = 0; i < readingsPerEvent - 1; i++) {
        totalDiff += abs(diffBuff[readingsPerEvent - 1] - diffBuff[i]);
    }
    return totalDiff;
}
RGBC* ColorDetector::getBuffer() {
    return sample_buf;
}