#ifndef _COLOR_DETECTOR_H_
#define _COLOR_DETECTOR_H_

#include "mbed.h"
#include "GroveColourSensor.h"

#define CD_BUFF_LEN 512
#define DIFF_BUFF_LEN 4


/** ColorDetector class.
 *  Used for detecting changes in color readings from a Grove colour sensor: http://www.seeedstudio.com/depot/Grove-I2C-Color-Sensor-p-854.html.
 */
class ColorDetector {
public:
    /**
     *    Constructor.
     *
     *    @param sensor Pointer to instantiated and powered up Grove colour sensor.
     *    @param threshold Total change in channel values needed to be considered a "differnet" reading from the baseline. (default 15)
     *    @param varianceThreshold The maximum difference in variance between samples that is allowed for a "steady" reading. (default 3)
     *    @param diffThreshold The maximum difference between past readings that is allowed for a "consistent" reading. (default 2)
     *    @param samplesPerReading Specifies the number of samples to take per sample() call. (default 256)
     *    @param readingsPerEvent Number of readings that must be "steady", "consistent", and "different" before triggering a "change detected" event. (default 4)
     *        NOTE: Must be less than or equal to DIFF_BUFF_LEN (default 4).
     */
    ColorDetector(GroveColourSensor *sensor, uint16_t threshold = 15, uint16_t varianceThreshold = 3, uint16_t diffThreshold = 2, uint16_t samplesPerReading = 256, uint16_t readingsPerEvent = 4);
    
    /**
     *    This establishes the average "resting" channel values. This should be called before calling sample().
     */
    void setBaseline();
    
    /**
     *    Takes a new reading from the sensor. First, it checks to see if the variance between the samples in the reading is
     *    less than the varianceThreshold. If it is, the reading is considered "steady" and a candidate "change detected" event.
     *    Next, it checks to see if the total average difference between the last samples is greater
     
     It trackes whether the change in color rises above the threshold given in the constructor.
     *    If the total change in color rises above and then falls bellow this threshold, or if the sample buffer is filled,
     *    this will return a value greater than 0 that indicates how many samples are in the buffer. Otherwise, it returns 0.
     */
    int sample();
    
    /**
     *    Returns a pointer to the sample buffer (pointer does not change).
     */
    RGBC* getBuffer();
    
private:    
    RGBC sample_buf[CD_BUFF_LEN];
    GroveColourSensor *sensor;
    uint16_t threshold;
    uint16_t varianceThreshold;
    uint16_t diffThreshold;
    RGBC baselineAvg;
    RGBC lastAvg;
    uint32_t curBuffPointer;
    bool changeDetected;
    uint16_t samplesPerReading;
    int32_t diffBuff[DIFF_BUFF_LEN];
    uint16_t readingsPerEvent;
    
    /**
     *    Takes the specified number of samples and returns the average color channel values.
     *
     *    @param numSamples Number of samples to take.
     */
    RGBC takeSamples(uint16_t numSamples);
    
    /**
     *    Calculates the variance of a sample given the mean.
     *
     *    @param variance pointer to an RGBC instance where the variances of each channel will be stored
     *    @param buffer pointer to the buffer that stores 
     *    @param numSamples number of samples to read from the buffer
     *    @param mean the mean value of the samples in the buffer
     */
    void calculateVariance(RGBC *variance, RGBC *buffer, int numSamples, RGBC mean);
    
    /**
     *    Determines if each channel of the given variance is within the class's varianceThreshold limit
     *
     *    @param variance pointer to an RGBC instance that constains variance data (see calculateVariance)
     */
    bool isSteady(RGBC *variance);
    
    /**
     *    Returns the absolute difference between the past differences in average color measurement
     */
    int32_t getTotalPastDiff();
};
    
#endif