/////////////////////////////////////////////////////////////
// APP 3                                                   //
//                                                         //
// Université de Sherbrooke                                //
// Génie informatique                                      //
// Session 5, Hiver 2017                                   //
//                                                         //
// Date:    14 février 2017                                //
//                                                         //
// Auteurs: Maxime Dupuis,       dupm2216                  //
//          Bruno Allaire-Lemay, allb2701                  //
/////////////////////////////////////////////////////////////

#include "xbee.h"
#include <cassert>

DigitalOut led_1(LED1);
Mail<ingoing_value_t, 30> parsed_frames;
RawSerial xbee(p13, p14);
Mutex mutex;
DigitalOut error_led(p6);
Thread error_led_thread;

const int FRAME_SPECIFIC_DATA_BEGIN_BEFORE_ADDRESS[2] = {0x10, 0x01};
const int FRAME_SPECIFIC_DATA_BEGIN_AFTER_ADDRESS[4] = {0xFF, 0xFE, 0x00, 0x00};
const int AT_COMMAND_LED_FRAME_SPECIFIC_DATA_BEGIN[15] = {0x17, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFE, 0x02, 0x50, 0x32};
const char LED_COMMAND_POWER_ON = 0x05;
const char LED_COMMAND_POWER_OFF = 0x04;
const int RECEIVE_PACKET_MESSAGE_START_INDEX = 15;
const int RECEIVE_PACKET_SOURCE_ADDRESS_START_INDEX = 4;

const char START_DELIMITER = 0x7E;

bool is_character_that_needs_escape(const char character)
{
    //printf("%X\r\n", character);
    switch(character)
    {
        case 0x7E: return true;
        case 0x7D: return true;
        case 0x11: return true;
        case 0x13: return true;
        default: return false;
    }
}

void send_message_via_xbee(const char* message, const int length, int address[8])
{
    const vector<char> transmit_request = generate_transmit_request(message, length, address);
    
    mutex.lock();
    for(int i=0; i<transmit_request.size(); i++)
    {
        xbee.putc(transmit_request[i]);
    }
    mutex.unlock();
}

vector<char> generate_transmit_request(const char* message, const int length, int address[8])
{
    vector<char> request;
    
    unsigned char checksum = 0xFF;
    
    request.push_back(START_DELIMITER);
    
    const uint16_t frame_length = 0x0E + length;
    const uint8_t frame_length_msb = frame_length >> 8;
    const uint8_t frame_length_lsb = (frame_length << 8) >> 8;
    request.push_back(frame_length_msb);
    request.push_back(frame_length_lsb);
    
    for(int i=0; i<2; i++)
    {
        request.push_back(FRAME_SPECIFIC_DATA_BEGIN_BEFORE_ADDRESS[i]);
        checksum -= FRAME_SPECIFIC_DATA_BEGIN_BEFORE_ADDRESS[i];
    }
    
    for(int i=0; i<8; i++)
    {
        request.push_back(address[i]);
        checksum -= address[i];
    }
    
    for(int i=0; i<4; i++)
    {
        request.push_back(FRAME_SPECIFIC_DATA_BEGIN_AFTER_ADDRESS[i]);
        checksum -= FRAME_SPECIFIC_DATA_BEGIN_AFTER_ADDRESS[i];
    }
    
    for(int i=0; i<length; i++)
    {
        request.push_back(message[i]);
        checksum -= message[i];
    }
    
    request.push_back(checksum);
    
    return request;
}

vector<char> generate_led_command(const bool power_on)
{
    vector<char> request;
    
    unsigned char checksum = 0xFF;
    
    request.push_back(START_DELIMITER);
    
    const uint16_t frame_length = 0x10;
    const uint8_t frame_length_msb = frame_length >> 8;
    const uint8_t frame_length_lsb = (frame_length << 8) >> 8;
    request.push_back(frame_length_msb);
    request.push_back(frame_length_lsb);
    
    for(int i=0; i<15; i++)
    {
        request.push_back(AT_COMMAND_LED_FRAME_SPECIFIC_DATA_BEGIN[i]);
        checksum -= AT_COMMAND_LED_FRAME_SPECIFIC_DATA_BEGIN[i];
    }
    
    const char led_power_state = power_on ? LED_COMMAND_POWER_ON : LED_COMMAND_POWER_OFF;
    request.push_back(led_power_state);
    checksum -= led_power_state;
    
    request.push_back(checksum);
    
    return request;
}

void read_frame()
{
    while(true)
    {
        while(xbee.getc() != 0x7E);
        
        vector<char> frame;
        
        frame.push_back(0x7E);
        
        const uint8_t frame_size_msb = xbee.getc();
        const uint8_t frame_size_lsb = xbee.getc();
        frame.push_back(frame_size_msb);
        frame.push_back(frame_size_lsb);
        const uint16_t frame_size = (frame_size_msb << 8) + frame_size_lsb;
        
        for(int i=0; i<frame_size + 1; i++)
        {
            frame.push_back(xbee.getc());
        }
        
        handle_frame(frame);
    }
}

vector<char> parse_receive_packet(const vector<char>& frame)
{
    vector<char> message;
    for(int i=0; i<8; i++)
    {
        message.push_back(frame.at(RECEIVE_PACKET_SOURCE_ADDRESS_START_INDEX + i));
    }
    for(int i=0; i< frame.size() - RECEIVE_PACKET_MESSAGE_START_INDEX; i++)
    {
        message.push_back(frame.at(RECEIVE_PACKET_MESSAGE_START_INDEX + i));
    }
    return message;
}

vector<char> parse_transmit_status(const vector<char>& frame)
{    
    vector<char> relevant_content;

    const char delivery_status = frame.at(8);    
    relevant_content.push_back(delivery_status);
    
    return relevant_content;
}

vector<char> parse_at_command_response(const vector<char>& frame)
{    
    vector<char> relevant_content;

    const char command_status = frame.at(7);    
    relevant_content.push_back(command_status);
    
    return relevant_content;
}

vector<char> parse_remote_command_response(const vector<char>& frame)
{    
    vector<char> relevant_content;

    const char command_status = frame.at(17);    
    relevant_content.push_back(command_status);
    
    return relevant_content;
}

void manage_error_led()
{
    while(true)
    {
        osSignalWait(0x1, osWaitForever);
        error_led = 1;
        wait_ms(1000);
        error_led = 0;
    }
}

void send_blink_led_at_command(const bool toggle_current_command)
{
    static bool is_current_command_turn_on = false;
    if(toggle_current_command)
    {
        is_current_command_turn_on = !is_current_command_turn_on;   
    }
    
    const vector<char> led_command = generate_led_command(is_current_command_turn_on);
    for(int i=0; i<led_command.size(); i++)
    {
        xbee.putc(led_command[i]);
    }
}

void handle_parsed_frames_from_mailbox(void const *args)
{
    while(true)
    {
        osEvent event = parsed_frames.get();
        assert(event.status == osEventMail);
        
        ingoing_value_t *parsed_frame = (ingoing_value_t*)event.value.p;
        parse_nerfus_message(ingoing_value_to_vector(*parsed_frame), (void(*)(vector<uint8_t>, int*))args);
        parsed_frames.free(parsed_frame);
    }
}

vector<char> ingoing_value_to_vector(const ingoing_value_t& value)
{
    vector<char> result;
    for(int i=0; i<value.size; i++)
    {
        result.push_back(value.content[i]);
    }
    return result;
}

void handle_frame(const vector<char>& frame)
{
    ingoing_value_t *parsed_frame = parsed_frames.alloc();
    
    const vector<char> parsed_frame_vector = parse_frame(frame);
    for(int i=0; i<parsed_frame_vector.size(); i++)
    {
        parsed_frame->content[i] = parsed_frame_vector.at(i);
    }
    parsed_frame->size = parsed_frame_vector.size();
    
    parsed_frames.put(parsed_frame);
}

vector<char> parse_frame(const vector<char>& frame)
{   
    vector<char> parsed_frame;
    
    const char frame_type = frame.at(3);
    parsed_frame.push_back(frame_type);
  
    vector<char> parsed_frame_relevant_content;
  
    switch(frame_type)
    {
        case FRAME_TYPE_RECEIVE_PACKET: 
        {
            parsed_frame_relevant_content = parse_receive_packet(frame);
            break;
        }
        case FRAME_TYPE_TRANSMIT_STATUS: 
        {
            parsed_frame_relevant_content = parse_transmit_status(frame);
            break;
        }
        case FRAME_TYPE_AT_COMMAND_RESPONSE:
        {
            parsed_frame_relevant_content = parse_at_command_response(frame);
            break;
        }
        case FRAME_TYPE_REMOTE_COMMAND_RESPONSE:
        {
            parsed_frame_relevant_content = parse_remote_command_response(frame);
            break;
        }
        default:
            printf("Unsupported frame:\r\n");
            for(int i=0; i<frame.size(); i++)
            {
                printf("Bit #%d: %d\r\n", i, frame.at(i));
            }
    }
    
    for(vector<char>::iterator it = parsed_frame_relevant_content.begin(); it < parsed_frame_relevant_content.end(); it++)
    {
        parsed_frame.push_back(*it);
    }
    
    return parsed_frame;
}

void parse_nerfus_message(const vector<char>& parsed_frame, void (*callback)(vector<uint8_t>, int*))
{    
    if(parsed_frame[0] == FRAME_TYPE_RECEIVE_PACKET)
    {
        int address[8];
        for(int i=1; i<9; i++)
        {
            address[i-1] = parsed_frame[i];
        }
        vector<uint8_t> message;
        for(int i=9; i<13; i++)
        {
            message.push_back(parsed_frame[i]);
        }
        callback(message, address);
    }
}

void parsed_button_event_frame_to_string(const vector<char>& parsed_frame, char* readable_string_output)
{
    const char button_state = parsed_frame[2];
    switch(button_state)
    {
        case BUTTON_RELEASED:
            strcpy(readable_string_output, "Button state: released");
            break;
            
        case BUTTON_PRESSED:
            strcpy(readable_string_output, "Button state: pressed");
            break;
            
        default:
            strcpy(readable_string_output, "Button state: invalid state");
            break;
    }
}

void parsed_accelerometer_event_frame_to_string(const vector<char>& parsed_frame, char* readable_string_output)
{
    const uint8_t x = parsed_frame[2];
    const uint8_t y = parsed_frame[3];
    const uint8_t z = parsed_frame[4];
    sprintf(readable_string_output, "Accelerometer state: x=0x%X, y=0x%X, z=0x%X", x, y, z);
}

vector<string> read_file(string path)
{
    LocalFileSystem local("local");  
    vector<string> result;
    char buffer[128] = "";
    FILE *fp = fopen(path.c_str(), "r");
    bool ret = (fgets(buffer, 64, fp)) ;
    
    string value(buffer);
    string first_value = "";
    string second_value = "";
    bool which_value = true;
    
    for(int i=0; i<value.size(); i++)
    {
        if(buffer[i] == ';')
        {
            which_value = false;
        }
        else
        {
            if(which_value)
            {
                first_value += buffer[i];
            }
            else
            {
                second_value += buffer[i];
            }
        }
    }
    result.push_back(first_value);
    result.push_back(second_value);
    return result;
}

char hexa_char_to_dec(char hexa_char)
{
    if('0' <= hexa_char && hexa_char <= '9')
    {
        return hexa_char - 48;
    }
    if('A' <= hexa_char && hexa_char <= 'F')
    {
        return hexa_char - 55;
    }
    if('a' <= hexa_char && hexa_char <= 'f')
    {
        return hexa_char - 87;
    }
    assert(false && "Wtf");
    return hexa_char;
}

vector<char> string_to_data(string pan_id)
{
    vector<char> result;
    
    for(int i=0; i < pan_id.size(); i += 2)
    {
        const char dec_val = ( hexa_char_to_dec(pan_id[i]) ) * 16 + hexa_char_to_dec(pan_id[i]);
        result.push_back(dec_val);
    }
    
    return result;
}

void set_pan_id(string pan_id)
{
    const int pan_id_beginning[7] = {0x7E, 0x00, 0x06, 0x08, 0x01, 0x49, 0x44};
    unsigned char checksum = 0xFF;
    
    xbee.putc(pan_id_beginning[0]);
    xbee.putc(pan_id_beginning[1]);
    xbee.putc(pan_id_beginning[2]);
    
    for(int i=3; i<7; i++)
    {
        checksum -= pan_id_beginning[i];
        xbee.putc(pan_id_beginning[i]);
    }
    
    vector<char> pan_id_char = string_to_data(pan_id);
    for(int i=0; i<pan_id_char.size(); i++)
    {
        checksum -= pan_id_char.at(i);
        xbee.putc(pan_id_char.at(i));
    }
    
    xbee.putc(checksum);
}