#include "gmock/gmock.h"
#include "gtest/gtest.h"

class Adder
{
    public:
        virtual ~Adder() {}
        virtual int add(int a, int b)
        {
            return a + b;
        }
        virtual int add_array(int some_array[8], int count)
        {
            int sum = 0;
            for(int i=0; i<count; i++)
            {
                sum += some_array[i];
            }
            return sum;
        }
	virtual int add_vector(std::vector<int> some_vector)
        {
            int sum = 0;
            for(int i=0; i<some_vector.size(); i++)
            {
                sum += some_vector[i];
            }
            return sum;
        }
};

class MockAdder : public Adder
{
    public:
        MOCK_METHOD2(add, int(int a, int b));
        MOCK_METHOD2(add_array, int(int some_array[8], int count));
        MOCK_METHOD1(add_vector, int(std::vector<int> some_vector));
};

TEST(GoogleMockTest, ExpectCalled)
{
    MockAdder adder;

    EXPECT_CALL(adder, add(1, 2));

    adder.add(1, 2);
}

TEST(GoogleMockTest, DefineReturnedValue)
{
    using ::testing::Return;
    MockAdder adder;

    EXPECT_CALL(adder, add(1, 2))
        .WillOnce(Return(42));

    ASSERT_EQ(42, adder.add(1, 2));
}

int expected_array[8] = {1,2,3,4,5,6,7,8};
TEST(GoogleMockTest, AssertCalledWithArrayPointer)
{
    MockAdder adder;

    EXPECT_CALL(adder, add_array(expected_array, 8))
        .WillOnce(::testing::Return(36));

    adder.add_array(expected_array, 8);
}

TEST(GoogleMockTest, UseMatcherOnMethodParameter)
{
    MockAdder adder;

    std::vector<int> expected_vector;
    expected_vector.push_back(1);
    expected_vector.push_back(2);
    expected_vector.push_back(3);

    std::vector<int> some_vector;
    some_vector.push_back(1);
    some_vector.push_back(2);
    some_vector.push_back(3);

    EXPECT_CALL(adder, add_vector(::testing::ContainerEq(expected_vector)))
        .WillOnce(::testing::Return(6));

    adder.add_vector(some_vector);
}

TEST(GoogleMockTest, UseMatcherOnArrayPointerPointedContent)
{
    MockAdder adder;

    int expected_array[8] = {1,2,3,4,5,6,7,8};

    EXPECT_CALL(adder, add_array(::testing::_, 8))
	.With(::testing::Args<0,1>(::testing::ContainerEq(expected_array)))
        .WillOnce(::testing::Return(36));

    int actual_array[8] = {1,2,3,4,5,6,7,8};
    adder.add_array(actual_array, 8);
}
