#include "TargetManager.hpp"
#include <cassert>

TargetManager::TargetManager(std::vector<TargetInterface*>& targets,
		std::vector<NerfusTickerInterface*>& nerfus_tickers) :
	targets(targets),
	nerfus_tickers(nerfus_tickers)
{
	for(int i=0; i<targets.size(); i++)
	{
		is_active_target.push_back(false);
	}
}

void TargetManager::execute(const TargetInfo& target_info)
{
	assert(0 <= target_info.id && target_info.id < targets.size() && "ID is out of range");
	TargetInterface& target = *(targets[target_info.id]);

	if(target_info.type == TARGET_TYPE_ALLY)
	{
		target.ally_command();
	}
	else if(target_info.type == TARGET_TYPE_ENEMY)
	{
		target.enemy_command();
	}
	else
	{
		assert(false && "Target type invalid");
	}

	is_active_target[target_info.id] = true;
	nerfus_tickers[target_info.id]->start(target_info.timeout_ms);
}

void TargetManager::execute(const std::vector<uint8_t>& target_info_bytes, int *address)
{
	const int id = target_info_bytes[0];
	const TargetType type = (target_info_bytes[1] == 0) ? TARGET_TYPE_ALLY : TARGET_TYPE_ENEMY;
	const int timeout_msb = target_info_bytes[2];
	const int timeout_lsb = target_info_bytes[3];
	const int timeout = (timeout_msb << 8) + timeout_lsb;
	const TargetInfo target_info = make_TargetInfo(id, type, timeout);
	execute(target_info);
}

void TargetManager::target_hit(int target_number)
{
	const int time_ms = nerfus_tickers[target_number]->get_time_ms();
	if(is_active_target[target_number] && time_ms > 750)
	{
		is_active_target[target_number] = false;
		targets[target_number]->hit(time_ms);
		nerfus_tickers[target_number]->stop();
	}
}

void TargetManager::target_missed(int target_number)
{
	if(is_active_target[target_number])
	{
		is_active_target[target_number] = false;
		targets[target_number]->timeout(nerfus_tickers[target_number]->get_time_ms());
		nerfus_tickers[target_number]->stop();
	}
}
