#include "gmock/gmock.h"
#include "gtest/gtest.h"

#include "mocks/MockNerfusTicker.hpp"
#include "mocks/MockTarget.hpp"
#include "TargetManager.hpp"

#include "stdint.h"

class TargetManagerTest : public ::testing::Test
{
    public:
        TargetManagerTest()
        {
            targets.push_back(&mock_target_a);
            targets.push_back(&mock_target_b);

            tickers.push_back(&mock_nerfus_ticker_a);
            tickers.push_back(&mock_nerfus_ticker_b);
        }

        MockTarget mock_target_a;
        MockTarget mock_target_b;
        std::vector<TargetInterface*> targets;

        MockNerfusTicker mock_nerfus_ticker_a;
        MockNerfusTicker mock_nerfus_ticker_b;
        std::vector<NerfusTickerInterface*> tickers;
};

TEST_F(TargetManagerTest, LibraryBuilds)
{
    TargetManager target_manager(targets, tickers);
}

TEST_F(TargetManagerTest, WhenNoTargetIsActiveThenCallsToTargetHitAndTargetMissedDoNothing)
{
    TargetManager target_manager(targets, tickers);

    target_manager.target_missed(0);
    target_manager.target_hit(0);
}

TEST_F(TargetManagerTest, WhenExecutingAllyTargetThenUseTheCorrectTargetTypeAndTimeout)
{
    EXPECT_CALL(mock_target_a, ally_command());
    EXPECT_CALL(mock_nerfus_ticker_a, start(1000));

    TargetManager target_manager(targets, tickers);
    target_manager.execute(make_TargetInfo(0, TARGET_TYPE_ALLY, 1000));
}

TEST_F(TargetManagerTest, WhenExecutingEnemyTargetThenUseTheCorrectTargetTypeAndTimeout)
{
    EXPECT_CALL(mock_target_b, enemy_command());
    EXPECT_CALL(mock_nerfus_ticker_b, start(2000));

    TargetManager target_manager(targets, tickers);
    target_manager.execute(make_TargetInfo(1, TARGET_TYPE_ENEMY, 2000));
}

TEST_F(TargetManagerTest, WhenActiveTargetIsHitThenSendTheEvent)
{
    EXPECT_CALL(mock_target_a, ally_command());
    EXPECT_CALL(mock_nerfus_ticker_a, start(1200));
    EXPECT_CALL(mock_nerfus_ticker_a, get_time_ms())
        .WillOnce(::testing::Return(1200));
    EXPECT_CALL(mock_nerfus_ticker_a, stop());
    EXPECT_CALL(mock_target_a, hit(1200));

    TargetManager target_manager(targets, tickers);
    target_manager.execute(make_TargetInfo(0, TARGET_TYPE_ALLY, 1200));

    target_manager.target_hit(0);
}

TEST_F(TargetManagerTest, WhenActiveTargetIsHitTooSoonBefore750MsThenIgnoreTheEvent)
{
    EXPECT_CALL(mock_target_a, ally_command());
    EXPECT_CALL(mock_nerfus_ticker_a, start(700));
    EXPECT_CALL(mock_nerfus_ticker_a, get_time_ms())
        .WillOnce(::testing::Return(700));

    TargetManager target_manager(targets, tickers);
    target_manager.execute(make_TargetInfo(0, TARGET_TYPE_ALLY, 700));

    target_manager.target_hit(0);
}

TEST_F(TargetManagerTest, WhenActiveTargetIsMissedThenSendTheEvent)
{
    EXPECT_CALL(mock_target_a, ally_command());
    EXPECT_CALL(mock_nerfus_ticker_a, start(1000));
    EXPECT_CALL(mock_nerfus_ticker_a, get_time_ms())
        .WillOnce(::testing::Return(42));
    EXPECT_CALL(mock_nerfus_ticker_a, stop());
    EXPECT_CALL(mock_target_a, timeout(42));

    TargetManager target_manager(targets, tickers);
    target_manager.execute(make_TargetInfo(0, TARGET_TYPE_ALLY, 1000));

    target_manager.target_missed(0);
}

TEST_F(TargetManagerTest, WhenReceivingCommandInBytesThenExecuteIt)
{
    TargetManager target_manager(targets, tickers);

    EXPECT_CALL(mock_target_b, ally_command());
    EXPECT_CALL(mock_nerfus_ticker_b, start(0x1234));

    std::vector<uint8_t> target_info_bytes;
    target_info_bytes.push_back(0x01); //Second target
    target_info_bytes.push_back(0x00); //Mode ally
    target_info_bytes.push_back(0x12); //Timeout MSB
    target_info_bytes.push_back(0x34); //Timeout MSB

    target_manager.execute(target_info_bytes);
}

