#include "gmock/gmock.h"
#include "gtest/gtest.h"

#include "Target.hpp"
#include "mocks/MockServomotor.hpp"
#include "mocks/MockLedController.hpp"
#include "mocks/MockXbeeTransmitter.hpp"

class TargetTest : public ::testing::Test
{
    public:
        TargetTest() :
            target(mock_servomotor, mock_ally_leds, mock_enemy_leds, mock_xbee_transmitter, 32)
        {
        }

        MockServomotor mock_servomotor;
        MockLedController mock_ally_leds;
        MockLedController mock_enemy_leds;
        MockXbeeTransmitter mock_xbee_transmitter;
        Target target;
};

TEST_F(TargetTest, AllyCommandRaisesTarget)
{
    EXPECT_CALL(mock_ally_leds, turn_on())
        .Times(1);

    EXPECT_CALL(mock_servomotor, set_position_up())
        .Times(1);

    target.ally_command();
}

TEST_F(TargetTest, EnemyCommandRaisesTargetAndTurnsOnLED)
{
    EXPECT_CALL(mock_enemy_leds, turn_on())
        .Times(1);

    EXPECT_CALL(mock_servomotor, set_position_up())
        .Times(1);

    target.enemy_command();
}

TEST_F(TargetTest, WhenTargetTimesOutThenTransmitRightMessage)
{
    uint8_t expected_message[4];
    target.generate_message(32, Target::RESULT_MISSED, 42, expected_message);

    EXPECT_CALL(mock_xbee_transmitter, transmit(::testing::_, 4, coordinator_address))
        .With(::testing::Args<0,1>(::testing::ContainerEq(expected_message)))
        .Times(1);

    EXPECT_CALL(mock_servomotor, set_position_down());
    EXPECT_CALL(mock_ally_leds, turn_off());
    EXPECT_CALL(mock_enemy_leds, turn_off());

    target.timeout(42);
}

TEST_F(TargetTest, WhenTargetIsHitThenTransmitRightMessage)
{
    uint8_t expected_message[4];
    target.generate_message(32, Target::RESULT_HIT, 42, expected_message);

    EXPECT_CALL(mock_xbee_transmitter, transmit(::testing::_, 4, coordinator_address))
        .With(::testing::Args<0,1>(::testing::ContainerEq(expected_message)))
        .Times(1);

    EXPECT_CALL(mock_servomotor, set_position_down());
    EXPECT_CALL(mock_ally_leds, turn_off());
    EXPECT_CALL(mock_enemy_leds, turn_off());

    target.hit(42);
}

TEST_F(TargetTest, GenerateMessageTargetHit)
{
    const int target_number = 0x05;
    const int time_taken_ms = 0x4254;

    uint8_t message[4];
    target.generate_message(target_number, Target::RESULT_HIT, time_taken_ms, message);

    uint8_t expected_message[4];
    expected_message[0] = 0x05;
    expected_message[1] = Target::RESULT_HIT;
    expected_message[2] = 0x42;
    expected_message[3] = 0x54;
    EXPECT_THAT(message, ::testing::ContainerEq(expected_message));
}

TEST_F(TargetTest, GenerateMessageTargetMissed)
{
    const int target_number = 0x05;
    const int time_taken_ms = 0x4254;

    uint8_t message[4];
    target.generate_message(target_number, Target::RESULT_MISSED, time_taken_ms, message);

    uint8_t expected_message[4];
    expected_message[0] = 0x05;
    expected_message[1] = Target::RESULT_MISSED;
    expected_message[2] = 0x42;
    expected_message[3] = 0x54;
    EXPECT_THAT(message, ::testing::ContainerEq(expected_message));
}

