/*------------------------------------------------------------------------------------*/
/*  Stepper motor driver for FRDM-33879A-EVB (MC33879A)                               */
/*  See application note AN5221 at www.nxp.com for more information                   */
/*------------------------------------------------------------------------------------*/

/*--COMPANY-----AUTHOR------DATE------------REVISION----NOTES-------------------------*/
/*  NXP         mareikeFSL  2015.12.14      rev 1.0     initial                       */
/*                                                                                    */
/*------------------------------------------------------------------------------------*/
/*  STEPPER MOTOR DRIVER                                                              */
/*  -> 5 seconds CW, 5 seconds CCW                                                    */
/*  -> 250ms steps                                                                    */
/*  -> green LED indicates forward direction                                          */
/*  -> red LED indicates backward direction                                           */
/*------------------------------------------------------------------------------------*/

/*--SPI-------------------------------------------------------------------------------*/
/*  CLK     (CLK)   =   PTD1    (J2/12)                                               */
/*  MOSI    (MOSI)  =   PTD2    (J2/8)                                                */
/*  MISO    (MISO)  =   PTD3    (J2/10)                                               */
/*------------------------------------------------------------------------------------*/
/*--GPIO------------------------------------------------------------------------------*/
/*  CS      (CS)    =   PTD0    (J2/6)                                                */
/*  IO12    (EN)    =   PTA1    (J1/2)                                                */
/*  PWM0    (IN5)   =   PTA12   (J1/8)                                                */
/*  PWM1    (IN6)   =   PTA5    (J1/12)                                               */
/*------------------------------------------------------------------------------------*/


/*--INCLUDES--------------------------------------------------------------------------*/
#include "mbed.h"


/*--DEFINES---------------------------------------------------------------------------*/


/*--INITS-----------------------------------------------------------------------------*/
SPI spi(PTD2, PTD3, PTD1);  //mosi, miso, clk
DigitalOut cs(PTD0);        //cs

DigitalOut en(PTA1);        //en

DigitalOut in5(PTA12);      //in5
DigitalOut in6(PTA5);       //in6

Ticker timer1;              //direction timer
Ticker timer2;              //step timer

DigitalOut cw(LED_GREEN);   //forward LED
DigitalOut ccw(LED_RED);    //backward LED


/*--CONSTANTS-------------------------------------------------------------------------*/
unsigned const ClockwiseA   =   0x55;   //turn on 1-7, 3-5
unsigned const ClockwiseB   =   0x69;   //turn on 1-7, 4-6
unsigned const ClockwiseC   =   0xAA;   //turn on 2-8, 4-6
unsigned const ClockwiseD   =   0x96;   //turn on 2-8, 3-5


/*--VARIABLES-------------------------------------------------------------------------*/
unsigned short direction = 1;   //direction of motor
unsigned short step = 0;        //step of stepper motor


/*--FUNCTION DECLARATIONS-------------------------------------------------------------*/
void init_spi(void);                    //initializes SPI
void send_spi(unsigned const word);     //sends SPI commands
void reverse(void);                     //changes LED color
void turn_motor(void);                  //spins motor
int main(void);                         //main


/*--FUNCTION DEFINITIONS--------------------------------------------------------------*/

/***************************************************************************INIT_SPI***/
void init_spi(void)
{
    spi.format(8,1);        //8-bit transfer, mode 1 (POL 0, PHA 1)
    spi.frequency(4000000); //freq
    
}   //end init_spi()

/***************************************************************************SEND_SPI***/
void send_spi(unsigned const word)
{
    cs = 0;             //set cs low
    spi.write(0x00);    //send 0x00
    spi.write(word);    //send 0xXX
    cs = 1;             //set cs high
    
}   //end send_spi()

/****************************************************************************REVERSE***/
void reverse(void)
{
    direction = !direction; //switch direction
    if(direction)
    {
        cw = 0;             //turn on green LED
        ccw = 1;            //turn off red LED
    }
    else
    {
        cw = 1;             //turn off green LED
        ccw = 0;            //turn on red LED
    }
    
}   //end reverse()

/*************************************************************************TURN_MOTOR***/
void turn_motor(void)
{
    switch(step%4)
    {
        case 0:
            send_spi(ClockwiseA);   //send 0x0055
            if(direction)
                step++;             //if forward, increase step
            else
                step--;             //if backward, decrease step
            break;
        case 1:
            send_spi(ClockwiseB);   //send 0x0069
            if(direction) step++;   //if forward, increase step
            else step--;            //if backward, decrease step
            break;
        case 2:
            send_spi(ClockwiseC);   //send 0x00AA
            if(direction)
                step++;             //if forward, increase step
            else
                step--;             //if backward, decrease step
            break;
        case 3:
            send_spi(ClockwiseD);   //send 0x0096
            if(direction)
                step++;             //if forward, increase step
            else
                step--;             //if backward, decrease step
            break;
        default:
            break;
    }   //end switch

}   //end turn_motor()

/*******************************************************************************MAIN***/
int main(void)
{
    cw = 1;     //turn off green LED
    ccw = 1;    //turn off red LED  
    cs = 1;     //set cs high
    
    reverse();  //set initial direction  
     
    init_spi(); //initialize SPI

    in5 = 0;    //set in5 PWM low (not in use)
    in6 = 0;    //set in6 PWM low (not in use)
    en = 1;     //set en high (enable device)
    
    timer1.attach(&reverse, 5);             //reverse direction every 5 seconds
    timer2.attach_us(&turn_motor, 250000);  //step every 250ms seconds

    while(true){}
    
}   //end main()