/*------------------------------------------------------------------------------------*/
/*  DC motor driver for FRDM-33879A-EVB (MC33879A)                                    */
/*  See application note AN5221 at www.nxp.com for more information                   */
/*------------------------------------------------------------------------------------*/

/*--COMPANY-----AUTHOR------DATE------------REVISION----NOTES-------------------------*/
/*  NXP         mareikeFSL  2015.12.14      rev 1.0     initial                       */
/*                                                                                    */
/*------------------------------------------------------------------------------------*/
/*  DC MOTOR DRIVER                                                                   */
/*  -> 1 second full on forward, 1 second full on backward                            */
/*  -> 3 seconds PWMing forward, 3 seconds PWMing backward                            */
/*  -> green LED indicates forward direction                                          */
/*  -> red LED indicates backward direction                                           */
/*------------------------------------------------------------------------------------*/

/*--SPI-------------------------------------------------------------------------------*/
/*  CLK     (CLK)   =   PTD1    (J2/12)                                               */
/*  MOSI    (MOSI)  =   PTD2    (J2/8)                                                */
/*  MISO    (MISO)  =   PTD3    (J2/10)                                               */
/*------------------------------------------------------------------------------------*/
/*--GPIO------------------------------------------------------------------------------*/
/*  CS      (CS)    =   PTD0    (J2/6)                                                */
/*  IO12    (EN)    =   PTA1    (J1/2)                                                */
/*------------------------------------------------------------------------------------*/
/*--PWM-------------------------------------------------------------------------------*/
/*  PWM0    (IN5)   =   PTA12   (J1/8)                                                */
/*  PWM1    (IN6)   =   PTA5    (J1/12)                                               */
/*------------------------------------------------------------------------------------*/


/*--INCLUDES--------------------------------------------------------------------------*/
#include "mbed.h"


/*--DEFINES---------------------------------------------------------------------------*/


/*--INITS-----------------------------------------------------------------------------*/
SPI spi(PTD2, PTD3, PTD1);  //mosi, miso, clk
DigitalOut cs(PTD0);        //cs

DigitalOut en(PTA1);        //en

PwmOut in5(PTA12);          //in5
PwmOut in6(PTA5);           //in6

DigitalOut cw(LED_GREEN);   //forward LED
DigitalOut ccw(LED_RED);    //backward LED


/*--CONSTANTS-------------------------------------------------------------------------*/
unsigned const ON1 = 0x01;
unsigned const ON2 = 0x02;
unsigned const ON3 = 0x04;
unsigned const ON4 = 0x08;
unsigned const ON5 = 0x10;
unsigned const ON6 = 0x20;
unsigned const ON7 = 0x40;
unsigned const ON8 = 0x80;
unsigned const ALL_OFF = 0x00;


/*--VARIABLES-------------------------------------------------------------------------*/


/*--FUNCTION DECLARATIONS-------------------------------------------------------------*/
void init_spi(void);                    //initializes SPI
void send_spi(unsigned const word);     //sends SPI commands
int main(void);                         //main


/*--FUNCTION DEFINITIONS--------------------------------------------------------------*/

/***************************************************************************INIT_SPI***/
void init_spi(void)
{
    spi.format(8,1);        //8-bit transfer, mode 1 (POL 0, PHA 1)
    spi.frequency(4000000); //freq
    
}   //end init_spi()

/***************************************************************************SEND_SPI***/
void send_spi(unsigned const word)
{
    cs = 0;             //set cs low
    spi.write(0x00);    //send 0x00
    spi.write(word);    //send 0xXX
    cs = 1;             //set cs high
    
}   //end send_spi()

/*******************************************************************************MAIN***/
int main(void)
{
    cw = 1;     //turn off green LED
    ccw = 1;    //turn off red LED
    cs = 1;     //set cs high
     
    init_spi(); //initialize SPI

    in5 = 0;    //set in5 PWM low
    in6 = 0;    //set in6 PWM low
    en = 1;     //set en high (enable device)
    
    while(true)
    {
        //SPI only
        cw = 0;                 //turn on green LED
        ccw = 1;                //turn off red LED
        send_spi(ON5 | ON7);    //5 and 7 ON (forward)
        wait_ms(1000);          //wait 1 second
        
        cw = 1;                 //turn off green LED
        ccw = 0;                //turn on red LED        
        send_spi(ON6 | ON8);    //6 and 8 ON (backward)
        wait_ms(1000);          //wait 1 second

        //SPI with PWMing
        cw = 0;                 //turn on green LED
        ccw = 1;                //turn off red LED
        in5.period_us(100);     //set period for 5
        in5.write(0.5);         //set duty cycle for 5
        send_spi(ON7);          //7 ON
        wait_ms(3000);          //wait 3 seconds
        in5.write(0);           //set duty cycle for 5 (OFF)
        send_spi(ALL_OFF);      //turn all outputs off

        cw = 1;                 //turn off green LED
        ccw = 0;                //turn on red LED        
        in6.period_us(100);     //set period for 6
        in6.write(0.5);         //set duty cycle for 6
        send_spi(ON8);          //8 ON
        wait_ms(3000);          //wait 3 seconds
        in6.write(0);           //set duty cycle for 6 (OFF)
        send_spi(ALL_OFF);      //turn all outputs off
        
    }   //end while()
    
}   //end main()