/*--------------------------------------------------------------------------------*/
/*  Plotter Demo Code for K64F (Accelerometer, Ethernet Client)                   */
/*--------------------------------------------------------------------------------*/

/*--COMPANY-----AUTHOR------DATE------------REVISION----NOTES---------------------*/
/*  NXP         mareikeFSL  2016.02.05      rev 1.0     initial                   */
/*  NXP         mareikeFSL  2016.03.02      rev 1.1     added feedback / enable   */
/*                                                                                */
/*--------------------------------------------------------------------------------*/

/*--PROGRAM FLOW------------------------------------------------------------------*/
/*  MAIN                                                                          */
/*  ->                                                                            */
/*--------------------------------------------------------------------------------*/

/*--HARDWARE----------------------------------------------------------------------*/
/*  FRDM-K64F #2        REV: 700-28163 REV C, SCH-28163 REV E                     */
/*  FRDM-STBC-AGM01     REV: 700-28556 REV B, SCH-28566 REV C                     */
/*  FRDM-STBC-AGM01 J5 is (1-2), J6 is (1-2), J7 is (1-2)                         */
/*  Ethernet cable from FRDM-K64F (#1) to FRDM-K64F (#2)                          */
/*--------------------------------------------------------------------------------*/

/*--GPIO--------------------------------------------------------------------------*/
/*  INT1_8700       (INT1)      =   PTB19   (J1-3)                                */
/*  INT2_8700       (INT2)      =   PTC1    (J1-5)                                */
/*  INT1_21002      (INT1_G)    =   PTB9    (J1-6)                                */
/*  INT2_21002      (INT2_G)    =   PTE26   (J2-1)                                */
/*  SPI_CSB_21002   (CS_G)      =   PTD0    (J2-6)                                */
/*  RST_GPIO        (RESET)     =   PTB11   (J10-4)                               */
/*  SA1_CSB_8700    (CS)        =   PTB10   (J10-3)                               */
/*  RST_MCU         (U_RESET)   =   RESET   (J9-3)                                */
/*--------------------------------------------------------------------------------*/
/*--SPI---------------------------------------------------------------------------*/
/*  SPI_MOSI        (MOSI)      =   PTD2    (J2-4)                                */
/*  SA0_MISO        (MISO)      =   PTD3    (J2-5)                                */
/*  SPI_SCLK        (CLK)       =   PTD1    (J2-6)                                */
/*--------------------------------------------------------------------------------*/
/*--I2C---------------------------------------------------------------------------*/
/*  I2C_SDA0        (DATA0)     =   PTE25   (J2-9)                                */
/*  I2C_SCL0        (CLK0)      =   PTE24   (J2-10)                               */
/*  I2C_SCL1        (CLK1)      =   PTC10   (J10-6)                               */
/*  I2C_SDA1        (DATA1)     =   PTC11   (J10-5)                               */
/*--------------------------------------------------------------------------------*/
/*--OTHER-------------------------------------------------------------------------*/
/*  GND             (GND)       =   GND     (J2-8)                                */
/*  GND             (GND)       =   GND     (J9-7)                                */
/*  GND             (GND)       =   GND     (J9-6)                                */
/*  VDD             (VDD)       =   VDD     (J9-4)                                */
/*  VDD             (VDD)       =   VDD     (J9-2)                                */
/*--------------------------------------------------------------------------------*/


/*--INCLUDES----------------------------------------------------------------------*/
#include "mbed.h"
#include "EthernetInterface.h"
#include <string>


/*--DEFINES-----------------------------------------------------------------------*/
#define STATUS              0x00    //read
#define OUT_X_MSB           0x01    //read
#define OUT_Y_MSB           0x03    //read
#define OUT_Z_MSB           0x05    //read
#define WHO_AM_I            0x0D    //read
#define CTRL_REG1           0x2A    //read/write
#define ADDRESS             0x3A    //address of FXOS8700CQ (on schematic)
#define WHO_AM_I_VALUE      0xC7    //value of who_am_i

#define CODE_MASK_CLEAR 0x3F;
#define BOARD1_PWMA 0x00    //board 1, pwm a = 0b0000
#define BOARD1_PWMB 0x40    //board 1, pwm b = 0b0100
#define BOARD2_PWMA 0x80    //board 2, pwm a = 0x1000
#define BOARD2_PWMB 0xC0    //board 2, pwm b = 0x1100
#define X_STOP 0xFA
#define Y_STOP  0xFB

#define BL_0_max        0x0A
#define BL_80_max       0x1D
#define BL_90_max       0x31
#define BL_100_max      0x80

#define BL_0to80_lim    0x06
#define BL_80to0_lim    0x0F
#define BL_80to90_lim   0x19
#define BL_90to80_lim   0x22
#define BL_90to100_lim  0x2D
#define BL_100to90_lim  0x36

#define FR_80_max       0xF4
#define FR_90_max       0xE1
#define FR_100_max      0xCD

#define FR_0to80_lim    0xF9
#define FR_80to0_lim    0xF0
#define FR_80to90_lim   0xE6
#define FR_90to80_lim   0xDD
#define FR_90to100_lim  0xD2
#define FR_100to90_lim  0xC9

#define DC_0            0x00
#define DC_80           0x19    //0x28 = 80, 0x19 = 50
#define DC_90           0x23    //0x2D = 90, 0x23 = 70
#define DC_100          0x32


/*--CONSTANTS---------------------------------------------------------------------*/
const int PORT = 7;
static const char* SERVER_IP = "192.168.1.101"; //IP of motor board
static const char* CLIENT_IP = "192.168.1.102"; //IP of accelerometer board
static const char* MASK = "255.255.255.0";      //mask
static const char* GATEWAY = "192.168.1.1";     //gateway


/*--INITS-------------------------------------------------------------------------*/
DigitalOut red(LED_RED);        //debug led
DigitalOut green(LED_GREEN);    //debug led
DigitalOut blue(LED_BLUE);      //debug led

EthernetInterface eth;          //create ethernet
UDPSocket sock;                 //creat socket
Endpoint server;                //create endpoint

I2C i2c0(PTE25, PTE24);         //I2C0


/*--VARIABLES---------------------------------------------------------------------*/
int n;                          //size of received message
char in_buffer[1];              //create receive buffer
char counter1[1] = {0};         //sample send buffer
char counter2[1] = {5};         //sample send buffer
char X_MSB_data[1] = {0x00};
char Y_MSB_data[1] = {0x00};
char x_axis = 1;
char y_axis = 0;
char prev_X = 0;
char prev_Y = 0;


/*--FUNCTION DECLARATIONS---------------------------------------------------------*/
void init_eth(void);                //initializes ethernet

void init_i2c0(void);               //initializes I2C0
void init_acc(void);
char read_register(char address,    //reads from register
                   char reg,
                   char if_print);
void write_register(char address,   //writes to register
                    char reg,
                    char write_data,
                    bool if_print);
void read_acc(void);
char convert_acc_to_pwm(char input, char axis);
int main(void);


/*--FUNCTION DEFINITIONS----------------------------------------------------------*/

/***********************************************************************INIT_ETH***/
void init_eth(void)
{
    eth.init(CLIENT_IP, MASK, GATEWAY);     //set up IP
    eth.connect();                          //connect ethernet, takes ~15s
    sock.init();                            //initialize socket
    server.set_address(SERVER_IP, PORT);    //set address of server

}   //end init_eth()

/**********************************************************************INIT_I2C0***/
void init_i2c0(void)
{
    i2c0.frequency(100000); //freq (100kHz)
    
}   //end init_i2c0()

/******************************************************************READ REGISTER***/
char read_register(char address, char reg, bool if_print)
{
    char return_data = 0;
    char data[1] = {reg};
    
    i2c0.write(address, data, 1, true); //write I2C address, then register to read
    i2c0.read(address, data, 1);        //read from register, store result in data
    
    return_data = data[0];
        
    return return_data;
    
}   //end read_register()

/*****************************************************************WRITE REGISTER***/
void write_register(char address, char reg, char write_data, bool if_print)
{
    char data[2] = {reg, write_data};
    
    i2c0.write(address, data, 2);    //write I2C address, then register, then data
    
}   //end write_register()

/***********************************************************************INIT_ACC***/
void init_acc(void)
{
    write_register(ADDRESS, CTRL_REG1, 0x00, false);            //standby
    write_register(ADDRESS, CTRL_REG1, 0x01, false);            //enable
    char success = read_register(ADDRESS, WHO_AM_I, false);
    
}   //end init_acc()

/***********************************************************************READ_ACC***/
void read_acc(void)
{
    X_MSB_data[0] = read_register(ADDRESS, OUT_X_MSB, false);
    
    Y_MSB_data[0] = read_register(ADDRESS, OUT_Y_MSB, false);

}   //end read_acc()

/*************************************************************CONVERT ACC TO PWM***/
char convert_acc_to_pwm(char input, char axis)
{
    char send;
    
    if(input <= BL_0_max) //- - - - - - - - - - - - - - - - - - - - - - - - BL_0_max
    {
        if(axis)
        {
            if((prev_X == DC_80) && (input >= BL_0to80_lim))
                send = BOARD1_PWMA | DC_80;
            else
            {
                send = X_STOP; /*BOARD1_PWMA | DC_0;*/
                prev_X = DC_0;
            }
        }
        else
        {
            if((prev_Y == DC_80) && (input >= BL_0to80_lim))
                send = BOARD2_PWMB | DC_80;
            else
            {
                send = Y_STOP; /*BOARD2_PWMB | DC_0;*/
                prev_Y = DC_0;
            }
        }
    }
    else if(input <= BL_80_max) //- - - - - - - - - - - - - - - - - - - - -BL_80_max
    {
        if(axis)
        {
            if((prev_X == DC_90) && (input >= BL_80to90_lim))
                send = BOARD1_PWMA | DC_90;
            else if((prev_X == DC_0) && input <= (BL_80to0_lim))
                send = X_STOP; /*BOARD1_PWMA | DC_0;*/
            else
            {
                send = BOARD1_PWMA | DC_80;
                prev_X = DC_80;
            }
        }
        else
        {
            if((prev_Y == DC_90) && (input >= BL_80to90_lim))
                send = BOARD2_PWMB | DC_90;
            else if((prev_Y == DC_0) && input <= (BL_80to0_lim))
                send = Y_STOP; /*BOARD2_PWMB | DC_0;*/
            else
            {
                send = BOARD2_PWMB | DC_80;
                prev_Y = DC_80;
            }
        }
    }
    else if(input <= BL_90_max) //- - - - - - - - - - - - - - - - - - - - -BL_90_max
    {
        if(axis)
        {
            if((prev_X == DC_100) && (input >= BL_90to100_lim))
                send = BOARD1_PWMA | DC_100;
            else if((prev_X == DC_80) && input <= (BL_90to80_lim))
                send = BOARD1_PWMA | DC_80;
            else
            {
                send = BOARD1_PWMA | DC_90;
                prev_X = DC_90;
            }
        }
        else
        {
            if((prev_Y == DC_100) && (input >= BL_90to100_lim))
                send = BOARD2_PWMB | DC_100;
            else if((prev_Y == DC_80) && input <= (BL_90to80_lim))
                send = BOARD2_PWMB | DC_80;
            else
            {
                send = BOARD2_PWMB | DC_90;
                prev_Y = DC_90;
            }
        }
    }
    else if(input <= BL_100_max) //- - - - - - - - - - - - - - - - - - - -BL_100_max
    {
        if(axis)
        {
            if((prev_X == DC_90) && input <= (BL_100to90_lim))
                send = BOARD1_PWMA | DC_90;
            else
            {
                send = BOARD1_PWMA | DC_100;
                prev_X = DC_100;
            }
        }
        else
        {
            if((prev_Y == DC_90) && input <= (BL_100to90_lim))
                send = BOARD2_PWMB | DC_90;
            else
            {
                send = BOARD2_PWMB | DC_100;
                prev_Y = DC_100;
            }
        }
    }
    else if(input <= FR_100_max) //- - - - - - - - - - - - - - - - - - - -FR_100_max
    {
        if(axis)
        {
            if((prev_X == DC_90) && input >= (FR_100to90_lim))
                send = BOARD1_PWMB | DC_90;
            else
            {
                send = BOARD1_PWMB | DC_100;
                prev_X = DC_100;
            }
        }
        else
        {
            if((prev_Y == DC_90) && input >= (FR_100to90_lim))
                send = BOARD2_PWMA | DC_90;
            else
            {
                send = BOARD2_PWMA | DC_100;
                prev_Y = DC_100;
            }
        }
    }
    else if(input <= FR_90_max) //- - - - - - - - - - - - - - - - - - - - -FR_90_max
    {
        if(axis)
        {
            if((prev_X == DC_80) && (input >= FR_90to80_lim))
                send = BOARD1_PWMB | DC_80;
            else if((prev_X == DC_100) && input <= (FR_100to90_lim))
                send = BOARD1_PWMB | DC_100;
            else
            {
                send = BOARD1_PWMB | DC_90;
                prev_X = DC_90;
            }
        }
        else
        {
            if((prev_Y == DC_80) && (input >= FR_90to80_lim))
                send = BOARD2_PWMA | DC_80;
            else if((prev_Y == DC_100) && input <= (FR_100to90_lim))
                send = BOARD2_PWMA | DC_100;
            else
            {
                send = BOARD2_PWMA | DC_90;
                prev_Y = DC_90;
            }
        }
    }
    else if(input <= FR_80_max) //- - - - - - - - - - - - - - - - - - - - -FR_80_max
    {
        if(axis)
        {
            if((prev_X == DC_0) && (input >= FR_80to0_lim))
                send = X_STOP; /*BOARD1_PWMB | DC_0;*/
            else if((prev_X == DC_90) && input <= (FR_80to90_lim))
                send = BOARD1_PWMB | DC_90;
            else
            {
                send = BOARD1_PWMB | DC_80;
                prev_X = DC_80;
            }
        }
        else
        {
            if((prev_Y == DC_0) && (input >= FR_80to0_lim))
                send = Y_STOP; /*BOARD2_PWMA | DC_0;*/
            else if((prev_Y == DC_90) && input <= (FR_80to90_lim))
                send = BOARD2_PWMA | DC_90;
            else
            {
                send = BOARD2_PWMA | DC_80;
                prev_Y = DC_80;
            }
        }
    }
    else /*if(input <= FR_0_max)*/ //- - - - - - - - - - - - - - - - - - - -FR_0_max
    {
        if(axis)
        {
            if((prev_X == DC_80) && input <= (FR_0to80_lim))
                send = BOARD1_PWMB | DC_80;
            else
            {
                send = X_STOP; /*BOARD1_PWMB | DC_0;*/
                prev_X = DC_0;
            }
        }
        else
        {
            if((prev_Y == DC_80) && input <= (FR_0to80_lim))
                send = BOARD2_PWMA | DC_80;
            else
            {
                send = Y_STOP; /*BOARD2_PWMA | DC_0;*/
                prev_Y = DC_0;
            }
        }
    }

    return send;

}   //end convert_acc_to_pwm()


/**********************************************************************************/
/***************************************************************************MAIN***/
/**********************************************************************************/
int main(void)
{
    red = 0;
    green = 0;
    blue = 0;
    
    init_i2c0();
    init_acc();
    init_eth(); //initialize the Ethernet connection
    
    red = 0;    //client = red
    green = 1;
    blue = 1;
    
    while(true)
    {
        read_acc();
        
        X_MSB_data[0] = convert_acc_to_pwm(X_MSB_data[0], x_axis);
        Y_MSB_data[0] = convert_acc_to_pwm(Y_MSB_data[0], y_axis);        
        
        sock.sendTo(server, X_MSB_data, sizeof(X_MSB_data));
        sock.sendTo(server, Y_MSB_data, sizeof(Y_MSB_data));
    }
    
}   //end main()