#include "mbed.h"
#include "EthernetInterface.h"
#include "C12832_lcd.h"
#include "nsdl_support.h"
#include "dbg.h"

// Include resources
#include "accelerometer.h"
#include "joystick.h"
#include "buzzer.h"
#include "led.h"
#include "lcd.h"

static C12832_LCD lcd;
Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
//#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "192.168.1.108"
#define MASK    "255.255.255.0"
#define GW      ""

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "192.168.1.101"; /* demo NSP address */
static const int NSP_PORT = 5683;
char endpoint_name[16] = "mbed-";
uint8_t ep_type[] = {"mbed_device"};
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;

static void ethernet_init()
{
    char mbed_uid[33]; // for creating unique name for the board

    /* Initialize network */
#ifdef DHCP
    NSDL_DEBUG("DHCP in use\r\n");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    if(eth.connect(30000) == 0)
        pc.printf("Connect OK\n\r");

    mbed_interface_uid(mbed_uid);
    mbed_uid[32] = '\0';
    strncat(endpoint_name, mbed_uid + 27, 15 - strlen(endpoint_name));

    lcd.locate(0,11);
    lcd.printf("IP:%s", eth.getIPAddress());

    NSDL_DEBUG("IP Address:%s ", eth.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);

    lcd.locate(0,22);
    lcd.printf("EP name:%s\n", endpoint_name);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Dynamic resources
    create_accelerometer_resource(resource_ptr);
    create_joystick_resource(resource_ptr);
    create_buzz_resource(resource_ptr);
    create_led_resource(resource_ptr);
    create_lcd_resource(resource_ptr);

    /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("NS Game Controller Demo\n");
    
    // Initialize Ethernet interface first
    ethernet_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
