/******************************************************************************
 *  NM500 NeuroShield Board (Alphabet) Gesture Recognition Demo
 *  revision 1.1.5, 2020/02/11
 *  Copyright (c) 2017 nepes inc.
 *
 *  Please use the NeuroShield library v1.1.5 or later
 ******************************************************************************/
 
#include "mbed.h"
#include <NeuroShield.h>
#include <NeuroShieldSPI.h>
#include <mpu6050.h>
#include "SDFileSystem.h"

//#define CALCULATE_TIME

#define DEFAULT_MAXIF   0x2000

#define UPPER_LIMIT         600
#define LOWER_LIMIT         100
#define IGNORE_CAPTURE_LEN  70
#define CAPTURE_LENGTH      256     //384
#define CAPTURE_LEN_FLOAT   256.0   //384.0

NeuroShield hnn;
MPU6050 mpu(0x68, PB_9, PB_8);
SDFileSystem sd(D11, D12, D13, D10, "sd");
Serial pc(USBTX, USBRX);

DigitalOut sdcard_ss(D6);
DigitalOut arduino_con(D5);
//DigitalOut trig_port(D10);

bool sd_detected = false;

int16_t ax, ay, az, gx, gy, gz;

uint8_t learn_cat = 0;           // category to learn
uint8_t precat = 0;              // previously recognized category
uint16_t learn_len;
uint16_t read_cat[6], read_dist[6], read_nid[6];
uint16_t nsr, ncount;            // response from the neurons
uint16_t prev_ncount = 0;
uint16_t fpga_version;

#if defined(CALCULATE_TIME)
Timer timer;
unsigned long cal_time[3];
#endif

uint8_t vector_ax[NEURON_SIZE];
uint8_t vector_ay[NEURON_SIZE];
uint8_t vector_az[NEURON_SIZE];
uint8_t vector_gx[NEURON_SIZE];
uint8_t vector_gy[NEURON_SIZE];
uint8_t vector_gz[NEURON_SIZE];

int16_t capture_ax[CAPTURE_LENGTH];
int16_t capture_ay[CAPTURE_LENGTH];
int16_t capture_az[CAPTURE_LENGTH];
int16_t capture_gx[CAPTURE_LENGTH];
int16_t capture_gy[CAPTURE_LENGTH];
int16_t capture_gz[CAPTURE_LENGTH];

uint16_t neuron_data[260];

int16_t extractFeatureVector(int wait_value)
{
    int i;
    uint8_t capture_start = 0;
    uint16_t capture_count = 0, wait_count = 0;;
    int16_t check_g = -1;       // 1:capture, 0:capture-end, -1:wait-input
    int16_t max_a = -1, min_a = 0x4000, max_g = -1, min_g = 0x4000;
    
#if defined(CALCULATE_TIME)
    timer.start();
    cal_time[0] = timer.read_ms();
#endif

    for (i = 0; i < NEURON_SIZE; i++) {
        vector_ax[i] = 0;
        vector_ay[i] = 0;
        vector_az[i] = 0;
        vector_gx[i] = 0;
        vector_gy[i] = 0;
        vector_gz[i] = 0;
    }
    for (i = 0; i < CAPTURE_LENGTH; i++) {
        capture_ax[i] = 0;
        capture_ay[i] = 0;
        capture_az[i] = 0;
        capture_gx[i] = 0;
        capture_gy[i] = 0;
        capture_gz[i] = 0;
    }
    
    while (1) {
        for (i = 0; i < 6; i++) {
            mpu.getMotion6(&ax, &ay, &az, &gx, &gy, &gz);
        }
            
        if ((gx > UPPER_LIMIT) || (gx < (0 - UPPER_LIMIT)) || (gy > UPPER_LIMIT) || (gy < (0 - UPPER_LIMIT)) || (gz > UPPER_LIMIT) || (gz < (0 - UPPER_LIMIT)))
            check_g = 1;
        else if ((gx < LOWER_LIMIT) && (gx > (0 - LOWER_LIMIT)) && (gy < LOWER_LIMIT) && (gy > (0 - LOWER_LIMIT)) && (gz < LOWER_LIMIT) && (gz > (0 - LOWER_LIMIT)))
            check_g = 0;
        else
            check_g = -1;
            
        if (capture_start == 0) {
            if (check_g == 1) {
                capture_start = 1;
            }
            else {
                wait_count++;
                if (wait_count >= wait_value) {
                    return(-1);    // timeout
                }
            }
        }
        else if (capture_start == 1) {
            if (check_g == 0) {
                capture_start = 0;
                break;             // go to feature extraction
            }
            else {
                if (ax > max_a) max_a = ax; else if (ax < min_a) min_a = ax;
                if (ay > max_a) max_a = ay; else if (ay < min_a) min_a = ay;
                if (az > max_a) max_a = az; else if (az < min_a) min_a = az;
                if (gx > max_g) max_g = gx; else if (gx < min_g) min_g = gx;
                if (gy > max_g) max_g = gy; else if (gy < min_g) min_g = gy;
                if (gz > max_g) max_g = gz; else if (gz < min_g) min_g = gz;
                
                capture_ax[capture_count] = ax;
                capture_ay[capture_count] = ay;
                capture_az[capture_count] = az;
                capture_gx[capture_count] = gx;
                capture_gy[capture_count] = gy;
                capture_gz[capture_count] = gz;
                
                capture_count++;
                if (capture_count >= CAPTURE_LENGTH) {
                    printf("capture_count = %d\n", capture_count);
                    return(0);  // exceed data length
                }
            }
        }
    }

#if defined(CALCULATE_TIME)
    cal_time[1] = timer.read_ms();
#endif
    
    for (i = 0; i < NEURON_SIZE; i++) {
        uint16_t base_index;
        float base_ratio, next_ratio, resample_ax, resample_ay, resample_az, resample_gx, resample_gy, resample_gz;
        
        next_ratio = (float)i;
        next_ratio = next_ratio * capture_count / CAPTURE_LEN_FLOAT;
        base_index = (uint16_t)next_ratio;
        next_ratio = next_ratio - base_index;
        base_ratio = 1.0 - next_ratio;
        
        resample_ax = (base_ratio * capture_ax[base_index]) + (next_ratio * capture_ax[base_index + 1]);
        resample_ay = (base_ratio * capture_ay[base_index]) + (next_ratio * capture_ay[base_index + 1]);
        resample_az = (base_ratio * capture_az[base_index]) + (next_ratio * capture_az[base_index + 1]);
        resample_gx = (base_ratio * capture_gx[base_index]) + (next_ratio * capture_gx[base_index + 1]);
        resample_gy = (base_ratio * capture_gy[base_index]) + (next_ratio * capture_gy[base_index + 1]);
        resample_gz = (base_ratio * capture_gz[base_index]) + (next_ratio * capture_gz[base_index + 1]);
        
        resample_ax = (resample_ax - min_a) * 255 / (max_a - min_a);
        resample_ay = (resample_ay - min_a) * 255 / (max_a - min_a);
        resample_az = (resample_az - min_a) * 255 / (max_a - min_a);
        resample_gx = (resample_gx - min_g) * 255 / (max_g - min_g);
        resample_gy = (resample_gy - min_g) * 255 / (max_g - min_g);
        resample_gz = (resample_gz - min_g) * 255 / (max_g - min_g);
        
        vector_ax[i] = (uint8_t)resample_ax;
        vector_ay[i] = (uint8_t)resample_ay;
        vector_az[i] = (uint8_t)resample_az;
        vector_gx[i] = (uint8_t)resample_gx;
        vector_gy[i] = (uint8_t)resample_gy;
        vector_gz[i] = (uint8_t)resample_gz;
    }

#if defined(CALCULATE_TIME)
    cal_time[2] = timer.read_ms();
    printf("Total: %dms, [1]: %dms, [2]: %dms\n", (unsigned long)(cal_time[2] - cal_time[0]), (unsigned long)(cal_time[1] - cal_time[0]), (unsigned long)(cal_time[2] - cal_time[1]));
#endif
    
    return(capture_count);
}

void neuron2Sd()
{
    uint16_t nm_nsr, nm_cnt;
    
    sdcard_ss = LOW;
    FILE* fp = fopen("/sd/backup.hex", "wb+");
    sdcard_ss = HIGH;
    if (fp == NULL) {
        printf("Error opening backup.hex\n");
    }
    else {
        printf("Save neuron data to SD-CARD (backup.hex) ");
        nm_nsr = hnn.getNsr();
        nm_cnt = hnn.getNcount();
        if ((nm_cnt == 0xFFFF) || (nm_cnt == 0x7FFF)) {
            sdcard_ss = LOW;
            fclose(fp);
            sdcard_ss = HIGH;
            printf("  neuron full!!\n");
            return;
        }
        
        uint16_t header[4] = { 0x1704, 0, 0, 0 };
        header[1] = NEURON_SIZE;
        header[2] = ncount = hnn.getNcount();
        //header[3] = reserved for upper byte of ncount
        
        sdcard_ss = LOW;
        fwrite(header, (sizeof(uint16_t)), 4, fp);
        sdcard_ss = HIGH;
        
        for (int i = 0; i < NEURON_SIZE; i++)
            capture_ax[i] = 0;
        
        hnn.setNsr(0x0010);
        hnn.resetChain();
        for (int i = 1; i <= nm_cnt; i++) {
            // read neuron data from NM500
            neuron_data[0] = hnn.getNcr();
            hnn.readCompVector(&neuron_data[1], NEURON_SIZE);
            neuron_data[257] = hnn.getAif();
            neuron_data[258] = hnn.getMinif();
            neuron_data[259] = hnn.getCat();
            
            // save data to backup.hex
            sdcard_ss = LOW;
            fwrite(neuron_data, sizeof(uint16_t), 260, fp);
            sdcard_ss = HIGH;
            
            if (!(i % 10))
                printf(".");
        }
        hnn.setNsr(nm_nsr);
        
        sdcard_ss = LOW;
        fclose(fp);
        sdcard_ss = HIGH;
        
        printf(". Done(%d)\n", nm_cnt);
    }
}

void sd2Neuron()
{
    uint16_t nm_nsr;
    
    sdcard_ss = LOW;
    FILE* fp = fopen("/sd/backup.hex", "r");
    sdcard_ss = HIGH;
    if (fp == NULL) {
        printf("  Error opening backup.hex\n");
    }
    else {
        printf("Restore neuron data from SD-CARD (backup.hex) ");
        
        uint16_t header[4];
        sdcard_ss = LOW;
        fread(header, (sizeof(uint16_t)), 4, fp);
        sdcard_ss = HIGH;
        if (header[0] != 0x1704) {
            printf("Header error\n");
            return;
        }
        if (header[1] != NEURON_SIZE) {
            printf("Neuron size error\n");
            return;
        }
        if (header[2] > hnn.total_neurons) {
            printf("Ncount error\n");
            return;
        }
        ncount = header[2];    
        
        nm_nsr = hnn.getNsr();
        hnn.forget();
        hnn.setNsr(0x0010);
        hnn.resetChain();
        
        // read from SD
        for (int i = 1; i <= ncount; i++) {
            sdcard_ss = LOW;
            fread(neuron_data, (sizeof(uint16_t)), 260, fp);
            sdcard_ss = HIGH;

            hnn.setNcr(neuron_data[0]);
            hnn.writeCompVector(&neuron_data[1], NEURON_SIZE);
            hnn.setAif(neuron_data[257]);
            hnn.setMinif(neuron_data[258]);
            hnn.setCat(neuron_data[259]);
            
            if (!(i % 10))
                printf(".");
        }        
        hnn.setNsr(nm_nsr);
        hnn.setMaxif(DEFAULT_MAXIF);

        sdcard_ss = LOW;
        fclose(fp);
        sdcard_ss = HIGH;
        
        printf(". Done(%d)\n", ncount);
    }
}

void displayNeurons()
{
    uint16_t nm_ncr, nm_aif, nm_cat, nm_cnt;
    
    // display the content of the committed neurons
    nm_cnt = hnn.getNcount();
    printf("  Display the neurons, ncount=%d\n", nm_cnt);
    uint16_t temp_nsr = hnn.getNsr();
    hnn.setNsr(0x0010);
    hnn.resetChain();
    for (int i = 1; i <= nm_cnt; i++) {
        nm_ncr = hnn.getNcr();
        hnn.readCompVector(neuron_data, NEURON_SIZE);
        nm_aif = hnn.getAif();
        nm_cat = hnn.getCat();
        printf("  neuron#%d \tvector=", i);
        for (int j = 0; j < 30;/*NEURON_SIZE;*/ j++) {
            printf("0x%02x, ", neuron_data[j]);
        }
        printf(" \tncr=%d, \taif=%d, \tnm_cat=%d", nm_ncr, nm_aif, (nm_cat & 0x7FFF));
        if (nm_cat & 0x8000) printf(" (degenerated)");
        printf("\n");
    }
    hnn.setNsr(temp_nsr);
}

void mpu6050Calibration()
{
    int i, j;
    int32_t sum_ax = 0, sum_ay = 0, sum_az = 0, sum_gx = 0, sum_gy = 0, sum_gz = 0;
    int16_t mean_ax, mean_ay, mean_az, mean_gx, mean_gy, mean_gz;
    int16_t ax_offset, ay_offset, az_offset, gx_offset, gy_offset, gz_offset;
    
    for (i = 0; i < 100; i++) {
        mpu.getMotion6(&ax, &ay, &az, &gx, &gy, &gz);
    }
    
    for (j = 0; j < 5; j++) {
        for (i = 0; i < 100; i++) {
            mpu.getMotion6(&ax, &ay, &az, &gx, &gy, &gz);
            sum_ax += ax;
            sum_ay += ay;
            sum_az += az;
            sum_gx += gx;
            sum_gy += gy;
            sum_gz += gz;
        }
        
        mean_ax = sum_ax / 100;
        mean_ay = sum_ay / 100;
        mean_az = sum_az / 100;
        mean_gx = sum_gx / 100;
        mean_gy = sum_gy / 100;
        mean_gz = sum_gz / 100;
        
        // MPU6050_GYRO_FS_1000 : offset = (-1) * mean_g
        // MPU6050_ACCEL_FS_8   : offset = (-0.5) * mean_a
        ax_offset = (-mean_ax) / 2;
        ay_offset = (-mean_ay) / 2;
        az_offset = (-mean_az) / 2;
        gx_offset = -mean_gx;
        gy_offset = -mean_gy;
        gz_offset = -mean_gz;
        
        // set
        mpu.setXAccelOffset(ax_offset);
        mpu.setYAccelOffset(ay_offset);
        mpu.setZAccelOffset(az_offset);
        mpu.setXGyroOffset(gx_offset);
        mpu.setYGyroOffset(gy_offset);
        mpu.setZGyroOffset(gz_offset);
    }
}

int main()
{
    int i, input_key[2];
    uint8_t firing_cat_cnt[27];
    
    arduino_con = LOW;
    sdcard_ss = HIGH;
    wait(0.5);
    
    if (hnn.begin() != 0) {
        fpga_version = hnn.fpgaVersion();
        if ((fpga_version & 0xFF00) == 0x0000) {
            printf("\n\n#### NeuroShield Board (Board v%d.0 / FPGA v%d.0) ####\n", ((fpga_version >> 4) & 0x000F), (fpga_version & 0x000F));
        }
        else if ((fpga_version & 0xFF00) == 0x0100) {
            printf("\n\n#### Prodigy Board (Board v%d.0 / FPGA v%d.0) ####\n", ((fpga_version >> 4) & 0x000F), (fpga_version & 0x000F));
        }
        else {
            printf("\n\n#### Unknown Board (Board v%d.0 / FPGA v%d.0) ####\n", ((fpga_version >> 4) & 0x000F), (fpga_version & 0x000F));
        }
        printf("\nStart NM500 initialization...\n");
        printf("  NM500 is initialized!\n");
        printf("  There are %d neurons\n", hnn.total_neurons);
        hnn.setMaxif(DEFAULT_MAXIF);
    }
    else {
        printf("\n\nStart NM500 initialization...\n");
        printf("  NM500 is not connected properly!!\n");
        printf("  Please check the connection and reboot!\n");
        while (1);
    }
    
    // initialize SD card
    printf("\nStart SD-CARD initialization...\n");
    sdcard_ss = LOW;
    mkdir("/sd/testdir", 0777);
    FILE *fp = fopen("/sd/testdir/sdtest.txt", "w");
    if (fp == NULL) {
        printf("  SD-CARD initialization failed\n");
        printf("  Insert SD-CARD and reboot\n");
        fclose(fp);
        while (1);
    }
    else {
        printf("  SD-CARD initialization success\n");
        sd_detected = true;
        fprintf(fp, "Hello World!");
        fclose(fp);
    }
    sdcard_ss = HIGH;
    
    // initialize mpu6050
    printf("\nStart MPU-6050 initialization...\n");
    mpu.initialize();
    // set gyro & accel range
    mpu.setFullScaleGyroRange(MPU6050_GYRO_FS_1000);
    mpu.setFullScaleAccelRange(MPU6050_ACCEL_FS_8);
    
    // verify connection
    for (i = 0; i < 10; i++) {
        if (mpu.testConnection()) {
            printf("  MPU-6050 is connected successfully\n");
            break;
        }
        else if (i == 9) {
            printf("  MPU-6050 connection failed\n");
            printf("  Please check the connection and reboot!\n");
            while (1);
        }
        wait(0.1);
    }
    
    // wait for ready
    printf("  Trying to calibrate. Make sure the board is stable and upright\n");
    // reset offsets
    mpu.setXAccelOffset(0);
    mpu.setYAccelOffset(0);
    mpu.setZAccelOffset(0);
    mpu.setXGyroOffset(0);
    mpu.setYGyroOffset(0);
    mpu.setZGyroOffset(0);
    mpu6050Calibration();
    printf("  MPU-6050 calibration is complete!!\n\n");
    
    //sd2Neuron();
    
    printf("Alphabet Gesture Recognition loop...\n");
    printf("Type Alphabet and enter, to learn alphabet motion\n");
    printf("Type '1' and enter, to save Neuron to SD-CARD\n");
    printf("Type '2' and enter, to restore Neuron from SD-CARD\n\n");
    
    // main loop
    while (1) {
        if (pc.readable()) {
            input_key[0] = input_key[1];
            input_key[1] = pc.getc();
            
            if (input_key[1] == 0x0D) {     // enter key
                learn_cat = input_key[0];
                if (learn_cat == '1') {
                    //displayNeurons();
                    neuron2Sd();
                }
                else if (learn_cat == '2') {
                    sd2Neuron();
                    //displayNeurons();
                }
                else if (learn_cat == '5') {
                    displayNeurons();
                }
                else if (learn_cat == '9') {
                    printf("\nClear neuron data\n");
                    hnn.forget();
                    hnn.setMaxif(DEFAULT_MAXIF);
                    ncount = hnn.getNcount();
                }
                else if ((learn_cat == '0') || ((learn_cat >= 'a') && (learn_cat <= 'z')) || ((learn_cat >= 'A') && (learn_cat <= 'Z'))) {
                    if (learn_cat == '0')
                        learn_cat = 0;
                    else if (learn_cat <= 'Z')
                        learn_cat = learn_cat - 'A' + 1;
                    else
                        learn_cat = learn_cat - 'a' + 1;
                    printf("Learning motion category '%c'\n", (learn_cat + 'A' - 1));
                    
                    learn_len = extractFeatureVector(300);
                    
                    if (learn_len > IGNORE_CAPTURE_LEN) {
                        hnn.setContext(1);     // ax
                        hnn.learn(vector_ax, NEURON_SIZE, learn_cat);
                        hnn.setContext(2);     // ay
                        hnn.learn(vector_ay, NEURON_SIZE, learn_cat);
                        hnn.setContext(3);     // az
                        hnn.learn(vector_az, NEURON_SIZE, learn_cat);
                        hnn.setContext(4);     // gx
                        hnn.learn(vector_gx, NEURON_SIZE, learn_cat);
                        hnn.setContext(5);     // gy
                        hnn.learn(vector_gy, NEURON_SIZE, learn_cat);
                        hnn.setContext(6);     // gz
                        ncount = hnn.learn(vector_gz, NEURON_SIZE, learn_cat);
                        
                        printf("  Neurons=%d, learn_len=%d\n\n", ncount, learn_len);
                    }
                    else {
                        printf("  Learning Error: timeout or exceed data length\n\n");
                    }
                }
            }
        }
        else {
            // recognize
            learn_len = extractFeatureVector(4);
            if (learn_len > IGNORE_CAPTURE_LEN) {
                hnn.setContext(1);     // ax
                hnn.classify(vector_ax, NEURON_SIZE, &read_dist[0], &read_cat[0], &read_nid[0]);
                hnn.setContext(2);     // ay
                hnn.classify(vector_ay, NEURON_SIZE, &read_dist[1], &read_cat[1], &read_nid[1]);
                hnn.setContext(3);     // az
                hnn.classify(vector_az, NEURON_SIZE, &read_dist[2], &read_cat[2], &read_nid[2]);
                hnn.setContext(4);     // gx
                hnn.classify(vector_gx, NEURON_SIZE, &read_dist[3], &read_cat[3], &read_nid[3]);
                hnn.setContext(5);     // gy
                hnn.classify(vector_gy, NEURON_SIZE, &read_dist[4], &read_cat[4], &read_nid[4]);
                hnn.setContext(6);     // gz
                hnn.classify(vector_gz, NEURON_SIZE, &read_dist[5], &read_cat[5], &read_nid[5]);
                
                for (i = 0; i < 27; i++) {
                    firing_cat_cnt[i] = 0;
                }
                
                for (i = 0; i < 6; i++) {
                    if ((read_cat[i] >= 1) && (read_cat[i] <= 26)) {
                        firing_cat_cnt[read_cat[i]]++;
                    }
                }
                
                for (i = 0; i < 6; i++)
                    read_dist[i] = read_dist[i] & 0x7FFF;
                
                for (i = 1; i < 27; i++) {
                    if (firing_cat_cnt[i] >= 3) {
                        int firing_cat = i + 'A' - 1;
                        printf("%c\n", firing_cat);
                    }
                }
                
#if 0 // for debug
                printf("%d", learn_len);
                for (i = 0; i < 6; i++) {
                    if ((read_cat[i] >= 1) && (read_cat[i] <= 26))
                        printf(" \t[%c(%d),%d,%d]", (read_cat[i] + 'A' - 1), read_cat[i], read_dist[i], read_nid[i]);
                    else
                        printf(" \t[0x%04X,%d,%d]", read_cat[i], read_dist[i], read_nid[i]);
                }
                printf("\n");
#endif
            }
#if 0 // for debug
            else {
                printf(", %d", learn_len);
            }
#endif
        }
    }
}