/**
* @file PID_controller.h
* @brief コンパスセンサを使ったPIDコントローラ
*
* Example :
* @code
* #include "mbed.h"
* #include "PID_controller.h"
*
* PIDC pidc;
*
* int main()
* {
*     while(1) {
*         pidc.confirm();
*         pc.printf("Hi, %f\r\n", pid.getCo());
*     }
* }
* @endcode
*/
#ifndef PID_CONTROLLER_H
#define PID_CONTROLLER_H

#include "mbed.h"
#include "pin_config.h"

#include "PID.h"
#include "HMC6352.h"
const double KC = 9.0;
const double TI = 0.0;
const double TD = 0.00008;
const float INTERVAL  = 0.01;
const float INPUT_LIMIT = 360.0;
const float OUTPUT_LIMIT = 0.4;
const float BIAS = 0.0;
const int SENSED_THRESHOLD = 1800;
/**
* @brief コンパスセンサを使ったPIDコントローラ
*/
class PIDC : public PID, HMC6352, Ticker
{
public :
    /**
    * @brief defaultコンストラクタ,タイマ割り込みでの計算開始
    */
    PIDC();

    /**
     * @brief コンストラクタ
     * @param sda      sda HMC6352
     * @param scl      scl HMC6352
     * @param kc       KC
     * @param ti       TI
     * @param td       TD
     * @param interval interval
     */
    PIDC(PinName sda, PinName scl, float kc, float ti, float td, float interval);

    /**
     * @brief センサの値とPIDの値をアップデート
     */
    void confirm();

    /**
     * @brief 回転用座標系リセット
     */
    void resetOffset();

    /**
     * @brief PIDの計算結果を取得
     * @return PIDの計算結果
     */
    float getCalculationResult() const;

    /**
     * 現在の角度を取得
     * @return 現在の角度(degree)
     */
    int getCurrentDegree() const;

    /**
     * センサの生値を取得
     * @return コンパスセンサの生値
     */
    int getRawDegree();

    void setPoint(float point);

    /**
     * @brief キャリブレーションする
     * @param mode ENTER OR EXIT
     */
    void calibration(int mode);
private :

    void updateOutput();

    int offSetDegree;
    int turnOverNumber;
    int beforeDegree;

protected :
    int rawDegree;
    float calculationResult;
    int currentDegree;
};

#endif//PID_CONTROLLER_H
