/**
* @file PID_controller.h
* @brief コンパスセンサを使ったPIDコントローラ
*
* Example :
* @code
* #include "mbed.h"
* #include "PID_controller.h"
*
* PIDC pidc;
*
* int main()
* {
*     while(1) {
*         pidc.confirm();
*         pc.printf("Hi, %f\r\n", pid.getCo());
*     }
* }
* @endcode
*/
#ifndef PID_CONTROLLER_H
#define PID_CONTROLLER_H

#include "mbed.h"
#include "pin_config.h"

#include "PID.h"
#include "R1307.h"

const double KC = 7.0;
const double TI = 50.0;
const double TD = 0.000002;
const float INTERVAL  = 0.001;
const float INPUT_LIMIT = 360.0;
const float OUTPUT_LIMIT = 1.0;
const float BIAS = 0.0;
const float SENSED_THRESHOLD = 180.0;

/**
* @brief コンパスセンサを使ったPIDコントローラ
*/
class PIDC
{
public :
    /**
    * @brief defaultコンストラクタ,タイマ割り込みでの計算開始
    */
    PIDC();

    /**
     * @brief コンストラクタ
     * @param sda      sda HMC6352
     * @param scl      scl HMC6352
     * @param kc       KC
     * @param ti       TI
     * @param td       TD
     * @param interval interval
     */
    PIDC(PinName tx, PinName scl, float kc, float ti, float td, float interval);

    /**
     * @brief センサの値とPIDの値をアップデート
     */
    void confirm();

    /**
     * @brief 回転用座標系リセット
     */
    void resetOffset();

    /**
     * @brief PIDの計算結果を取得
     * @return PIDの計算結果
     */
    float getCalculationResult() const;

    /**
     * 現在の角度を取得
     * @return 現在の角度(degree)
     */
    float getCurrentDegree() const;

    /**
     * センサの生値を取得
     * @return コンパスセンサの生値
     */
    float getRawDegree();

    void setPoint(float point);

private :
    float offSetDegree;
    int turnOverNumber;
    float beforeDegree;

    PID pid;
    R1307 r1370;
    Serial pidcSerial;

protected :

    float rawDegree;
    float calculationResult;
    float currentDegree;
};

#endif//PID_CONTROLLER_H
