/**
* @file motor_driver.h
* @brief モータ駆動クラス
*
* Example :
* @code
* #include "mbed.h"
* #include "motor_driver.h"
*
* MotorDriver(PB_9, PB_8, PA_5);
* Serial pc(USBTX, USBRX, 115200);
*
* int main()
* {
*     while(1) {
*         motor.goXY(0.1, 0.1, 0);
*     }
* }
* @endcode
*/
#ifndef MOTOR_DRIVER_H
#define MOTOR_DRIVER_H

#include "mbed.h"
#include "pin_config.h"

#include "arm_unit.h"
#include "omni.h"
#include "MotorDriverController.h"

const int WHEEL_NUMBER = 4;
const float WHEEL_RADIAN = M_PI / 2.0;
const int WHEEL_MDC_ADDR =  7;
const int ARM_MDC_ADDR =    6;
const int SLIDER_NUMBER =   0;
const int ARM_NUMBER =      1;
const int SWORD_NUMBER =    2;
const int SHAKE_NUMBER =    3;

/**
* @brief モータ駆動クラス
*/
class MotorDriver : MDC {
public :
    /**
    * @brief コンストラクタ
    */
    MotorDriver();

    MotorDriver(PinName sda, PinName scl, PinName solPin);

    /**
    * @brief スライダを動かす
    */
    void moveSlider(float speed);

    /**
    * @brief ひみつ道具部分を動かす
    */
    void destroy(float speed);

    /**
     * @brief 剣を振る
     * @param speed 速度
     */
    void swing(float speed);

    /**
     * @brief ひみつ道具の首振り
     * @param speed 速度
     */
    void shakeHead(float speed);

    /**
     * @brief 解放機構
     */
    void release();

    /**
    * @brief あしまわりを動かす(X, Y)
    */
    void goXY(float X, float Y, float moment);

    /**
     * @brief 足回りを動かす(circular)
     * @param r      半径
     * @param rad    角度
     * @param moment 回転
     */
    void goCircular(float r, float rad, float moment);

private :
    Arm arm;
    Omni quadOmni;
    DigitalOut solenoid;
};

#endif//MOTOR_DRIVER_H
