/**
* @file controller.h
* @brief FEPを使ったコントローラ受信部
*
* Example :
* @code
* #include "mbed.h"
* #include "controller.h"
*
* Controller pad(PA_9, PA_10, 200);
* Serial pc(USBTX, USBRX, 115200);
*
* int main()
* {
*     while(1) {
*         if(pad.receiveState()) {
*             for(int i = 0; i < 7; i++) pc.printf("%d, ", pad.getButton1(i));
*             for(int i = 0; i < 6; i++) pc.printf("%d, ", pad.getButton2(i));
*             pc.printf("\r\n");
*         } else {
*             pc.printf("ERROR\n\r");
*         }
*     }
* }
* @endcode
*/
#ifndef CONTROLLER_H
#define CONTROLLER_H

#include "mbed.h"
#include "pin_config.h"
#include <math.h>

#include "FEP.h"

// const double M_PI =     3.141592653589793;
const int ADDR =        203;
const bool FEP_SUCCESS =0;
const int DATA_SIZE =   6;
const float STICK_DIVIDE =  255.0;
const float STICK_NEWTRAL = 0.1;
const float STICK_NORM_MAX =1.0;

/**
* @brief FEPを使ったコントローラのクラス
*/
class Controller : public FEP
{
public :

    /**
    * @brief コンストラクタ
    */
    Controller();

    /**
     * @brief コンストラクタ
     * @param FEPtx FEPtx
     * @param FEPrx FEPrx
     * @param addr  address
     */
    Controller(PinName FEPtx, PinName FEPrx,int addr);

    /**
    * @brief メンバ変数にボタンのステートを格納
    */
    bool receiveState();

    /**
     * ボタン1の状態を取得
     * @param  number button number
     * @return        status
     */
    bool getButton1(int number) const;

    /**
     * ボタン2の状態を取得
     * @param  number button number
     * @return        status
     */
    bool getButton2(int number) const;

    /**
     * スティックの値を取得
     * @param  number sticknumber(x, y, x, y)
     * @return        stick value
     */
    float getStick(int number) const;

    /**
     * スチィックの角度を取得
     * @param  number left...0 right...1
     * @return        radian
     */
    float getRadian(int number) const;

    /**
     * スティックの距離を取得
     * @param  number left..0 right..1
     * @return        norm
     */
    float getNorm(int number) const;

private :
    void setStick();

    char data[6];
    uint8_t fepTemp;

protected :
    bool button1[7];
    bool button2[6];
    double stick[4];
    double radian[2];
    double norm[2];
};

#endif//CONTROLLER_H
