#ifndef _FEP_H
#define _FEP_H

#define FEP_NO_RESPONSE  (0x10) /* レスポンスが得られなかった */
#define FEP_P0  (0x11)          /* 正常終了 */
#define FEP_P1  (0x12)          /* コマンド受理、データ送信中 */
#define FEP_N0  (0x13)          /* コマンドエラー */
#define FEP_N1  (0x14)          /* データ送信失敗(宛先の無線モデムの応答なし、キャリアセンスで送信出来なかった)  */
#define FEP_N2  (0x15)          /* 割り当てなし */
#define FEP_N3  (0x16)          /* データ送信失敗(宛先の無線モデムのバッファがフルで受信できない) */

#define FEP_DT_ERR (0x30)
#define FEP_DT_STR (0x31)
#define FEP_DT_BIN (0x32)

#define FEP_RETRY 10    /* 再送回数 */

#define FEP_TIMEOUT_COUNT 40000
#define FEP_LINE_LEN 273 /* 256(=maximam data length)+14(=header length)+2(=CRLF)+1(=null character) */

#include "mbed.h"

/** FEP-0x class.
 *  Used for communicating between MCU and MCU.
 *
 * Example(Receiver NculeoF446RE):
 * @code
 * #include "mbed.h"
 * #include "FEP.h"
 * 
 * FEP fep(PB_10,PB_11,199);
 * Serial pc(USBTX,USBRX,115200);
 * 
 * int main()
 * {
 *     char data[256];
 *     uint8_t temp;
 *     while(1) {
 *        temp = fep.read(data, 6);
 *        if(temp == 0) {
 *            pc.printf("recieved\r\n");
 *        } else if(temp == 1) {
 *            pc.printf("recieved data error\r\n");
 *        } else if(temp == FEP_DT_ERR) {
 *            pc.printf("program or recieve error\r\n");
 *        } else if(temp == FEP_DT_BIN) {
 *            pc.printf("transmitter is binary mode\r\n");
 *        } else {
 *            pc.printf("No response\r\n");
 *        }
 *     }
 * }
 * @endcode
 *
 * Example(Transmitter NucleoF103RB):
 * @code
 * #include "mbed.h"
 * #include "FEP.h"
 * 
 * FEP fep(PB_10,PB_11,198);
 * Serial pc(PB_6,PB_7,115200);
 * 
 * int main()
 * {
 *     char data[256];
 *     uint8_t temp,intensity;
 *     while(1) {
 *         data[0] = 'A';
 *         data[1] = 'B';
 *         data[2] = 'D';
 *         data[3] = 'E';
 *         temp=fep.puts(data,4);
 *         if(temp==FEP_P0) pc.printf("reciever has recieved\r\n");
 *         else pc.printf("Anything error\r\n");
 *         intensity = fep.getIntensity();
 *     }
 * }
 * @endcode
 */

class FEP
{
public:
    /** Create FEP instance
     * @param tx Serial_TX
     * @param rx Serial_RX
     * @param addr 送信相手モジュールのアドレス(0~255)
     * @param baud 通信速度(default=38400)
     */
    FEP(PinName tx, PinName rx, uint8_t addr, int baud=115200);
    /** FEP destructor */
    ~FEP();
    /** setn any data using txt mode
     * @param *data 送信するデータの配列
     * @param len 送信するデータの長さ
     * @returns
     *   FEP_NO_RESPONSE レスポンスなし\n
     *   FEP_P0 正常終了\n
     *   FEP_N0 コマンドエラー\n
     *   FEP_N1 データ送信失敗(宛先の無線モデムの応答なし、キャリアセンスで送信出来なかった)\n
     *   FEP_N2 割り当てなし\n
     *   FEP_N3 データ送信失敗(宛先の無線モデムのバッファがフルで受信できない)
     */
    uint8_t puts(char *data, uint8_t len);
    /** setn any data using binary mode
     * @param *data 送信するデータの配列
     * @param len 送信するデータの長さ
     * @returns
     *   FEP_NO_RESPONSE レスポンスなし\n
     *   FEP_P0 正常終了\n
     *   FEP_N0 コマンドエラー\n
     *   FEP_N1 データ送信失敗(宛先の無線モデムの応答なし、キャリアセンスで送信出来なかった)\n
     *   FEP_N2 割り当てなし\n
     *   FEP_N3 データ送信失敗(宛先の無線モデムのバッファがフルで受信できない)
     */
    uint8_t putbin(char *ary, size_t len);
    /** recieve any data(txt or binary mode)
     * @param *data 受信データを格納する配列
     * @param len 受信データの長さ
     * @returns
     *   FEP_NO_RESPONSE レスポンスなし\n
     *   FEP_DT_ERR 受信エラー\n
     *   FEP_DT_STR TXT形式のデータを受信\n
     *   FEP_DT_BIN Binary形式のデータを受信\n
     */
    uint8_t gets(char *data, uint8_t len);
    /** recieve any data(txt mode)
     * @param *data 受信データを格納する配列
     * @param len 受信データの長さ
     * @returns
     *   FEP_NO_RESPONSE レスポンスなし\n
     *   FEP_DT_ERR 受信エラー\n
     *   FEP_DT_STR TXT形式のデータを受信\n
     *   FEP_DT_BIN Binary形式のデータを受信\n
     *   0 正常終了\n
     *   1 受信データエラー
     */
    uint8_t read(char *data, uint8_t len);
    /** get transmitter address
     * @returns
     *   transmitter address(0~255)
     */
    uint8_t getTransmitterAddr();
    /** get intensity
     * @returns
     *   intensity(absolute)
     */
    uint8_t getIntensity();
    /** reset the module
     * @returns
     *   FEP_P1 正常終了
     */
    uint8_t reset();
private:
    Serial serial;
    void hex2dec(char *data,char *str,uint8_t data_len);
    int rxHandler(char *data);
    int ctoi(char c);
    uint8_t _address;
    uint8_t _transmitterAddr;
    uint8_t _intensity;
    uint8_t getResponse();
};

#endif /*_FEP_H*/
