/**
 *  @file   controller.h
 *  @brief  2021Bcon受信用ライブラリ
 *  @author 安澤瑠
 *  @date   21/10/16
 */
#ifndef CONTROLLER_H
#define CONTROLLER_H

#define TIMEOUT_COUNT 5
#define DATANUM 5 // ボタン(BusIn)が1 スティックが4 BController_TX側のFEPﾗｲﾌﾞﾗﾘと連携していないとダメ

#include "mbed.h"

/**
 *  @class con 
 *  @brief class for receiving 2021Bcon data
 *    0       4
 *  1   3   5   7
 *    2       6
 *
 *   0        1
 *  @note  mbed-os 6 では使えません。  Not compatible with mbed-os 6
 */
class Bcon : public RawSerial {
public :
    /** constructor
     *  @param tx   FEPと接続するSerialTX pin
     *  @param rx   FEPと接続するSerialRX pin
     *  @param addr コントローラーについてるFEPのアドレス
     *  @param baud 通信速度(デフォルト115200)
     */
    Bcon(PinName tx, PinName rx, uint8_t addr_, int baud=115200);

    /** Start receiving
     */
    void StartReceive();
    
    /** Check timeout
     * @brief 0.1秒毎のループで受信のタイムアウトをチェック
     */
    void TimeoutLoop();

    /** Interrupt input
     */
    void ReceiveBytes();

    /** extract the message
     */
    void CheckData();

    /** Write the received message
     *  @param data 受信メッセージを格納する配列
     */
    void GetData(uint8_t *data);
    
    bool getButton(uint8_t n); //! BusInの計算をしてボタンの値を返す。
    int16_t getStick(uint8_t n); //! スティックの値を返す。中心を0,範囲は-128 ~ 128とする。
    
    bool status; //! (TIMEOUT_COUNT * 0.1)秒間通信が来ていないと0
    uint8_t sum;          //! BusInの値を格納

private :
    
    Ticker timeoutTimer;

    uint8_t addr;         //! 通信相手のアドレス
    uint8_t buffer[256];  //! 全受信データを格納するリングバッファ配列
    uint8_t retdata[128]; //! メッセージのみを格納する配列
    uint8_t bufindex;     //! buffer の添え字変数
    uint8_t retindex;     //! retdata の添え字変数
    uint8_t timeout;      //! 0.1秒間通信が成功していないと1増える
    uint8_t stick[4];     //! スティックの値を格納
};

#endif