#include "StateMachine.hpp"
#include "StaticDefs.hpp"
 
StateMachine::StateMachine() {
    _timeout = 20;            // generic timeout for every state, seconds
    _yo_time = 400;         // timeout for a dive or rise yo, not set for other ops
    _state_transition_time = 60;  // time to allow motors to come to rest in float_broadcast
    _pitchTolerance = 5.0;     // pitch angle tolerance for FLOAT_LEVEL state
 
    _bceFloatPosition = bce().getTravelLimit();      // bce position for "float" states                  (max travel limit for BCE is 320 mm)
    _battFloatPosition = batt().getTravelLimit();    // batt position tail high for "broadcast" state    (max travel limit for battery is 75 mm)
 
    _depth_command = 2.0;                        // user keyboard depth (default)
    _pitch_command = -20.0;                      // user keyboard pitch (default)
    _heading_command = 0.0;
    
    //new commands
    _BCE_dive_offset = 0.0;     //starting at the limits
    _BMM_dive_offset = 0.0;
    //new commands
    
    _neutral_timer        = 0;                  //timer used in FIND_NEUTRAL sub-FSM
////////////////////////////// 
    _state = SIT_IDLE;                          // select starting state here
    _isTimeoutRunning = false;                  // default timer to not running
    _isSubStateTimerRunning = false;            // default timer to not running
/////////////////////////////    
    _multi_dive_counter = 0;
    _multi_leg_counter = 0;
    _depth_KP = depthLoop().getControllerP();  // load current depth value
    _depth_KI = depthLoop().getControllerI();  // load current depth value
    _depth_KD = depthLoop().getControllerD();  // load current depth value
    
    _pitch_KP = pitchLoop().getControllerP();  // load current pitch value
    _pitch_KI = pitchLoop().getControllerI();  // load current pitch value
    _pitch_KD = pitchLoop().getControllerD();  // load current pitch value
    
    _neutral_bce_pos_mm = depthLoop().getOutputOffset(); //load current neutral buoyancy position offset
    _neutral_batt_pos_mm = pitchLoop().getOutputOffset(); //load current neutral buoyancy position offset
    
    _state_array_counter = 1;                   //used to iterate through and record states
    _substate_array_counter = 0;                //used to iterate through and record substates

    _state_array[0] = SIT_IDLE;  //system starts in the SIT_IDLE state, record this
    
    _substate = NEUTRAL_SINKING;                //start sub-FSM in NEUTRAL_SINKING
    _previous_substate = -1;                    //to start sub-FSM
    _previous_state = -1;                       //for tracking FSM states
    
    _max_recorded_depth_neutral = -99;          //float to record max depth
    _max_recorded_depth_dive = -99;             //float to record max depth
    
    _max_recorded_auto_neutral_depth = -99;
    
    _debug_menu_on = false;                     //toggle between debug and simple menu screens
    
    //new file stuff
    _pitch_filter_freq = pitchLoop().getFilterFrequency();
    _pitch_deadband = pitchLoop().getDeadband();
    
    _depth_filter_freq = depthLoop().getFilterFrequency();
    _depth_deadband = depthLoop().getDeadband();
}
 
//Finite State Machine (FSM)
int StateMachine::runStateMachine() {   // ends about line 956
    // finite state machine ... each state has at least one exit criteria
    static int lpd_oneshots=0;
    static int lpr_oneshots=0;
    static int finish_leg=0;  // allow a rise to complete ( if it takes less than a yo_time, then exit to FB)
    static float leg_max_depth = 0;
    static float leg_min_depth =0;
    static float leg_heading = 90;  //go east!
    char buf[256];
    switch (_state) {
    case SIT_IDLE :
    case FB_EXIT :  
    case KEYBOARD :
        // there actually is no timeout for SIT_IDLE, but this enables some one-shot actions
        if (!_isTimeoutRunning) {
            //tare pressure sensor
            depth().tare(); // tares to ambient (do on surface)
            
            if (_debug_menu_on)
                printDebugMenu();
            else
                printSimpleMenu();
            xbee().printf("\r\n\nstate: SIT_IDLE\r\n");
            _isTimeoutRunning = true; 
 
            // what is active?
            bce().pause();
            batt().pause();
                     
            //reset sub FSM
            _isSubStateTimerRunning = false;
        }
        
        // how exit?
        keyboard(); // keyboard function will change the state if needed
        break;
        
    case CHECK_TUNING :                 // state used to check the tuning of the pressure vessel
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: CHECK_TUNING\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();    //this is now active
            batt().unpause();   //this is now active
 
            // what are the commands? (DRIVE THE MOTORS "DIRECTLY")
            bce().setPosition_mm(_neutral_bce_pos_mm);              //this variable is loaded from the file at initialization
            batt().setPosition_mm(_neutral_batt_pos_mm);            //this variable is loaded from the file at initialization
            
            // getSetPosition_mm is the commanded position in the LinearActuator class
            
            xbee().printf("CHECK_TUNING: BCE cmd: %3.1f (BCE current position: %3.1f)\r\n", bce().getSetPosition_mm(), bce().getPosition_mm());
            xbee().printf("CHECK_TUNING: BATT cmd: %3.1f (BATT current position: %3.1f)\r\n", batt().getSetPosition_mm(), bce().getPosition_mm());
        }
    
        // how exit?
        if (_fsm_timer > _timeout) {
            xbee().printf("CHECK_TUNING: timed out!\r\n");
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        //WHAT IS ACTIVE?
        // the inner loop position controls are maintaining the positions of the linear actuators
        
        //print status to screen continuously
        xbee().printf("CHECK_TUNING: BCE_position: %0.1f, BATT_position: %0.1f (BCE_cmd: %0.1f, BATT_cmd: %0.1f)(depth: %0.1f ft,pitch: %0.1f deg,heading: %0.1f)     [%0.1f sec]\r",bce().getPosition_mm(),batt().getPosition_mm(),bce().getSetPosition_mm(),batt().getSetPosition_mm(),depthLoop().getPosition(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());
        
        break;
 
    case EMERGENCY_CLIMB :
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: EMERGENCY_CLIMB\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _yotimer.reset();
            _yotimer.start();
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands?
            bce().setPosition_mm(bce().getTravelLimit());
            batt().setPosition_mm(10.0);    //pull nose up (0.0 was sketchy)    
        }
        
        // how exit?
        if ((_fsm_timer > _timeout ) || (_yotimer > _yo_time)) {
            xbee().printf("EC: timed out\r\n");
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset(); _yotimer.reset();
            _isTimeoutRunning = false;
        }
        else if (depthLoop().getPosition() < 2.0) { //if the depth is greater than 0.2 feet, go to float broadcast
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset(); _yotimer.reset();
            _isTimeoutRunning = false;
        }
        
        //WHAT IS ACTIVE?
        //print status to screen continuously
        xbee().printf("EC: depth: %3.1f, pitch: %0.1f deg [BCE:%0.1f (cmd: %0.1f) BMM:%0.1f (cmd: %0.1f)] [%0.1f sec]\r",depthLoop().getPosition(),pitchLoop().getPosition(),bce().getPosition_mm(), bce().getSetPosition_mm(),batt().getPosition_mm(), batt().getSetPosition_mm(),_fsm_timer.read());
        
        break;
 
    case FIND_NEUTRAL :
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: FIND_NEUTRAL\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true;
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
            
            //start with a small offset from MANUAL neutral positions on the BCE only, BMM was pitching too much
            float bce_find_neutral_mm = _neutral_bce_pos_mm + 10.0;
            //float batt_find_neutral_mm = _neutral_batt_pos_mm + 10.0;
            
            bce().setPosition_mm(bce_find_neutral_mm);
            batt().setPosition_mm(_neutral_batt_pos_mm);    //set battery to the same neutral position
            
            //first iteration goes into Neutral Finding Sub-FSM 
            //set the first state of the FSM, and start the sub-FSM
            _substate = NEUTRAL_SINKING;        //first state in neutral sub-FSM is the pressure vessel sinking
            _previous_substate = -1;
            
            //save this state to the array
            _substate_array[_substate_array_counter] = NEUTRAL_SINKING;  //save to state array
            _substate_array_counter++;     
                   
            runNeutralStateMachine(); 
        }
 
        // how exit? (exit with the timer, if timer still running continue processing sub FSM)
        if (_fsm_timer > _timeout) {
            xbee().printf("FN: timed out [time: %0.1f sec]\r\n", _fsm_timer.read());
            _state = EMERGENCY_CLIMB;         //new behavior (if this times out it emergency surfaces)
            _fsm_timer.reset();
            _isTimeoutRunning = false;
            
            //record this to the NEUTRAL sub-FSM tracker
            _substate_array[_substate_array_counter] = EMERGENCY_CLIMB;  //save to state array
            _substate_array_counter++;
        }
        
        //what is active? (neutral finding sub-function runs until completion)        
        //check if substate returned exit state, if so stop running the sub-FSM
        else if (runNeutralStateMachine() == NEUTRAL_EXIT) { 
            //if successful, FIND_NEUTRAL then goes to RISE
            xbee().printf("*************************************** FIND_NEUTRAL sequence complete.  Rising.\r\n\n");
            _state = RISE;
            _isTimeoutRunning = false;
        }
        
        break;   
        
    case DIVE :
        // start local state timer and init any other one-shot actions
               
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: DIVE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands?
            depthLoop().setCommand(_depth_command);
            pitchLoop().setCommand(_pitch_command);
            
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("DIVE: depth cmd: %3.1f\r\n",depthLoop().getCommand());
            xbee().printf("DIVE: pitch cmd: %3.1f\r\n",pitchLoop().getCommand());
            xbee().printf("DIVE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
            
            //reset max dive depth
            _max_recorded_depth_dive = -99;            //float to record max depth
        }
 
        // how exit?
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("DIVE: timed out\r\n\n");
            _state = RISE; //new behavior 11/17/2017
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        else if (depthLoop().getPosition() > depthLoop().getCommand() - 0.5) { // including offset for low momentum approaches
            xbee().printf("DIVE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = RISE;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
 
        // WHAT IS ACTIVE?
        xbee().printf("DIVE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg [cmd:%6.1f], heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),pitchLoop().getCommand(),imu().getHeading(),_fsm_timer.read());
        bce().setPosition_mm(depthLoop().getOutput());  //constantly checking the Outer Loop output to move the motors
        batt().setPosition_mm(pitchLoop().getOutput());
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
        
        if (depthLoop().getPosition() > _max_recorded_depth_dive) {  //debug
            _max_recorded_depth_dive = depthLoop().getPosition();    //new max depth recorded
        }
        
        break;
    
    case RISE :
        // start local state timer and init any other one-shot actions
        
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: RISE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands?
            depthLoop().setCommand(-1.0);           //make sure to get towards the surface (saw issues at LASR pool)
            pitchLoop().setCommand(-_pitch_command);
            
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("RISE: depth cmd: %3.1f\r\n",depthLoop().getCommand());
            xbee().printf("RISE: pitch cmd: %3.1f\r\n",pitchLoop().getCommand());
            xbee().printf("RISE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
        }
 
        // how exit?
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("RISE: timed out\r\n");
            _state = EMERGENCY_CLIMB;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        //modified from (depthLoop().getPosition() < depthLoop().getCommand() + 0.5) 
        //did not work correctly in bench test (stuck in rise state)
        else if (depthLoop().getPosition() < 0.5) {
            xbee().printf("RISE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
 
        // WHAT IS ACTIVE?
        xbee().printf("RISE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg [cmd:%6.1f], heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),pitchLoop().getCommand(),imu().getHeading(),_fsm_timer.read());
        bce().setPosition_mm(depthLoop().getOutput());  //constantly checking the Outer Loop output to move the motors
        batt().setPosition_mm(pitchLoop().getOutput());
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
         
        break;
        
// NEW DIVE AND RISE SEQUENCES 
    case POSITION_DIVE :               
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: POSITION DIVE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
            rudder().unpause();
 
            // what are the commands? (using inner loops except for heading outer loop)
            // These actions happen ONCE in the POSITION_DIVE sequence
            batt().setPosition_mm(_neutral_batt_pos_mm + _BMM_dive_offset);
            bce().setPosition_mm(_neutral_bce_pos_mm - _BCE_dive_offset);
            
            //DEPTH COMMAND
            depthLoop().setCommand(_depth_command);
                        
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("POS DIVE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
            xbee().printf("POS DIVE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
            xbee().printf("POS DIVE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
            
            //reset max dive depth
            _max_recorded_depth_dive = -99;            //float to record max depth
        }
 
        // how exit?
        // timer runs out goes to POSITION_RISE
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("POS DIVE timed out\r\n\n");
            _state = POSITION_RISE; //new behavior 11/17/2017
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        // when you reach the dive threshold, surface
        else if (depthLoop().getPosition() > depthLoop().getCommand() - 0.5) { // including offset for low momentum approaches
            xbee().printf("POS DIVE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = POSITION_RISE;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
 
        // what is active?
        xbee().printf("POS DIVE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg, heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());
        
        if (depthLoop().getPosition() > _max_recorded_depth_dive) {
            _max_recorded_depth_dive = depthLoop().getPosition();    //new max depth recorded when it is larger than previous values
        }
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
        
        break;
 
    case LEG_POSITION_DIVE :               
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: LEG POSITION DIVE first time - start timer\r\n");
            sprintf(buf, "LEG POSITION DIVE start first dive start timer\n\n\r");
            mbedLogger().appendDiagFile(buf,0);
            _fsm_timer.reset(); // timer goes back to zero  I am not sure about this reset  jcw
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            _yotimer.reset();
            _yotimer.start();   //sets the yo_timer running on the dive
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
            rudder().unpause();
 
            // what are the commands? (using inner loops except for heading outer loop)
            // These actions happen ONCE in the POSITION_DIVE sequence
            batt().setPosition_mm(_neutral_batt_pos_mm + _BMM_dive_offset);
            bce().setPosition_mm(_neutral_bce_pos_mm - _BCE_dive_offset);
            //retrieve commands from structs (loaded from legfile.txt file)
            stateMachine().getLegParams();
            
            leg_max_depth = currentLegStateStruct.max_depth;
            leg_heading = currentLegStateStruct.heading;
    
            
            //DEPTH COMMAND
            //depthLoop().setCommand(_depth_command);  // I want this to be the max_depth in the legStruct
            depthLoop().setCommand(leg_max_depth);  // I want this to be the max_depth in the legStruct
                        
            //headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            headingLoop().setCommand(leg_heading);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("LEG POS DIVE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
            xbee().printf("LEG POS DIVE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
            xbee().printf("LEG POS DIVE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
            
            //reset max dive depth
            _max_recorded_depth_dive = -99;            //float to record max depth
            lpd_oneshots = 1;
        }
  
        if (!lpd_oneshots) {
                xbee().printf("\r\n\nstate: LEG POSITION DIVE oneshots\r\n");
                //   _fsm_timer.reset(); // timer goes back to zero  I am not sure about this reset  jcw
                // sm_timer.start(); // background timer starts running
                _yotimer.reset();   //sets the yo_timer running  on the NEW dive
                _yotimer.start();
                //_isTimeoutRunning = true;

                // what needs to be started?
                bce().unpause();
                batt().unpause();
                rudder().unpause();

                // what are the commands? (using inner loops except for heading outer loop)
                // These actions happen ONCE in the POSITION_DIVE sequence
                batt().setPosition_mm(_neutral_batt_pos_mm + _BMM_dive_offset);
                bce().setPosition_mm(_neutral_bce_pos_mm - _BCE_dive_offset);
                //retrieve commands from structs (loaded from missionleg.cfg file)
                leg_max_depth = currentLegStateStruct.max_depth;
                leg_heading = currentLegStateStruct.heading;
                sprintf(buf, "LEG POSITION DIVE entered via Leg_pos_rise - lpd_oneshots, only restart _yotimer\n\n\r");
                   mbedLogger().appendDiagFile(buf,0);

                //DEPTH COMMAND
                //depthLoop().setCommand(_depth_command);  // I want this to be the max_depth in the legStruct
                depthLoop().setCommand(leg_max_depth);  // I want this to be the max_depth in the legStruct

                //headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
                headingLoop().setCommand(leg_heading);     //ACTIVE HEADING (mimic of dive and rise code)

                xbee().printf("LEG POS DIVE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
                xbee().printf("LEG POS DIVE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
                xbee().printf("LEG POS DIVE: heading cmd: %3.1f\r\n",headingLoop().getCommand());

                lpd_oneshots = 1;
                //reset max dive depth
                //_max_recorded_depth_dive = -99;            //float to record max depth
          }  // lpd_oneshots  if timer is already running

            // how exit?
            // timer runs out goes to LEG_POSITION_RISE but finish_leg flag is turned on.
            if (_fsm_timer.read() > _timeout) {
                xbee().printf("LEG POSITION DIVE timed out for overall leg time\r\n\n");
                _state = LEG_POSITION_RISE; //  now start a timer on yo_timer for rise
                finish_leg =1;
                lpr_oneshots=0;
                sprintf(buf, "go to LEG POSITION DIVE on overall leg timeout, set finish_leg=1\n\n\r");
            mbedLogger().appendDiagFile(buf,0);
                // _fsm_timer.reset();
                // _isTimeoutRunning = false;
            }

            // when you reach the dive threshold, surface
            else if (depthLoop().getPosition() > depthLoop().getCommand() - 0.5) { // including offset for low momentum approaches
                xbee().printf("LEG POS DIVE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
                _state = LEG_POSITION_RISE;
                lpr_oneshots = 0;
                
                // _fsm_timer.reset();  // reset time if still inside legg long timeout?
                // _isTimeoutRunning = false;
            } else if(_yotimer.read() > _yo_time ) {
            xbee().printf("LEG POS DIVE: yo_time timed out - go to LEG_POSITION_RISE actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
                sprintf(buf, "LEG POS DIVE: yo_time timed out - go to LEG_POSITION_RISE\n\n\r");
    mbedLogger().appendDiagFile(buf,3);
                _state = LEG_POSITION_RISE;
                finish_leg =1;
                //_yotimer.reset();
                //_yotimer.start();    // restart the yo timer for next yo in rise mode    -- test if depth is small fraction of max_depth and exit to EC if small enough??
                lpr_oneshots=0;
                if (depthLoop().getPosition() < 0.7* depthLoop().getCommand()) {     //too slow - exit
                    _state = EMERGENCY_CLIMB;  // overridden here for testing
                    finish_leg = 1;
                    xbee().printf("LEG POS DIVE:descent too slow (<0.7*maxdepth) - go to EMERGENCY_CLIMB actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
                    sprintf(buf, "LEG POS DIVE: descent too slow yo_time timed out - go to emergency climb but not really for testing\n\n\r");
                    mbedLogger().appendDiagFile(buf,3);
                   _state = LEG_POSITION_RISE;
              //      _fsm_timer.reset();
              //      _isTimeoutRunning = false;
                }
            }

            // what is active?
            xbee().printf("LEG POS DIVE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg, heading_imu:%6.1f deg>>[%0.2f sec]                                         \r",
                          bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),
                          depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());

            if (depthLoop().getPosition() > _max_recorded_depth_dive) {
            _max_recorded_depth_dive = depthLoop().getPosition();    //new max depth recorded when it is larger than previous values
            }

            // ACTIVE RUDDER CONTROL
            rudder().setPosition_deg(headingLoop().getOutput());

            break;  // end LEG POSITION DIVE
    
    case POSITION_RISE :
        // start local state timer and init any other one-shot actions
        
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: POSITION RISE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands? (using inner loops except for heading outer loop)            
            batt().setPosition_mm(_neutral_batt_pos_mm - _BMM_dive_offset);          //reversing the BCE and BATT positions
            bce().setPosition_mm(_neutral_bce_pos_mm + _BCE_dive_offset);            //reversing the BCE and BATT positions
            
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("POS RISE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
            xbee().printf("POS RISE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
            xbee().printf("POS RISE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
        }
 
        // how exit?
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("POS RISE: timed out\r\n");
            _state = EMERGENCY_CLIMB;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        else if (depthLoop().getPosition() < 0.5) {
            xbee().printf("POS RISE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
 
        // what is active?
        xbee().printf("POS RISE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg, heading_imu:%6.1f deg>>[%0.2f sec]                                         \r",
            bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),
            rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
         
        break;   //end of POS RISE
        
       case LEG_POSITION_RISE :
        // start local state timer and init any other one-shot actions
           leg_min_depth = currentLegStateStruct.min_depth;
           leg_heading = currentLegStateStruct.heading;
            
        if (!_isTimeoutRunning) {  // this should never occur. the only entry into this case is from LEG_POSITION_DIVE exit
            xbee().printf("\r\n\nstate: LEG POSITION RISE\r\n");
            sprintf(buf, "state: LEG POSITION RISE  entered  with timeout NOT running should not happen\n\n\r");
    mbedLogger().appendDiagFile(buf,3);
            //_fsm_timer.reset(); // timer goes back to zero  --no
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            _yotimer.reset();    //reset yo_timer;
            _yotimer.start();    //and start it
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
            stateMachine().getLegParams();
 
            // what are the commands? (using inner loops except for heading outer loop)            
            batt().setPosition_mm(_neutral_batt_pos_mm - _BMM_dive_offset);          //reversing the BCE and BATT positions
            bce().setPosition_mm(_neutral_bce_pos_mm + _BCE_dive_offset);            //reversing the BCE and BATT positions
            //retrieve commands from structs (loaded from missionleg.cfg file)
      
            headingLoop().setCommand(leg_heading);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("LEG POS RISE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
            xbee().printf("LEG POS RISE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
            xbee().printf("LEG POS RISE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
            lpr_oneshots = 1;
        }
        if (!lpr_oneshots) {
            xbee().printf("\r\n\nstate: LEG POSITION RISE  entered via LEG_POSiTION_DIVE finish\r\n");
            sprintf(buf, "state: LEG POSITION RISE  entered via LEG_POSITION_DIVE\n\n\r");
            mbedLogger().appendDiagFile(buf,3);
            //_fsm_timer.reset(); // timer goes back to zero  --no
            _yotimer.reset();    //reset yo_timer;
            _yotimer.start();    //and start it
            //_fsm_timer.start();  timer is already running // background timer starts running
            // _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands? (using inner loops except for heading outer loop)            
            batt().setPosition_mm(_neutral_batt_pos_mm - _BMM_dive_offset);          //reversing the BCE and BATT positions
            bce().setPosition_mm(_neutral_bce_pos_mm + _BCE_dive_offset);            //reversing the BCE and BATT positions
            //retrieve commands from structs (loaded from missionleg.cfg file)
      
            headingLoop().setCommand(leg_heading);     //ACTIVE HEADING (mimic of dive and rise code)
            
            xbee().printf("LEG POS RISE: BATT cmd: %3.1f\r\n",batt().getSetPosition_mm());  //get the actual commanded position
            xbee().printf("LEG POS RISE: BCE cmd: %3.1f\r\n",bce().getSetPosition_mm());    //get the actual commanded position
            xbee().printf("LEG POS RISE: heading cmd: %3.1f\r\n",headingLoop().getCommand());
            lpr_oneshots = 1;
        }    // end if(!lpr_oneshots)
 
        // how exit?
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("LEG POS RISE: timed out on overall timeout\r\n");
            //  _state = EMERGENCY_CLIMB;
            finish_leg = 1;  // not going immediately to Emergency_climb and not resetting timers means it will take 1 yo-time of rising first
            // _fsm_timer.reset();
            sprintf(buf, "LEG POSITION DIVE  Ended via overall timeout\n\n\r");
            mbedLogger().appendDiagFile(buf,0);
            // _isTimeoutRunning = false;
        }
        else if (depthLoop().getPosition() < leg_min_depth - 0.5  ) {  // add in check for finish_leg. Don't flip flop between states in that case
            xbee().printf("LEG POS RISE: actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = LEG_POSITION_DIVE;
            sprintf(buf, "LEG POSITION RISE  flip-flops to LEG_POSITION_DIVE on near surface pressure \n\n\r");
            mbedLogger().appendDiagFile(buf,0);
            //_fsm_timer.reset();
            //_isTimeoutRunning = false;
            lpd_oneshots=0;
               if(finish_leg ==1) { 
               _state = EMERGENCY_CLIMB;
               _fsm_timer.reset(); _yotimer.reset();
               _isTimeoutRunning = false;
               sprintf(buf, "LEG POSITION RISE ... ENDS,  and since finish_leg=1, go to Emergency climb\n\n\r");
            mbedLogger().appendDiagFile(buf,0);
               }
        }
        else if(_yotimer.read() > _yo_time ) {
            xbee().printf("LEG POS RISE: yo_time timed out - go to Emergency_CLIMB. Actual depth: %3.1f (cmd: %3.1f)\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = EMERGENCY_CLIMB; // add diagnostics message
            finish_leg = 1;
            _fsm_timer.reset(); _yotimer.reset();
            _isTimeoutRunning = false;
            sprintf(buf, "LEG POSITION DIVE ... ENDS on yo_time too long, exit to emergnecy climb + finish_leg=1\n\n\r");
            mbedLogger().appendDiagFile(buf,0);
            }
 
        // what is active?
        xbee().printf("LEG POS RISE: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg, heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", 
        bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),
        rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
         
        break;   // end LEG POS RISE
// NEW DIVE AND RISE SEQUENCES 
    
    case FLOAT_LEVEL :
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: FLOAT_LEVEL\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands?
            bce().setPosition_mm(_bceFloatPosition);
            pitchLoop().setCommand(0.0);
        }
        
        // how exit?
        if (_fsm_timer > _timeout) {
            xbee().printf("FL: timed out\r\n");
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        else if (fabs(imu().getPitch() - pitchLoop().getCommand()) < fabs(_pitchTolerance)) {         //current tolerance is 5 degrees
            xbee().printf("FL: pitch: %3.1f mm, set pos: %3.1f mm, deadband: %3.1f mm\r\n",imu().getPitch(), pitchLoop().getCommand(), _pitchTolerance);
            _state = FLOAT_BROADCAST;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        // what is active?
        xbee().printf("FL: pitchLoop output: %3.1f, batt pos: %3.1f, piston pos: %3.1f [%0.1f sec]\r", pitchLoop().getOutput(), batt().getPosition_mm(), bce().getPosition_mm(), _fsm_timer.read());
        batt().setPosition_mm(pitchLoop().getOutput());
        
        break;
    
    case FLOAT_BROADCAST :
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: FLOAT_BROADCAST\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _yotimer.reset();
            _yotimer.start();
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            // what are the commands?
            bce().setPosition_mm(_bceFloatPosition);        // 320.0
            batt().setPosition_mm(_battFloatPosition);      // 73.0
            
            //set rudder to center
            rudder().setPosition_deg(0.0);  //set rudder to center, zero degrees
        }
        
        // how exit?
        
        if (_fsm_timer > _timeout || ( _yotimer > _state_transition_time)) {
            xbee().printf("FB: timed out\r\n");
            _state = SIT_IDLE;
            _fsm_timer.reset();
            
            //stop recording data
            //mbedLogger().closeFile();
            
            _isTimeoutRunning = false;
            if(finish_leg == 1) { _state = FB_EXIT; }  // allows exit if finish_leg set by leg_POS_dive, or LEG_POS_RISE timeout or too slow yo
        }
        
        //fix on float_broadcast to account for BCE stopping early in current hardware
        else if ( (fabs(bce().getPosition_mm() - bce().getSetPosition_mm()) < 4.0 ) and
                  (fabs(batt().getPosition_mm() - batt().getSetPosition_mm()) < batt().getDeadband()) ) {
            xbee().printf("FB: position: %3.1f mm, set pos: %3.1f mm, deadband: %3.1f mm\r\n",bce().getPosition_mm(), bce().getSetPosition_mm(), bce().getDeadband());
            _state = SIT_IDLE;
            _fsm_timer.reset();
            
            //stop recording data
            //mbedLogger().closeFile();
            if(finish_leg == 1) { _state = FB_EXIT; }  // allows exit if finish_leg set by leg_POS_dive, or LEG_POS_RISE timeout or too slow yo
            _isTimeoutRunning = false;
        }
        
        // what is active?
        xbee().printf("FB: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg, heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", 
        bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),
        depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),imu().getHeading(),_fsm_timer.read());
        
        if (finish_leg == 1) { _state = FB_EXIT;}
        break;
        
    case MULTI_DIVE :
        // start local state timer and init any other one-shot actions        
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: MULTI-DIVE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
                
            // what needs to be started?
            bce().unpause();
            batt().unpause();
            
            //retrieve commands from structs (loaded from sequence.cfg file)
            float sequence_depth_command = currentStateStruct.depth;
            float sequence_pitch_command = currentStateStruct.pitch;
    
            // what are the commands?            
            depthLoop().setCommand(sequence_depth_command);
            pitchLoop().setCommand(sequence_pitch_command);
            
            
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            xbee().printf("MULTI-DIVE: depth cmd: %3.1f ft, pitch cmd: %3.1f deg\r\n",depthLoop().getCommand(), pitchLoop().getCommand());
            
            //no max depth recording right now
        }
        
        // how exit?
        if (_fsm_timer > _timeout) {
            xbee().printf("\r\n\nMULTI-DIVE: timed out [time: %0.1f]\r\n\n", _fsm_timer.read());
            _state = MULTI_RISE; //new behavior 11/17/2017
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        else if (depthLoop().getPosition() > depthLoop().getCommand()) {
            xbee().printf("MULTI-DIVE: depth: %3.1f, cmd: %3.1f\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            _state = MULTI_RISE;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        // WHAT IS ACTIVE?
        xbee().printf("MD: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg [cmd:%6.1f], heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),pitchLoop().getCommand(),imu().getHeading(),_fsm_timer.read());
        bce().setPosition_mm(depthLoop().getOutput());
        batt().setPosition_mm(pitchLoop().getOutput());
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
        
        break;
    
    case MULTI_RISE :
        // start local state timer and init any other one-shot actions
        if (!_isTimeoutRunning) {
            xbee().printf("\r\n\nstate: MULTI-RISE\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            // what needs to be started?
            bce().unpause();
            batt().unpause();
 
            //NEW: retrieve depth and pitch commands from config file struct
            // concept is to load this each time the multi-dive restarts
            stateMachine().getDiveSequence();
            
            //retrieve just pitch command from struct
            float sequence_pitch_command = currentStateStruct.pitch;
 
            // what are the commands? (send back to 0.5 feet, not surface) // 11/21/2017
            depthLoop().setCommand(0.5);
            pitchLoop().setCommand(-sequence_pitch_command);            
            
            headingLoop().setCommand(_heading_command);     //ACTIVE HEADING (mimic of dive and rise code)
            xbee().printf("MULTI-RISE: depth cmd: 0.0 ft, pitch cmd: %3.1f deg\r\n",depthLoop().getCommand(), pitchLoop().getCommand());
        }
        
        // how exit?
        if (_fsm_timer > _timeout) {
            xbee().printf("MULTI-RISE: timed out [time: %0.1f]\r\n\n", _fsm_timer.read());
            _state = EMERGENCY_CLIMB;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
            
            //reset multi-dive sequence to start
            _multi_dive_counter = 0;
            
//            //Reload the dive sequence on exit
//            sequenceController().loadSequence();
        }
        else if (depthLoop().getPosition() < 0.5) { // depth is less than 0.5 (zero is surface level)
            xbee().printf("MULTI-RISE: depth: %3.1f, cmd: %3.1f\r\n", depthLoop().getPosition(), depthLoop().getCommand());
            
            //going to next state            
            _isTimeoutRunning = false;
            
            //successful dive-rise sequence CONTINUES the multi-dive sequence
            _multi_dive_counter++;
            
            //UPDATE THE SEQUENCE DATA HERE
            stateMachine().getDiveSequence();
            
            //check if this is the end of the dive sequence
            //CHECK BEFORE ANYTHING ELSE that you have reached the "exit" state (FLOAT_BROADCAST)
            if (currentStateStruct.state == FLOAT_BROADCAST) {
//                //Reload the dive sequence on exit
//                sequenceController().loadSequence();
            
                _state = FLOAT_BROADCAST;
            }
            
            else 
                _state = MULTI_DIVE;    //Note: need to test if this else statement is necessary
            
            //have to stop this with the _multi_dive_counter variable!
        }
        
        // WHAT IS ACTIVE?
        xbee().printf("MR: BcePos (cmd):%6.1f mm(%0.1f), BattPos:%6.1f mm(%0.1f), RUD_deg_cmd: %5.1f <<current depth:%6.1f ft [cmd:%6.1f]), pitch:%6.1f deg [cmd:%6.1f], heading_imu:%6.1f deg>>[%0.2f sec]                                         \r", bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),depthLoop().getCommand(),pitchLoop().getPosition(),pitchLoop().getCommand(),imu().getHeading(),_fsm_timer.read());
        bce().setPosition_mm(depthLoop().getOutput());  //constantly checking the Outer Loop output to move the motors
        batt().setPosition_mm(pitchLoop().getOutput()); 
        
        // ACTIVE RUDDER CONTROL
        rudder().setPosition_deg(headingLoop().getOutput());
        
        break; 
        
    case TX_MBED_LOG:        
        if (!_isTimeoutRunning) {            
            xbee().printf("\r\n\nstate: TX_MBED_LOG\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
            
            //mbedLogger().openFileForTransmit();     //starts _fp file pointer at the beginning of the file
            
            xbee().printf("TX_MBED_LOG set to zero\n\r");
            
            //function to read the file and get number of lines (packets to transmit)
            _timeout = mbedLogger().getNumberOfPacketsInCurrentLog();
            
            mbedLogger().setTransmitPacketNumber(0); //reset to zero
        }
        
    //TIMED OUT
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("\r\nTX_MBED_LOG: timed out!\r\n");
            
            //STATE
            _state = SIT_IDLE;
            
            _fsm_timer.reset();
            _isTimeoutRunning = false;
            
            //mbedLogger().closeLogFile();
            xbee().printf("\r\n\nstate: TX_MBED_LOG (log filed closed)\r\n");
        }
        
    //received end transmission packet (if this is true)
        if (mbedLogger().endTransmitPacket()) {
             //STATE
             led3() = !led3();
             
            _state = SIT_IDLE;
            
            //reset timer
            _fsm_timer.reset();
            
            //mbedLogger().closeLogFile();
            
            xbee().printf("\r\n\nTX_MBED_LOG: Received end transmission packet.)\r\n");
        }
        
        //What is active?
        //mbedLogger().fsmTransmitData();
        mbedLogger().checkForPythonTransmitRequest();
        led1() = !led1();
        
        break; 
        
    case RX_SEQUENCE :
        xbee().printf("state: RX_SEQUENCE\r\n");
    
        if (!_isTimeoutRunning) {
            xbee().printf("RX_SEQUENCE _isTimeoutRunning\r\n");
            _fsm_timer.reset(); // timer goes back to zero
            _fsm_timer.start(); // background timer starts running
            _isTimeoutRunning = true; 
        }
        
        if (_fsm_timer.read() > _timeout) {
            xbee().printf("RX_SEQUENCE: timed out!\r\n");
            _state = SIT_IDLE;
            _fsm_timer.reset();
            _isTimeoutRunning = false;
        }
        
        // what is active?
        xbee().printf("Receive sequence active?\r\n");
        
        break;
    
    default :
        xbee().printf("DEBUG: SIT_IDLE\r\n");
        _state = SIT_IDLE;
    }
    
    //save the state to print to user
    if (_previous_state != _state) {
        _state_array[_state_array_counter] = _state;  //save to state array
        _state_array_counter++;
        
        _previous_state = _state;
    }
    
    return _state;
}   /* end of runStateMachine */
 
// output the keyboard menu for user's reference
void StateMachine::printSimpleMenu() {   
    xbee().printf("\r\r\n\nSIMPLE KEYBOARD MENU (08/13/2018):\r\r\n");        //make sure depth sensor tares itself on startup
    xbee().printf(" Neutral Positions BCE: %0.1f BMM: %0.1f (LIMIT: BCE: %0.1f BATT: %0.1f)\r\n\n", _neutral_bce_pos_mm, _neutral_batt_pos_mm, bce().getTravelLimit(),batt().getTravelLimit());
    
    xbee().printf("  V to POSITION DIVE (initiate motor position-based dive cycle)\r\n");
    xbee().printf("  J to float level\r\n");
    xbee().printf("  B to float at broadcast pitch\r\n");
    xbee().printf("  E to initiate emergency climb\r\n");
    xbee().printf("  P to print the current log file.\r\n");
    xbee().printf("  G to transmit MBED log file\r\n");
    xbee().printf("  I to receive multi-dive sequence file\r\n");
    xbee().printf("  ~ to erase mbed log file. (clear before logging more than a few runs)\r\n");
    
    xbee().printf("Q to TYPE in the BMM offset: %0.1f (BMM Dive POS: %0.1f, Rise POS: %0.1f) (positive input offset = pitch down)\r\n",_BMM_dive_offset, _neutral_batt_pos_mm + _BMM_dive_offset, _neutral_batt_pos_mm - _BMM_dive_offset);  
    xbee().printf("A to TYPE in the BCE offset: %0.1f (BCE Dive POS: %0.1f, Rise POS: %0.1f) (positive input offset = dive)\r\n",_BCE_dive_offset, _neutral_bce_pos_mm - _BCE_dive_offset, _neutral_bce_pos_mm + _BCE_dive_offset);
    xbee().printf("W to TYPE in the heading command: %0.1f deg (imu heading: %0.1f)\r\n",_heading_command,imu().getHeading());
    xbee().printf("S to TYPE in depth setpoint: %0.1f (Current depth: %0.1f ft)\r\n",_depth_command, depthLoop().getPosition());
    xbee().printf("T to TYPE in the timeout (default is 60 seconds): %d s\r\n",_timeout);    
    
    xbee().printf("  C See sensor readings (and max recorded depth of dive & neutral sequences)\r\n");
    xbee().printf("  8 STREAM SENSOR STATUS (and channel readings)\r\n");    
    xbee().printf("  ? to reset mbed\r\n");
    xbee().printf("  * (asterisk) to go to DEBUG keyboard menu\r\n");
}

void StateMachine::printDebugMenu() {
    xbee().printf("\r\r\n\nDEBUG KEYBOARD MENU (08/08/2018):\r\r\n");
    xbee().printf("  Y to go into CHECK NEUTRAL TUNING (This is on a timer! Uses NEUTRAL positions!)\r\n");
    xbee().printf("  N to find neutral\r\n");
    xbee().printf("  M to initiate multi-dive cycle\r\n");
    xbee().printf("  D to initiate dive cycle\r\n");
    xbee().printf("  R to initiate rise\r\n");
    xbee().printf("  J to float level\r\n");
    xbee().printf("  B to float at broadcast pitch\r\n");
    xbee().printf("  E to initiate emergency climb\r\n");
    xbee().printf("  '}' to HOME the BCE (5 second delay)\r\n");
    xbee().printf("  '|' to HOME the BMM (5 second delay)\r\n");
    xbee().printf("  Z to show FSM and sub-FSM states.\r\n");
    xbee().printf("  P to print the current log file.\r\n");
    xbee().printf("  X to print the list of log files.\r\n");
    xbee().printf("  I to receive data.\r\n");
    xbee().printf("  G to transmit MBED log file (60 second timeout)\r\n");
    xbee().printf("  ~ to erase mbed log file. (clear before logging more than a few runs)\r\n");
    xbee().printf("; or : to TYPE in the BCE neutral position: %0.1f\r\n", _neutral_bce_pos_mm);
    xbee().printf("[ or { to TYPE in the BMM neutral position: %0.1f\r\n", _neutral_batt_pos_mm);
    xbee().printf("Q to TYPE in pitch setpoint: %0.1f (Current IMU pitch: %0.1f deg)\r\n",_pitch_command,imu().getPitch());
    xbee().printf("A to TYPE in depth setpoint: %0.1f (Current depth: %0.1f ft)\r\n",_depth_command, depthLoop().getPosition());
    xbee().printf("W to TYPE in the heading command: %0.1f deg (imu heading: %0.1f)\r\n",_heading_command,imu().getHeading());
    xbee().printf("T to enter in the timeout (default is 60 seconds): %d s\r\n",_timeout);

    xbee().printf("  1 BCE PID sub-menu (type in inputs)\r\n");
    xbee().printf("  2 BATT PID sub-menu (type in inputs)\r\n");
    xbee().printf("  3 Depth PID sub-menu (type in inputs)\r\n");
    xbee().printf("  4 Pitch PID sub-menu (type in inputs)\r\n");
    xbee().printf("  5 Rudder (servo) sub-menu\r\n");
    xbee().printf("  6 HEADING PID sub-menu (type in inputs)\r\n");
    xbee().printf("  7 MANUAL_TUNING sub-menu (does not have a timer!)  *** MOTORS ARE ACTIVE *** (bce 200, bmm 40, rudder 1640)\r\n");
    xbee().printf("  8 STREAM SENSOR STATUS (and channel readings)\r\n");
    
    xbee().printf(" C See sensor readings (and max recorded depth of dive & neutral sequences)\r\n");
    xbee().printf(" ? to reset mbed\r\n");
    xbee().printf(" * (asterisk) to go to SIMPLE keyboard menu\r\n");
}
 
//FIND_NEUTRAL sub-Finite State Machine (sub-FSM)
// Note: the sub-FSM only moves the pistons once at the start of each timer loop
//  (timer completes, moves piston, timer completes, moves piston, etc)
int StateMachine::runNeutralStateMachine() {                
    switch (_substate) {
        case NEUTRAL_SINKING :
            //start the 10 second timer
            if (!_isSubStateTimerRunning) {                
                _neutral_timer = _fsm_timer.read() + 5; //record the time when this block is first entered and add 5 seconds
                
                xbee().printf("\r\n\nNEUTRAL_SINKING: Next retraction at %0.1f sec [current time: %0.1f] (pitch: %0.1f) (BCE getSetPosition: %0.1f)\r\n", _neutral_timer, _fsm_timer.read(), pitchLoop().getPosition(), bce().getSetPosition_mm());
                
                // what are the commands? (BCE linear actuator active, no BMM or pitch movement)
                bce().setPosition_mm(bce().getSetPosition_mm() - 2.5);
                
                xbee().printf("NEUTRAL_SINKING: Retracting piston 2.5 mm [BCE CMD : %0.1f] (pitch: %0.1f)\r\n", bce().getSetPosition_mm(), pitchLoop().getPosition());
                
                _isSubStateTimerRunning = true;    //disable this block after one iteration
            }
 
            // how exit?
            //once reached the travel limit, no need to keep trying, so exit
            if (bce().getPosition_mm() <= 0) {
                xbee().printf("\r\nDEBUG: BCE current position is %0.1f mm (NEXT SUBSTATE NEUTRAL EXIT)\r\n", bce().getPosition_mm());
                _substate = NEUTRAL_EXIT;
                _isSubStateTimerRunning = false; // reset the sub state timer
            }
            //Troy: Pressure vessel went beyond set depth limit, goes to next state
            //once deeper than the commanded setpoint...
            else if (depthLoop().getPosition() > _depth_command) {
                _substate = NEUTRAL_SLOWLY_RISE; // next state
                _isSubStateTimerRunning = false; //reset the sub state timer
            }
 
            // what is active?
            //once the 10 second timer is complete, reset the timeout so the state one-shot entry will move the setpoint
            if (_fsm_timer.read() >= _neutral_timer) {
                xbee().printf("\r\n\n NEUTRAL_SINKING TIMER COMPLETE! [current time: %0.1f]\r\n", _fsm_timer.read());
                
                _isSubStateTimerRunning = false; // reset the sub state timer to do one-shot actions again
            }
            
            // what is active? (only the buoyancy engine moved every 5 seconds at start)
            xbee().printf("BCE current pos: %0.1f mm (BCE setpoint: %0.1f mm) (current depth: %0.1f ft)\r", bce().getPosition_mm(),bce().getSetPosition_mm(),depthLoop().getPosition()); //debug
            
            //the BCE moves every 5 seconds. No BMM or rudder movement.
            
            break;
            
        case NEUTRAL_SLOWLY_RISE:
            if (!_isSubStateTimerRunning) {                                
                _neutral_timer = _fsm_timer.read()+ 5; //record the time when this block is first entered and add 5 seconds
                
                xbee().printf("\r\n\nNEUTRAL_SLOWLY_RISE: Next extension at %0.1f sec) [current time: %0.1f]\r\n",_neutral_timer,_fsm_timer.read());
                
                // what are the commands?
                //move piston at start of sequence (default: extend 2.0 mm)
                //Pressure vessel should slowly rise
                bce().setPosition_mm(bce().getSetPosition_mm() + 2.0);  //no depth command, only motor position
                
                //Troy: I commented out this command, we're finding pitch in the next state.
                // it's okay to run the pitch outer loop now since we've already found pitch level in the previous state
                //pitchLoop().setCommand(0.0);
                
                xbee().printf("NEUTRAL_SLOWLY_RISE: Extending BCE piston 2.0 mm [BCE CMD : %0.1f] (pitch: %0.1f)\r\n", bce().getSetPosition_mm(), pitchLoop().getPosition());

                _isSubStateTimerRunning = true;    //disable this block after one iteration
            }
            
            // how exit?
            //once at full travel limit (setPosition) and haven't yet risen, time to give up and exit
            if (bce().getSetPosition_mm() >= bce().getTravelLimit()) {
                _substate = NEUTRAL_EXIT;     
                _isSubStateTimerRunning = false; // reset the sub state timer
            }
            //Troy: Depth rate will go negative as the pressure vessel starts rising
            //depth rate or sink rate < 0 ft/s, go to the next substate the next iteration
            else if (depthLoop().getVelocity() < 0) { //less than zero ft/s
                xbee().printf("\r\n\nNEUTRAL_SLOWLY_RISE: Sink Rate < 0 ft/s [time: %0.1f]\r\n", _fsm_timer.read());
                _substate = NEUTRAL_CHECK_PITCH;
                _isSubStateTimerRunning = false; // reset the sub state timer
            }
            
            // what is active?
            //once 5 second timer complete, reset the timeout so the state one-shot entry will move the setpoint
            if (_fsm_timer.read() >= _neutral_timer) {
                xbee().printf("\r\n\n NEUTRAL_SLOWLY_RISE TIMER COMPLETE! [timer: %0.1f]\r\n", _fsm_timer.read());
   
                _isSubStateTimerRunning = false; // reset the sub state timer to do one-shot actions again
            }
                        
            // what is active? (only the buoyancy engine moved every 5 seconds)
            xbee().printf("depthLoop getOutput: %0.1f\r", depthLoop().getOutput()); //debug
            
            break;   
                
        case NEUTRAL_CHECK_PITCH : // fall thru to next state is desired
            // start local state timer and init any other one-shot actions
            
            if (!_isSubStateTimerRunning) {                    
                _neutral_timer = _fsm_timer.read() + 10; // record time when this block is entered and add several seconds
                xbee().printf("\r\nNEUTRAL_CHECK_PITCH: Next move in %0.1f sec \r\n",_neutral_timer - _fsm_timer.read());
                
                // what are the commands? (default: retract or extend 0.5 mm)
                if (pitchLoop().getPosition() > 2) { // nose is high (extend batteries)
                    batt().setPosition_mm(batt().getSetPosition_mm() + 0.5); // move battery forward (using setpoint from linear actuator)
                    xbee().printf("\r\nNeutral Check Pitch: moving battery FWD in 0.5 mm increments\r\n\n");
                }
                else if (pitchLoop().getPosition() < -2) { // nose is low (retract batteries)
                    batt().setPosition_mm(batt().getSetPosition_mm() - 0.5); // move battery aft (using setpoint from linear actuator)
                    xbee().printf("\r\nNeutral Check Pitch: moving battery AFT in 0.5 mm increments\r\n\n");
                }

                _isSubStateTimerRunning = true;    //disable this block after one iteration
            }
 
            // how exit?            
            //pitch angle and pitch rate within small tolerance
            //benchtop tests confirm angle needs to be around 2 degrees
            if ((fabs(pitchLoop().getPosition()) < 2.0) and (fabs(pitchLoop().getVelocity()) < 5.0)) { 
                xbee().printf("Debug: Found Level (NEUTRAL_CHECK_PITCH or NEUTRAL_FIRST_PITCH)\r\n");    //debug
                // found level, but don't need to save anything this time
                
                if (depthLoop().getPosition() > _max_recorded_depth_neutral) {  //debug
                    _max_recorded_depth_neutral = depthLoop().getPosition();    //new max depth recorded
                }
                
                // found level and at depth too, so save it all now               
                if (_substate == NEUTRAL_CHECK_PITCH) {
                    //save positions locally
                    _neutral_batt_pos_mm = batt().getPosition_mm();
                    _neutral_bce_pos_mm = bce().getPosition_mm();
                    
                    //set the neutral positions in each outer loop
                    depthLoop().setOutputOffset(_neutral_bce_pos_mm);
                    pitchLoop().setOutputOffset(_neutral_batt_pos_mm);
                    
                    // save into the depth.txt and pitch.txt files
       
                    configFileIO().savePitchData(_pitch_KP, _pitch_KI, _pitch_KD, _neutral_batt_pos_mm, _pitch_filter_freq, _pitch_deadband); //P,I,D,batt zeroOffset
                    configFileIO().saveDepthData(_depth_KP, _depth_KI, _depth_KD, _neutral_bce_pos_mm, _depth_filter_freq, _depth_deadband); //P,I,D, bce zeroOffset

                    xbee().printf("\r\n\n>>> Saving Positions: BCE: %0.1f mm, BATT: %0.1f <<<\r\n\n",_neutral_bce_pos_mm,_neutral_batt_pos_mm);
                    
                    _substate = NEUTRAL_EXIT;
                    _isSubStateTimerRunning = false; // reset the sub state timer to do one-shot actions again
                }
                
                else {
                    xbee().printf("\r\nDid not find NEUTRAL_CHECK_PITCH or NEUTRAL_FIRST_PITCH, how did I get here?!\r\n");
                    _substate = NEUTRAL_EXIT;
                }
            }
            
            // what is active?
            //once timer complete, reset the timeout so the state one-shot entry will move the setpoint
            if (_fsm_timer.read() >= _neutral_timer) {
                xbee().printf("\r\n\nlevel timer COMPLETE!");
                xbee().printf("\r\n\n (BATT POS: %0.1f) moving 1 mm [timer: %0.1f]\r\n", batt().getPosition_mm(), _fsm_timer.read());
                _isSubStateTimerRunning = false; // reset the sub state timer to do one-shot actions again
            }

            break;
             
        //this state could be removed, it is only used as a transition but is needed to stop entering this function
        case NEUTRAL_EXIT :
            xbee().printf("substate: NEUTRAL_EXIT\r\n");            
            break;
            
        default :
            xbee().printf("how did we get to substate: default?\r\n"); //debug
            //a default within the sub-state machine
            _substate = NEUTRAL_EXIT;            
            break;
    }
    
    // reset the sub-FSM if needed (useful if you need to redo the neutral-finding sequence)
    if (_substate == NEUTRAL_EXIT) {
        xbee().printf("********************************  EXITING sub-FSM! *******************************\r\n\n");

        //reset internal sub-state back to first entry conditions (first state is immediately sinking)
        _substate = NEUTRAL_SINKING;
        _isSubStateTimerRunning = false; // reset the sub state timer
        
        //record sub-states to view after sequence
        _substate_array[_substate_array_counter] = NEUTRAL_EXIT;  //save exit to state array
        _substate_array_counter++;
        
        //reset _previous_substate on exit (has to be done in FIND_NEUTRAL if emergency exit)
        _previous_substate = -1;

        //NEUTRAL_EXIT state is used to tell the greater FSM that this sub-FSM has completed
        return NEUTRAL_EXIT; // message to calling function we just exited
    }
    else {
        //record sub-states to view after sequence (when changed)
        if (_previous_substate != _substate) {
            _substate_array[_substate_array_counter] = _substate;  //save current state to state array
            _substate_array_counter++;
            
            //record the current substate for comparison 
            _previous_substate = _substate;
        }       
        
        return _substate; // message to calling function of what sub-state it's in
    }
}
 
/*  keyboard runs independently of the state machine, handling one key at a time
    keyboard updates the desired _keyboard_state that is used in the state machine
    and only allows input when the state is "idle" */
    
void StateMachine::keyboard() {
    char user_input;
 
    // check keyboard and make settings changes as requested
    // states can be changed only at the start of a sequence (when the system is in SIT_IDLE)
    
    //TEST
    int _keyboard_state = 0;   //made this a local variable because it was retaining the last keyboard state
    
    if (xbee().readable() && (_state == SIT_IDLE || _state == KEYBOARD)) {        
        // get the key
        user_input = xbee().getc();    
        
        //record that the keyboard was used
        _state_array[_state_array_counter] = KEYBOARD;
        _state_array_counter++;
        
        // keyboard has to reset timer each time it's used
        _isTimeoutRunning = false;
        
        // check command against desired control buttons
 
    /***************************** COMMON COMMANDS *****************************/ 
        if (user_input == 'W') {
            xbee().printf(">> Please enter the heading (deg).\r\n");
            _heading_command = getFloatUserInput();
        }
        
//////////////////// TEST                   
//        else if (user_input == 'l') {
//            xbee().printf("CUT OFF RUDDER SERVO????\n\r");
//            rudder().pwm_pulse_off();
//        }
//////////////////// TEST
        
        else if (user_input == 'K') {
            xbee().printf("(K) BLAST DATA AND EXIT! \n\r");
            mbedLogger().blastData();
        }
        
        else if (user_input == 'U') {
            xbee().printf("(U) TRANSMIT MULTIPLE PACKETS \n\r");
                        
            mbedLogger().transmitMultiplePackets();
        }
        
        else if (user_input == 'H') {
            xbee().printf("(H) BLAST DATA AND MULTIPLE PACKETS! \n\r");
            wait(1);
            mbedLogger().blastData();
            
            mbedLogger().transmitMultiplePackets(); 
        }
        
        else if (user_input == 'G') {
            xbee().printf("(G) BLAST DATA TEST and use FSM! \n\r");
            wait(1);
            mbedLogger().blastData();
            
            _keyboard_state = TX_MBED_LOG;
        }
        
        else if (user_input == 'I') {
            xbee().printf("(I) Receive Multi-Dive Sequence! \n\r");
            mbedLogger().receiveSequenceFile();    //receive sequence.txt files
        }
        
        else if (user_input == '8') {
            keyboard_menu_STREAM_STATUS();
        }
                    
        else if (user_input == '?') {
            xbee().printf("\n\n\n>>> Resetting MBED <<<\n\n\n");
            wait(0.5);
            mbed_reset();
        }
        
        else if (user_input == 'T') {
            xbee().printf("Please enter the timeout (timer) value below: \n\r");
            _timeout = fabs(getFloatUserInput());
        }
        
        else if (user_input == '~') {
            xbee().printf("MBED LOG FILE MENU!\r\n");
            stateMachine().logFileMenu();
                
            //xbee().printf("ERASING MBED LOG FILE\r\n");   //legacy method
            //mbedLogger().eraseFile();
        }
        
        else if (user_input == 'C' or user_input == 'c') {
                
            xbee().printf("\r\n\nCURRENT STATUS AND PARAMETERS:\r\n");
            
            xbee().printf("raw BCE pos: %d \r\n",adc().readCh0());
            xbee().printf("raw BMM pos: %d \r\n",adc().readCh1());
            xbee().printf("raw BCE current sense: %d \r\n",adc().readCh2());
            xbee().printf("raw BMM current sense: %d \r\n",adc().readCh3());
            xbee().printf("raw depth pressure: %d \r\n",adc().readCh4());
            xbee().printf("raw vessel pressure %d (internal psi: %0.1f)\r\n", adc().readCh5(),sensors().getInternalPressurePSI());
            //xbee().printf("raw vessel pressure: %d \r\n",adc().readCh5());
            xbee().printf("raw board voltage: %d (%0.1f volts)\r\n",adc().readCh6(),sensors().getVoltageInput());
            xbee().printf("raw board current: %d (%0.3f amps)\r\n",adc().readCh7(), sensors().getCurrentInput());
            xbee().printf("raw BCE limit switch: %d \r\n",bce().getSwitch());
            xbee().printf("raw BMM limit switch: %d \r\n",batt().getSwitch());
            xbee().printf("calc vessel pressure: %f (counts: %d) \r\n",sensors().getInternalPressurePSI(),adc().readCh5());
            // End of ADC Test
            
            xbee().printf("depth: %3.1f ft\r\n",depthLoop().getPosition());
            xbee().printf("pitch: %3.1f deg\r\n",imu().getPitch());
            xbee().printf("bce().getPosition_mm(): %3.1f\r\n",bce().getPosition_mm());
            xbee().printf("bce().getSetPosition_mm(): %3.1f\r\n",bce().getSetPosition_mm());
            xbee().printf("batt().getPosition_mm(): %3.1f\r\n",batt().getPosition_mm());
            xbee().printf("batt().getSetPosition_mm(): %3.1f\r\n",batt().getSetPosition_mm());
            xbee().printf("depthLoop().getCommand(): %3.1f\r\n",depthLoop().getCommand());
            xbee().printf("pitchLoop().getCommand(): %3.1f\r\n",pitchLoop().getCommand());
            
            xbee().printf("\r\nNeutral Buoyancy Positions: bce: %0.1f, batt: %0.1f\r\n",_neutral_bce_pos_mm,_neutral_batt_pos_mm);
            xbee().printf("depthLoop().getOutputOffset(): %0.1f\r\n",depthLoop().getOutputOffset());
            xbee().printf("pitchLoop().getOutputOffset(): %0.1f\r\n",pitchLoop().getOutputOffset());
            xbee().printf("Max recorded depth: neutral: %0.1f, dive: %0.1f, auto_neutral_depth: %0.1f\r\n\n",_max_recorded_depth_neutral, _max_recorded_depth_dive, _max_recorded_auto_neutral_depth);
            
            xbee().printf("\r\n");
            xbee().printf("bce      P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", bce().getControllerP(), bce().getControllerI(), bce().getControllerD(), bce().getZeroCounts(), bce().getTravelLimit(), bce().getPotSlope(), bce().getFilterFrequency(), bce().getDeadband());
            xbee().printf("batt     P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", batt().getControllerP(), batt().getControllerI(), batt().getControllerD(), batt().getZeroCounts(), batt().getTravelLimit(), batt().getPotSlope(), batt().getFilterFrequency(), batt().getDeadband());
            xbee().printf("rudder   min_pwm:%6.1f, center_pwm:%6.1f, max_pwm:%6.1f (min_deg:%6.1f max_deg:%6.1f)\r\n",rudder().getMinPWM(),rudder().getCenterPWM(),rudder().getMaxPWM(),rudder().getMinDeg(),rudder().getMaxDeg());
            xbee().printf("depth    P:%6.2f, I:%6.2f, D:%6.2f, output offset: %6.1f mm, filter_freq: %0.1f, deadband: %0.1f \r\n", depthLoop().getControllerP(), depthLoop().getControllerI(), depthLoop().getControllerD(), depthLoop().getOutputOffset(),depthLoop().getFilterFrequency(),depthLoop().getDeadband());
            xbee().printf("pitch    P:%6.2f, I:%6.2f, D:%6.2f, output offset: %6.1f mm, filter_freq: %0.1f, deadband: %0.1f \r\n", pitchLoop().getControllerP(), pitchLoop().getControllerI(), pitchLoop().getControllerD(), pitchLoop().getOutputOffset(),pitchLoop().getFilterFrequency(),pitchLoop().getDeadband());
            xbee().printf("heading  P:%6.2f, I:%6.2f, D:%6.2f, output offset: %6.1f mm, filter_freq: %0.1f, deadband: %0.1f \r\n", headingLoop().getControllerP(), headingLoop().getControllerI(), headingLoop().getControllerD(), headingLoop().getOutputOffset(),headingLoop().getFilterFrequency(),headingLoop().getDeadband());
        }
        
    /***************************** COMMON COMMANDS *****************************/
        
/***************************** DEBUG MENU *****************************/             
        if (_debug_menu_on) {            
            if (user_input == 'D') {
                _keyboard_state = DIVE;
            }
            
            else if (user_input == '}') {
                xbee().printf("HOMING the BCE (5 second delay)\r\n");
                wait(5);
                bce().homePiston();
            }
            
            else if (user_input == '|') {
                xbee().printf("HOMING the BMM (5 second delay)\r\n");
                wait(5);
                batt().homePiston();
            }
            
            else if (user_input == 'N') {
                _keyboard_state = FIND_NEUTRAL;
            }
            else if (user_input == 'M') {
                //currently does not run if there is no file.
                
                //need to add method to Sequence Controller that returns -1 
                //   or some check that insures you cannot run the dive sequence without a file
                
                //load sequence from file
                _multi_dive_counter = 0;
                sequenceController().loadSequence();
                wait(1);    //test if this resets the sequence
                
                stateMachine().getDiveSequence();               //get first sequence on keyboard press
                _keyboard_state = currentStateStruct.state;
                
                xbee().printf("Starting Dive Sequence Controller! (state: %d)\r\n", _keyboard_state);  //neutral sequence and dive cycles
            }
            else if (user_input == 'R') {
                _keyboard_state = RISE;
            }
            else if (user_input == 'J') {
                _keyboard_state = FLOAT_LEVEL;
            }
            else if (user_input == 'B') {
                _keyboard_state = FLOAT_BROADCAST;
            }
            else if (user_input == 'E') {
                _keyboard_state = EMERGENCY_CLIMB;
            }
            
            else if (user_input == 'Y') {
                _keyboard_state = CHECK_TUNING;
            }
            
            // some debug tools below
            else if (user_input == 'P') {
                //Print current SD card log file
                //printCurrentSdLog();
                mbedLogger().printCurrentLogFile();        //print the current log file to the screen
            }
            else if (user_input == 'X') {
                mbedLogger().printMbedDirectory();        //print all log files to the screen
            }
            else if (user_input == 'Z') {
                xbee().printf("FSG FSM States: \r\n");
                string string_state;
                
                for (int i = 0; i < _state_array_counter; i++) {
                    if (_state_array[i] == SIT_IDLE)
                        string_state = "SIT_IDLE              <END>";
                    else if (_state_array[i] == FIND_NEUTRAL)
                        string_state = "FIND_NEUTRAL";
                    else if (_state_array[i] == DIVE)
                        string_state = "DIVE";
                    else if (_state_array[i] == RISE)
                        string_state = "RISE";
                    else if (_state_array[i] == FLOAT_LEVEL)
                        string_state = "FLOAT_LEVEL";
                    else if (_state_array[i] == FLOAT_BROADCAST)
                        string_state = "FLOAT_BROADCAST";          
                    else if (_state_array[i] == EMERGENCY_CLIMB)
                        string_state = "EMERGENCY_CLIMB";
                    else if (_state_array[i] == MULTI_DIVE)
                        string_state = "MULTI_DIVE";
                    else if (_state_array[i] == MULTI_RISE)
                        string_state = "MULTI_RISE";
                    else if (_state_array[i] == KEYBOARD)
                        string_state = "KEYBOARD";                    
                    xbee().printf("State #%d: %d (%s)\r\n", i, _state_array[i], string_state.c_str());
                }
                
                xbee().printf("\r\nNeutral sub-FSM States: \r\n");
                string string_substate;
                
                for (int i = 0; i < _substate_array_counter; i++) {
                    if (_substate_array[i] == NEUTRAL_SINKING)
                        string_substate = "NEUTRAL_SINKING";
                    else if (_substate_array[i] == NEUTRAL_SLOWLY_RISE)
                        string_substate = "NEUTRAL_SLOWLY_RISE";
                    else if (_substate_array[i] == NEUTRAL_CHECK_PITCH)
                        string_substate = "NEUTRAL_CHECK_PITCH";
                    else if (_substate_array[i] == NEUTRAL_EXIT)
                        string_substate = "NEUTRAL_EXIT                  <--   ";
                    else if (_substate_array[i] == EMERGENCY_CLIMB)
                        string_substate = " -- > EMERGENCY_CLIMB  <-- ";                
                    xbee().printf("Neutral Substate #%d: %d (%s)\r\n", i, _state_array[i], string_substate.c_str());
                }
                xbee().printf("\r\n");  //make space between printouts
            }    
            //BATTERY/PITCH
            else if (user_input == '[' or user_input == '{') {
                xbee().printf("Please TYPE in the new BATT neutral position.\n\r");
                _neutral_batt_pos_mm = getFloatUserInput();
                pitchLoop().setOutputOffset(_neutral_batt_pos_mm); // decrease the batt neutral setpoint
                xbee().printf("Adjusting batt neutral position. new offset: %0.1f\r\n",pitchLoop().getOutputOffset());
                // save neutral pitch value to config file
                configFileIO().savePitchData(_pitch_KP, _pitch_KI, _pitch_KD, _neutral_batt_pos_mm, _pitch_filter_freq, _pitch_deadband); //P,I,D,batt zeroOffset
            }
            
            //BCE/DEPTH
            else if (user_input == ';' or user_input == ':') {
                xbee().printf("Please TYPE in the new BCE neutral position.\n\r");
                _neutral_bce_pos_mm = getFloatUserInput();
                depthLoop().setOutputOffset(_neutral_bce_pos_mm); // decrease the bce neutral setpoint
                xbee().printf("Adjusting bce neutral position. new offset: %0.1f\r\n",depthLoop().getOutputOffset());
                // save neutral depth value to config file
                configFileIO().saveDepthData(_depth_KP, _depth_KI, _depth_KD, _neutral_bce_pos_mm, _depth_filter_freq, _depth_deadband);
            }
     
    // change settings
            //heading is in the common controls        
            else if (user_input == 'Q') {
                xbee().printf(">> Please enter the desired PITCH (deg).\r\n");
                _pitch_command = getFloatUserInput();
            }
            else if (user_input == 'A') {
                xbee().printf(">> Please enter the desired DEPTH (ft).\r\n");
                _depth_command = getFloatUserInput();
            }
            
            else if (user_input == '5') {
                keyboard_menu_RUDDER_SERVO_settings();
            }
            
            else if (user_input == '6') {
                keyboard_menu_HEADING_PID_settings();
            }  
            
            // go to tuning sub-menu
            else if (user_input == '7') {
                keyboard_menu_MANUAL_TUNING();
            }
            
//            else if (user_input == 'U') {
//                keyboard_menu_POSITION_READINGS();
//            }
            
            // go to sub-menus for the PID gains (this is blocking)
            else if (user_input == '1') {
                keyboard_menu_BCE_PID_settings();
            }
            else if (user_input == '2') {
                keyboard_menu_BATT_PID_settings();
            }
            else if (user_input == '3') {
                keyboard_menu_DEPTH_PID_settings();
            }
            else if (user_input == '4') {
                keyboard_menu_PITCH_PID_settings();
            }
                     
            else if (user_input == '*') {
                xbee().printf("SWITCHING TO SIMPLE MENU!\r\n"); 
                wait(1);
                _debug_menu_on = false;
            }
        }   //end of debug menu
/***************************** DEBUG MENU *****************************/
            
/***************************** SIMPLE MENU *****************************/
        else {     
            if (user_input == 'V') {
                _keyboard_state = POSITION_DIVE;
            }
            else if (user_input == 'N') {
                _keyboard_state = FIND_NEUTRAL;
            }
            else if (user_input == 'J') {
                _keyboard_state = FLOAT_LEVEL;
            }
            else if (user_input == 'B') {
                _keyboard_state = FLOAT_BROADCAST;
            }
            else if (user_input == 'E') {
                _keyboard_state = EMERGENCY_CLIMB;
            }
            
            // some debug tools below
            else if (user_input == 'P') {
                //Print current SD card log file
                //printCurrentSdLog();
                mbedLogger().printCurrentLogFile();        //print the current log file to the screen
            }
            
//POSITION DIVE COMMANDS
            else if (user_input == 'Q') {
                xbee().printf(">> Please enter the desired BMM offset (mm).\r\n");
                _BMM_dive_offset = getFloatUserInput();
            }
            else if (user_input == 'A') {
                xbee().printf(">> Please enter the desired BCE offset (mm).\r\n");
                _BCE_dive_offset = getFloatUserInput();
            }
            
            else if (user_input == 'S') {
                xbee().printf(">> Please enter the desired DEPTH (ft).\r\n");
                _depth_command = getFloatUserInput();
            }
//POSITION DIVE COMMANDS
            
            else if (user_input == '*') {
                xbee().printf("SWITCHING TO DEBUG MENU!\r\n"); 
                _debug_menu_on = true;
                wait(1);
            }
        }
/***************************** SIMPLE MENU *****************************/
        
        //when you read the keyboard successfully, change the state
        _state = _keyboard_state;   //set state at the end of this function
        //xbee().printf("\r\n\n ********* KEYBOARD STATE: %d *********\r\n\n", _state);
    }
}

void StateMachine::keyboard_menu_STREAM_STATUS() {
    char STATUS_key;
        
    // show the menu
    xbee().printf("\r\n8: STATUS DEBUG MENU (EXIT WITH 'X' !)\r\n");
    
    while (1) {
        if (xbee().readable()) {
            STATUS_key = xbee().getc();   //get each keystroke
        }
        
        else {
            
            wait(1);         
            
            
//            xbee().printf("BCE POS (CMD): %0.1f (%0.1f) BATT POS: %0.1f (%0.1f) PRESS_psi: %0.2f [depth_ft: %0.2f], PITCH: %0.2f, HEADING: %0.2f, rudder_servo_pwm: %0.1f \n[FILT/RAW 0(%d,%d),1(%d),2(%d),6(%d),4(%d),5(%d),6(%d),7(%d)] Switch: BCE(%d) BMM(%d)\r",bce().getPosition_mm(), bce().getSetPosition_mm(),batt().getPosition_mm(), batt().getSetPosition_mm(),depth().getPsi(),depthLoop().getPosition(),imu().getPitch(),imu().getHeading(),rudder().getSetPosition_pwm(),adc().readCh0(),adc().readCh1(),adc().readCh2(),adc().readCh3(),adc().readCh4(),adc().readCh5(),adc().readCh6(),adc().readCh7(), bce().getHardwareSwitchStatus(),batt().getHardwareSwitchStatus()); 
            xbee().printf("BCE POS: %0.1f (cmd %0.1f) BATT POS: %0.1f (cmd %0.1f) PRESS_psi: %0.2f [depth_ft: %0.2f], PITCH: %0.2f, HEADING: %0.2f, rdr_pwm: %0.1f [FILT/RAW 0(%d,%d),1(%d,%d),2(%d,%d),3(%d,%d),4(%d,%d),5(%d,%d),6(%d,%d),7(%d,%d)] Switch: BCE(%d) BMM(%d)\r",bce().getPosition_mm(), bce().getSetPosition_mm(),batt().getPosition_mm(), batt().getSetPosition_mm(),depth().getPsi(),depthLoop().getPosition(),imu().getPitch(),imu().getHeading(),rudder().getSetPosition_pwm(),adc().readCh0(),adc().readRawCh0(),adc().readCh1(),adc().readRawCh1(),adc().readCh2(),adc().readRawCh2(),adc().readCh3(),adc().readRawCh3(),adc().readCh4(),adc().readRawCh4(),adc().readCh5(),adc().readRawCh5(),adc().readCh6(),adc().readRawCh6(),adc().readCh7(),adc().readRawCh7(),bce().getHardwareSwitchStatus(),batt().getHardwareSwitchStatus()); 
            
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // process the keys            
        if (STATUS_key == 'X') {  
            xbee().printf("\r\nX: EXITING STATUS DEBUG MENU\r\n");             
            break;  //exit the while loop
        }
        
        else {
            xbee().printf("\r\nThis key (%c) does nothing here.                                  ", STATUS_key);
        }
    }
}

void StateMachine::keyboard_menu_RUDDER_SERVO_settings() {
    //load current parameters from the rudder
    float rudder_min_pwm = rudder().getMinPWM();
    float rudder_max_pwm = rudder().getMaxPWM();
    float rudder_ctr_pwm = rudder().getCenterPWM();
    float rudder_min_deg = rudder().getMinDeg();
    float rudder_max_deg = rudder().getMaxDeg();
    
    char RUDDER_PID_key;
 
    // print the menu
    xbee().printf("\r\nRUDDER (servo driver) settings (MENU)");
    xbee().printf("\r\nAdjust min_pwm/max_pwm/center_pwm/min_deg/max_deg settings with the following keys: N, M, C, K, L");
    xbee().printf("\r\nHit shift + X to exit w/o saving.  Hit shift + S to save.\r\n");
    xbee().printf("RUDDER min pwm: %f, max pwm: %f, center pwm: %f, min deg: %f, max deg: %f\r\n", rudder().getMinPWM(), rudder().getMaxPWM(), rudder().getCenterPWM(), rudder().getMinDeg(), rudder().getMaxDeg());
 
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            RUDDER_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
 
    // handle the user's key input                
        if (RUDDER_PID_key == 'S') { // user wants to save the modified values
            // set global values
            rudder().setMinPWM(rudder_min_pwm);
            rudder().setMaxPWM(rudder_max_pwm);
            rudder().setCenterPWM(rudder_ctr_pwm);
            rudder().setMinDeg(rudder_min_deg);
            rudder().setMaxDeg(rudder_max_deg);
            
            // save rudder servo driver values for inner loop
            configFileIO().saveRudderData(rudder_min_deg, rudder_max_deg, rudder_ctr_pwm, rudder_min_pwm, rudder_max_pwm);
            xbee().printf("RUDDER min pwm: %f, max pwm: %f, center pwm: %f, min deg: %f, max deg: %f\r\n", rudder().getMinPWM(), rudder().getMaxPWM(), rudder().getCenterPWM(), rudder().getMinDeg(), rudder().getMaxDeg());
            xbee().printf("Adjust min_pwm/max_pwm/center_pwm/min_deg/max_deg settings with the following keys: N, M, C, K, L\r\n");
        }
        else if (RUDDER_PID_key == 'X') {    
            break;  //exit the while loop
        }
            // MIN PWM
        else if (RUDDER_PID_key == 'N') {
            xbee().printf(">> Type in rudder_min_pwm with keyboard.\r\n");
            rudder_min_pwm = getFloatUserInput();
        }
    // MAX PWM
        else if (RUDDER_PID_key == 'M') {
            xbee().printf(">> Type in rudder_max_pwm with keyboard.\r\n");
            rudder_max_pwm = getFloatUserInput();
        }
    // CENTER PWM
        else if (RUDDER_PID_key == 'C') {
            xbee().printf(">> Type in rudder_ctr_pwm with keyboard.\r\n");
            rudder_ctr_pwm = getFloatUserInput();
        }
    // MIN DEG
        else if (RUDDER_PID_key == 'K') {
            xbee().printf(">> Type in rudder_min_deg with keyboard.\r\n");
            rudder_min_deg = getFloatUserInput();
        }
    // MAX DEG
        else if (RUDDER_PID_key == 'L') {
            xbee().printf(">> Type in rudder_max_deg with keyboard.\r\n");
            rudder_max_deg = getFloatUserInput();
        }       
        else {
            xbee().printf("RUDDER SETUP: [%c] This key does nothing here.                           \r", RUDDER_PID_key);
        }
    }
}

void StateMachine::keyboard_menu_COUNTS_STATUS() {
    // DELETE REMOVE  
}

void StateMachine::keyboard_menu_MANUAL_TUNING() {
    char TUNING_key;
        
    //made these into internal parameters
    float _tuning_bce_pos_mm = 200.0;   //safe starting position
    float _tuning_batt_pos_mm = 40.0;   //safe starting position
    float _tuning_rudder_pos_deg = 0.0; //safe starting position
    float _tuning_rudder_pwm = 1640.0;
    
    //immediately start at those positions
    bce().setPosition_mm(_tuning_bce_pos_mm);
    batt().setPosition_mm(_tuning_batt_pos_mm);
    rudder().setPosition_deg(_tuning_rudder_pos_deg);
    
    // show the menu
    xbee().printf("\r\n7: MANUAL TUNING MENU (EXIT WITH 'X' !) (Pause and Unpause rudder ticker with P and U\n");
    xbee().printf("\r\n(Adjust BCE and BATT positions in real-time.  Timeout NOT running! (decrease/increase BCE with A/S, BATT with Q/W, RUDDER with E/R)\r\n");
    xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        
    // what needs to be started?
    bce().unpause();    //this is now active
    batt().unpause();   //this is now active
    rudder().unpause();
    
    while (1) {
        wait(0.1);  
                      
        if (xbee().readable()) {
            TUNING_key = xbee().getc();   //get each keystroke
        }
        
        else {   
            //xbee().printf("MT: POS (CMD) BCE %3.1f mm (%3.1f mm), BATT %3.1f mm (%3.1f mm) SERVO: %0.1f deg, (%0.1f pwm) PITCH: %0.1f,HEADING: %0.1f                                  \r", bce().getPosition_mm(), bce().getSetPosition_mm(), batt().getPosition_mm(), batt().getSetPosition_mm(), rudder().getSetPosition_deg(), rudder().getSetPosition_pwm(),imu().getPitch(),imu().getHeading());         
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // process the keys            
        if (TUNING_key == 'X') {    
            // STOP THE MOTORS BEFORE LEAVING! (Just in case.)
            bce().pause();
            batt().pause();
            rudder().pause();
            
            //right now the rudder is always active................................................hmm
            //deactivate the pin? new/delete?
            
            break;  //exit the while loop
        }
        
        //Buoyancy Engine
        else if (TUNING_key == 'A' or TUNING_key == 'a') {
            _tuning_bce_pos_mm = _tuning_bce_pos_mm - 1.0;
            bce().setPosition_mm(_tuning_bce_pos_mm);              //this variable is loaded from the file at initialization
            xbee().printf("\r\nMANUAL_TUNING: (BCE CHANGE: %0.1f)\r\n BCE_position: %0.1f, BATT_position: %0.1f (depth: %0.1f ft, pitch: %0.1f deg)\r",bce().getSetPosition_mm(),bce().getPosition_mm(),batt().getPosition_mm(),depthLoop().getPosition(),pitchLoop().getPosition());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else if (TUNING_key == 'S' or TUNING_key == 's') {
            _tuning_bce_pos_mm = _tuning_bce_pos_mm + 1.0;
            bce().setPosition_mm(_tuning_bce_pos_mm);              //this variable is loaded from the file at initialization
            xbee().printf("\r\nMANUAL_TUNING: (BCE CHANGE: %0.1f)\r\n BCE_position: %0.1f, BATT_position: %0.1f (depth: %0.1f ft, pitch: %0.1f deg)\r",bce().getSetPosition_mm(),bce().getPosition_mm(),batt().getPosition_mm(),depthLoop().getPosition(),pitchLoop().getPosition());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        //BATTERY
        else if (TUNING_key == 'Q' or TUNING_key == 'q') {
            _tuning_batt_pos_mm = _tuning_batt_pos_mm - 1.0;
            batt().setPosition_mm(_tuning_batt_pos_mm);              //this variable is loaded from the file at initialization
            xbee().printf("\r\nMANUAL_TUNING: (BATT CHANGE: %0.1f)\r\n BCE_position: %0.1f, BATT_position: %0.1f (depth: %0.1f ft, pitch: %0.1f deg)\r",batt().getSetPosition_mm(),bce().getPosition_mm(),batt().getPosition_mm(),depthLoop().getPosition(),pitchLoop().getPosition());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else if (TUNING_key == 'W' or TUNING_key == 'w') {
            _tuning_batt_pos_mm = _tuning_batt_pos_mm + 1.0;
            batt().setPosition_mm(_tuning_batt_pos_mm);              //this variable is loaded from the file at initialization
            xbee().printf("\r\nMANUAL_TUNING: (BATT CHANGE: %0.1f)\r\n BCE_position: %0.1f, BATT_position: %0.1f (depth: %0.1f ft, pitch: %0.1f deg)\r",batt().getSetPosition_mm(),bce().getPosition_mm(),batt().getPosition_mm(),depthLoop().getPosition(),pitchLoop().getPosition());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else if (TUNING_key == 'c' or TUNING_key == 'C') {
            xbee().printf("\r\nMT: (CURRENT POSITIONS) BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        //RUDER
        else if (TUNING_key == 'R' or TUNING_key == 'r') {
            _tuning_rudder_pos_deg = _tuning_rudder_pos_deg - 0.5;
            rudder().setPosition_deg(_tuning_rudder_pos_deg);
            xbee().printf("MT: RUDDER CHANGE %0.1f deg [servo pwm: %f, %0.1f deg] (headingLoop heading: % 0.1f deg, IMU heading: %0.1f deg)\r\n", _tuning_rudder_pos_deg, rudder().getSetPosition_pwm(), rudder().getSetPosition_deg(), headingLoop().getPosition(), imu().getHeading());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else if (TUNING_key == 'E' or TUNING_key == 'e') {
            _tuning_rudder_pos_deg = _tuning_rudder_pos_deg + 0.5;
            rudder().setPosition_deg(_tuning_rudder_pos_deg);
            xbee().printf("MT: RUDDER CHANGE %0.1f deg [servo pwm: %f, %0.1f deg] (headingLoop heading: % 0.1f deg, IMU heading: %0.1f deg)\r\n", _tuning_rudder_pos_deg, rudder().getSetPosition_pwm(), rudder().getSetPosition_deg(), headingLoop().getPosition(), imu().getHeading());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else if (TUNING_key == '-' or TUNING_key == '_') {
            _tuning_rudder_pwm -= 10.0;
            rudder().setPWM(_tuning_rudder_pwm);
            xbee().printf("MT: (-) RUDDER CHANGE %0.1f pwm\n\r", rudder().getSetPosition_pwm());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
            
        else if (TUNING_key == '=' or TUNING_key == '+') {
            _tuning_rudder_pwm += 10.0;
            rudder().setPWM(_tuning_rudder_pwm);
            xbee().printf("MT: (+) RUDDER CHANGE %0.1f pwm\n\r", rudder().getSetPosition_pwm());
            xbee().printf("\r\nMT: BCE pos:%5.1f (cmd:%5.1f), BMM:%5.1f (cmd:%5.1f), RUDDER:%6.1f (%6.1f deg) (depth: %0.1f ft, pitch: %0.1f deg, headingLoop heading: %0.1f deg, IMU heading: %0.1f deg)                                                  \n\n\r",bce().getPosition_mm(),bce().getSetPosition_mm(),batt().getPosition_mm(),batt().getSetPosition_mm(),rudder().getSetPosition_pwm(),rudder().getSetPosition_deg(),depthLoop().getPosition(),pitchLoop().getPosition(), headingLoop().getPosition(),imu().getHeading());
        }
        
        else {
            xbee().printf("\r\nMANUAL_TUNING: [%c] This key does nothing here.                                  \r", TUNING_key);
        }            
    }
}

void StateMachine::keyboard_menu_CHANNEL_READINGS() {
    char TUNING_key;
        
    // show the menu
    xbee().printf("\r\n8: CHANNEL READINGS (EXIT WITH 'X' !)");
    
    while (1) {
        if (xbee().readable()) {
            TUNING_key = xbee().getc();   //get each keystroke
        }
        
                // process the keys            
        if (TUNING_key == 'X') {    
            // STOP THE MOTORS BEFORE LEAVING! (Just in case.)
            bce().pause();
            batt().pause();
            
            break;  //exit the while loop
        }
        
        else {
            wait(0.5);                    
            xbee().printf("0(%d),1(%d),2(%d),6(%d),4(%d),5(%d),6(%d),7(%d)\r\n",adc().readCh0(),adc().readCh1(),adc().readCh2(),adc().readCh3(),adc().readCh4(),adc().readCh5(),adc().readCh6(),adc().readCh7()); 
            continue; // didn't get a user input, so keep waiting for it
        }            
    }
}
 
void StateMachine::keyboard_menu_BCE_PID_settings() {    
    char BCE_PID_key;
    
    // load current values from files
    float bce_KP = bce().getControllerP();
    float bce_KI = bce().getControllerI();
    float bce_KD = bce().getControllerD();
    
    float bce_deadband = bce().getDeadband();
    float bce_frequency = bce().getFilterFrequency();
    int bce_zero_offset = bce().getZeroCounts(); 
    //BCE frequency and deadband are hardcoded!
 
    // show the menu
    xbee().printf("\n\rBuoyancy Engine PID gain settings (MENU). ADJUST WITH CARE!");
    xbee().printf("\n\rAdjust PID settings with the following keys: P  I  D. Filter = F, deadband = B, zero offset = Z\n\r");
    xbee().printf("\n\r(Hit shift + X to exit w/o saving.  Hit shift + S to save.)\n\n\n\r");
    xbee().printf("bce      P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", bce().getControllerP(), bce().getControllerI(), bce().getControllerD(), bce().getZeroCounts(), bce().getTravelLimit(), bce().getPotSlope(), bce().getFilterFrequency(), bce().getDeadband());    
    
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            BCE_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // handle the user's key input
        if (BCE_PID_key == 'S') { // user wants to save these modified values
            // set values
            bce().setControllerP(bce_KP);
            bce().setControllerI(bce_KI);
            bce().setControllerD(bce_KD); 
            
            bce().setDeadband(bce_deadband);
            bce().setFilterFrequency(bce_frequency);
            bce().setZeroCounts(bce_zero_offset); //integer value
 
            // save to "BCE.TXT" file
            //saveBattData(float batt_p_gain, float batt_i_gain, float batt_d_gain, int batt_zeroOffset, float batt_filter_freq, float batt_deadband)
            configFileIO().saveBCEData(bce_KP, bce_KI, bce_KD, bce_zero_offset, bce_frequency, bce_deadband);
            xbee().printf("bce      P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", bce().getControllerP(), bce().getControllerI(), bce().getControllerD(), bce().getZeroCounts(), bce().getTravelLimit(), bce().getPotSlope(), bce().getFilterFrequency(), bce().getDeadband());
        }
        else if (BCE_PID_key == 'X') {    
            break;  //exit the while loop
        }
        else if (BCE_PID_key == 'P') {
            xbee().printf(">> Type in proportional gain with keyboard.\n\r");
            bce_KP = getFloatUserInput();
        }
        else if (BCE_PID_key == 'I') {
            xbee().printf(">> Type in integral gain with keyboard.\n\r");
            bce_KI = getFloatUserInput();
        }
        else if (BCE_PID_key == 'D') {
            xbee().printf(">> Type in derivative gain with keyboard.\n\r");
            bce_KD = getFloatUserInput();
        }
        else if (BCE_PID_key == 'F') {
            xbee().printf(">> Type in FILTER FREQUENCY with keyboard.\n\r");
            bce_frequency = getFloatUserInput();
        }
        else if (BCE_PID_key == 'B') {
            xbee().printf(">> Type in DEADBAND with keyboard.\n\r");
            bce_deadband = getFloatUserInput();
        }
        else if (BCE_PID_key == 'Z') {
            xbee().printf(">> Type in zero count offset with keyboard.\n\r");
            bce_zero_offset = (int)getFloatUserInput();
        }
        else {
            xbee().printf("\n\rBCE: [%c] This key does nothing here.                                  \r", BCE_PID_key);
        }
    }
}

void StateMachine::keyboard_menu_BATT_PID_settings() {    
    char BMM_PID_key;

    // load current values from files
    float batt_KP = batt().getControllerP();
    float batt_KI = batt().getControllerI();
    float batt_KD = batt().getControllerD();
    
    float batt_deadband = batt().getDeadband();
    float batt_frequency = batt().getFilterFrequency();
    int batt_zero_offset = batt().getZeroCounts(); 
    //BATT frequency and deadband are hardcoded!
 
    // print the menu
    xbee().printf("\n\rBattery Motor PID gain settings (MENU)");
    xbee().printf("\n\rAdjust PID settings with the following keys: P I D. Filter = F, deadband = B.\n\r");
    xbee().printf("\n\r(Hit shift + X to exit w/o saving.  Hit shift + S to save.)\n\n\n\r");
    xbee().printf("batt     P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", batt().getControllerP(), batt().getControllerI(), batt().getControllerD(), batt().getZeroCounts(), batt().getTravelLimit(), batt().getPotSlope(), batt().getFilterFrequency(), batt().getDeadband());

    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            BMM_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // handle the user's key input
        if (BMM_PID_key == 'S') { // user wants to save these modified values
            // set values
            batt().setControllerP(batt_KP);
            batt().setControllerI(batt_KI);
            batt().setControllerD(batt_KD);
            
            batt().setDeadband(batt_deadband);
            batt().setFilterFrequency(batt_frequency);
            batt().setZeroCounts(batt_zero_offset); //integer value
 
            // save to "BATT.TXT" file
            //saveBCEData(float bce_p_gain, float bce_i_gain, float bce_d_gain, int bce_zeroOffset, float bce_filter_freq, float bce_deadband)
            configFileIO().saveBattData(batt_KP, batt_KI, batt_KD, batt_zero_offset, batt_frequency, batt_deadband);
            xbee().printf("batt     P:%6.2f, I:%6.2f, D:%6.2f,   zero offset: %3i, limit %6.1f mm, slope %0.5f, filter_freq: %0.1f, deadband: %0.1f\r\n", batt().getControllerP(), batt().getControllerI(), batt().getControllerD(), batt().getZeroCounts(), batt().getTravelLimit(), batt().getPotSlope(), batt().getFilterFrequency(), batt().getDeadband());
        }
        else if (BMM_PID_key == 'X') {    
            break;  //exit the while loop
        }
        else if (BMM_PID_key == 'P') {
            xbee().printf(">> Type in proportional gain with keyboard.\n\r");
            batt_KP = getFloatUserInput();
        }
        else if (BMM_PID_key == 'I') {
            xbee().printf(">> Type in integral gain with keyboard.\n\r");
            batt_KI = getFloatUserInput();
        }
        else if (BMM_PID_key == 'D') {
            xbee().printf(">> Type in derivative gain with keyboard.\n\r");
            batt_KD = getFloatUserInput();
        }
        else if (BMM_PID_key == 'F') {
            xbee().printf(">> Type in FILTER FREQUENCY with keyboard.\n\r");
            batt_frequency = getFloatUserInput();
        }
        else if (BMM_PID_key == 'B') {
            xbee().printf(">> Type in DEADBAND with keyboard.\n\r");
            batt_deadband = getFloatUserInput();
        }
        else if (BMM_PID_key == 'Z') {
            xbee().printf(">> Type in zero count offset with keyboard.\n\r");
            batt_zero_offset = (int)getFloatUserInput();
        }
        else {
            xbee().printf("\n\rBATT: [%c] This key does nothing here.                                  \r", BMM_PID_key);
        }
    }
}
 
void StateMachine::keyboard_menu_DEPTH_PID_settings() {    
    char DEPTH_PID_key;
    
    float depth_KP = depthLoop().getControllerP();       // load current depth value
    float depth_KI = depthLoop().getControllerI();       // load current depth value
    float depth_KD = depthLoop().getControllerD();       // load current depth value
    
    float depth_freq = depthLoop().getFilterFrequency();
    float depth_deadband = depthLoop().getDeadband();
 
    // print the menu
    xbee().printf("\n\rDEPTH (Buoyancy Engine O.L.) PID gain settings (MENU)");
    xbee().printf("\n\rAdjust PID settings with the following keys: P I D. Filter = F, deadband = B.\n\r");
    xbee().printf("\n\r(Hit shift + X to exit w/o saving.  Hit shift + S to save.)\n\n\n\r");
    xbee().printf("DEPTH P: %3.3f, I: %3.3f, D %3.3f, offset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n", depthLoop().getControllerP(), depthLoop().getControllerI(), depthLoop().getControllerD(),depthLoop().getOutputOffset(),depthLoop().getFilterFrequency(),depthLoop().getDeadband());
    
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            DEPTH_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // handle the user's key input
        if (DEPTH_PID_key == 'S') { // user wants to save these modified values
            // set values
            depthLoop().setControllerP(depth_KP);
            depthLoop().setControllerI(depth_KI);
            depthLoop().setControllerD(depth_KD);
            
            depthLoop().setFilterFrequency(depth_freq);
            depthLoop().setDeadband(depth_deadband);
 
            // save to "DEPTH.TXT" file
            configFileIO().saveDepthData(depth_KP, depth_KI, depth_KD, _neutral_bce_pos_mm, depth_freq, depth_deadband); //P,I,D, bce zeroOffset
            
            xbee().printf("DEPTH P: %3.3f, I: %3.3f, D %3.3f, offset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n", depthLoop().getControllerP(), depthLoop().getControllerI(), depthLoop().getControllerD(),depthLoop().getOutputOffset(),depthLoop().getFilterFrequency(),depthLoop().getDeadband());
            
            //set class variables that will be used in find neutral sequence
            _depth_KP = depthLoop().getControllerP();       // load current depth value
            _depth_KI = depthLoop().getControllerI();       // load current depth value
            _depth_KD = depthLoop().getControllerD();       // load current depth value
        }
        else if (DEPTH_PID_key == 'X') {    
            break;  //exit the while loop
        }
        else if (DEPTH_PID_key == 'P') {
            xbee().printf(">> Type in proportional gain with keyboard.\n\r");
            depth_KP = getFloatUserInput();
        }
        else if (DEPTH_PID_key == 'I') {
            xbee().printf(">> Type in integral gain with keyboard.\n\r");
            depth_KI = getFloatUserInput();
        }
        else if (DEPTH_PID_key == 'D') {
            xbee().printf(">> Type in derivative gain with keyboard.\n\r");
            depth_KD = getFloatUserInput();
        }
        else if (DEPTH_PID_key == 'F') {
            xbee().printf(">> Type in FILTER FREQUENCY with keyboard.\n\r");
            depth_freq = getFloatUserInput();
        }
        else if (DEPTH_PID_key == 'B') {
            xbee().printf(">> Type in DEADBAND with keyboard.\n\r");
            depth_deadband = getFloatUserInput();
        }
        else {
            xbee().printf("\n\rDEPTH: [%c] This key does nothing here.                                  \r", DEPTH_PID_key);
        }
    }
}
 
void StateMachine::keyboard_menu_PITCH_PID_settings() {    
    char PITCH_PID_key;
    
    float pitch_KP = pitchLoop().getControllerP();       // load current pitch value
    float pitch_KI = pitchLoop().getControllerI();       // load current pitch value
    float pitch_KD = pitchLoop().getControllerD();       // load current pitch value

    float pitch_freq = pitchLoop().getFilterFrequency();
    float pitch_deadband = pitchLoop().getDeadband();
 
    // print the menu
    xbee().printf("\n\rPITCH (Battery Motor O.L.) PID gain settings (MENU)");
    xbee().printf("\n\rAdjust PID settings with the following keys: P I D. Filter = F, deadband = B.\n\r");
    xbee().printf("\n\r(Hit shift + X to exit w/o saving.  Hit shift + S to save.)\n\n\n\r");
    xbee().printf("PITCH P: %3.3f, I: %3.3f, D %3.3f, offset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n", pitchLoop().getControllerP(), pitchLoop().getControllerI(), pitchLoop().getControllerD(), pitchLoop().getOutputOffset(),pitchLoop().getFilterFrequency(),pitchLoop().getDeadband());
    
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            PITCH_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // handle the user's key input
        if (PITCH_PID_key == 'S') { // user wants to save these modified values
            // set values
            pitchLoop().setControllerP(pitch_KP);
            pitchLoop().setControllerI(pitch_KI);
            pitchLoop().setControllerD(pitch_KD);
            
            pitchLoop().setFilterFrequency(pitch_freq);
            pitchLoop().setDeadband(pitch_deadband);
 
            // save to "PITCH.TXT" file (doesn't modify neutral position)
            configFileIO().savePitchData(pitch_KP, pitch_KI, pitch_KD, _neutral_batt_pos_mm, pitch_freq, pitch_deadband);
            
            xbee().printf("PITCH P: %3.3f, I: %3.3f, D %3.3f, zeroOffset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n", pitchLoop().getControllerP(), pitchLoop().getControllerI(), pitchLoop().getControllerD(), pitchLoop().getOutputOffset(),pitchLoop().getFilterFrequency(),pitchLoop().getDeadband());

            _pitch_KP = pitchLoop().getControllerP();       // load current pitch value
            _pitch_KI = pitchLoop().getControllerI();       // load current pitch value
            _pitch_KD = pitchLoop().getControllerD();       // load current pitch value
        }
        else if (PITCH_PID_key == 'X') {    
            break;  //exit the while loop
        }
        else if (PITCH_PID_key == 'P') {
            xbee().printf(">> Type in proportional gain with keyboard.\n\r");
            pitch_KP = getFloatUserInput();
        }
        else if (PITCH_PID_key == 'I') {
            xbee().printf(">> Type in integral gain with keyboard.\n\r");
            pitch_KI = getFloatUserInput();
        }
        else if (PITCH_PID_key == 'D') {
            xbee().printf(">> Type in derivative gain with keyboard.\n\r");
            pitch_KD = getFloatUserInput();
        }
        else if (PITCH_PID_key == 'F') {
            xbee().printf(">> Type in FILTER FREQUENCY with keyboard.\n\r");
            pitch_freq = getFloatUserInput();
        }
        else if (PITCH_PID_key == 'B') {
            xbee().printf(">> Type in DEADBAND with keyboard.\n\r");
            pitch_deadband = getFloatUserInput();
        }
        else {
            xbee().printf("\n\rPITCH: [%c] This key does nothing here.                                  \r", PITCH_PID_key);
        }
    }
}

void StateMachine::keyboard_menu_HEADING_PID_settings() {    
    char HEADING_PID_key;
    
    float heading_KP = headingLoop().getControllerP();
    float heading_KI = headingLoop().getControllerI();
    float heading_KD = headingLoop().getControllerD(); 
       
    float heading_offset_deg = headingLoop().getOutputOffset();
    float heading_freq = headingLoop().getFilterFrequency();
    float heading_deadband = headingLoop().getDeadband();
 
    // print the menu
    xbee().printf("\n\rHEADING (rudder outer loop) PID gain settings (MENU)");
    xbee().printf("\n\rAdjust PID settings with the following keys: P I D. Filter = F, deadband = B.\n\r");
    xbee().printf("\n\r   Adjust zero offset with O (oh).");
    xbee().printf("\n\r(Hit shift + X to exit w/o saving.  Hit shift + S to save.\n\r");
    xbee().printf("HEADING P: %3.3f, I: %3.3f, D %3.3f, zeroOffset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n\r\n", headingLoop().getControllerP(),headingLoop().getControllerI(),headingLoop().getControllerD(),headingLoop().getOutputOffset(),headingLoop().getFilterFrequency(),headingLoop().getDeadband());
 
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            HEADING_PID_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
 
        // handle the user's key input     
        if (HEADING_PID_key == 'S') { // user wants to save the modified values
            // set global values
            headingLoop().setControllerP(heading_KP);
            headingLoop().setControllerI(heading_KI);
            headingLoop().setControllerD(heading_KD);
            headingLoop().setOutputOffset(heading_offset_deg);
            headingLoop().setFilterFrequency(heading_freq);
            headingLoop().setDeadband(heading_deadband);
 
            // save pitch PID values for outer loop (must save neutral position also)
            configFileIO().saveHeadingData(heading_KP, heading_KI, heading_KD, heading_offset_deg, heading_freq, heading_deadband);    //_neutral_heading_pos_deg);
            xbee().printf("HEADING P: %3.3f, I: %3.3f, D %3.3f, zeroOffset: %3.1f mm (filter: %0.2f, deadband: %0.2f)\r\n\r\n", headingLoop().getControllerP(),headingLoop().getControllerI(),headingLoop().getControllerD(),headingLoop().getOutputOffset(),headingLoop().getFilterFrequency(),headingLoop().getDeadband());
        }
        else if (HEADING_PID_key == 'X') {    
            break;  //exit the while loop
        }
        
        else if (HEADING_PID_key == 'P') {
            heading_KP = getFloatUserInput();;
        }
        else if (HEADING_PID_key == 'I') {
            heading_KI = getFloatUserInput();
        }
        else if (HEADING_PID_key == 'D') {
            heading_KD = getFloatUserInput();
        }
        else if (HEADING_PID_key == 'F') {
            xbee().printf(">> Type in FILTER FREQUENCY with keyboard.\n\r");
            heading_freq = getFloatUserInput();
        }
        else if (HEADING_PID_key == 'B') {
            xbee().printf(">> Type in DEADBAND with keyboard.\n\r");
            heading_deadband = getFloatUserInput();
        }
        else if (HEADING_PID_key == 'O') {
            heading_offset_deg = getFloatUserInput();
        }
        else {
            xbee().printf("HEADING SETUP: [%c] This key does nothing here.                           \r", HEADING_PID_key);
        }
    }
}
 
float StateMachine::getDepthCommand() {
    return _depth_command;
}
 
float StateMachine::getPitchCommand() {
    return _pitch_command;
}

float StateMachine::getDepthReading() {
    return _depth_reading;
}
 
float StateMachine::getPitchReading() {
    return _pitch_reading;
}

float StateMachine::getTimerReading() {
    return _timer_reading;
}

void StateMachine::setState(int input_state) {
    _state = input_state;
    
    _isTimeoutRunning = false;  //to start each state you have to reset this
}
 
int StateMachine::getState() {
    return _state;  //return the current state of the system
}
 
void StateMachine::setTimeout(float input_timeout) {
    _timeout = input_timeout;
}
 
void StateMachine::setDepthCommand(float input_depth_command) {
    _depth_command = input_depth_command;
}
 
void StateMachine::setPitchCommand(float input_pitch_command) {
    _pitch_command = input_pitch_command;
}
 
void StateMachine::setNeutralPositions(float batt_pos_mm, float bce_pos_mm) {
    _neutral_batt_pos_mm = batt_pos_mm;
    _neutral_bce_pos_mm = bce_pos_mm;
    
    xbee().printf("Neutral Buoyancy Positions: batt: %0.1f, bce: %0.1f\r\n",_neutral_batt_pos_mm,_neutral_bce_pos_mm);
}
 
//process one state at a time
void StateMachine::getDiveSequence() {
    //iterate through this sequence using the FSM
    currentStateStruct.state = sequenceController().sequenceStructLoaded[_multi_dive_counter].state;
    currentStateStruct.timeout = sequenceController().sequenceStructLoaded[_multi_dive_counter].timeout;
    currentStateStruct.depth = sequenceController().sequenceStructLoaded[_multi_dive_counter].depth;
    currentStateStruct.pitch = sequenceController().sequenceStructLoaded[_multi_dive_counter].pitch;
    
    _timeout = currentStateStruct.timeout;  //set timeout before exiting this function
}
//process one state at a time
void StateMachine::getLegParams() {
    //iterate through this sequence using the FSM
    currentLegStateStruct.state = legController().legStructLoaded[_multi_leg_counter].state;
    currentLegStateStruct.timeout = legController().legStructLoaded[_multi_leg_counter].timeout;
    currentLegStateStruct.yo_time = legController().legStructLoaded[_multi_leg_counter].yo_time;  //timer for each up/down segment
    currentLegStateStruct.min_depth = legController().legStructLoaded[_multi_leg_counter].min_depth;
    currentLegStateStruct.max_depth = legController().legStructLoaded[_multi_leg_counter].max_depth;
    currentLegStateStruct.heading = legController().legStructLoaded[_multi_leg_counter].heading;
    
    _timeout = currentLegStateStruct.timeout;  //set timeout before exiting this function
    _yo_time = currentLegStateStruct.yo_time;
    _state   = currentLegStateStruct.state;
}

void StateMachine::printCurrentSdLog() {
    xbee().printf("SD card log work in progress\r\n");
    //might be worth saving the last few logs to the MBED...
}

// 06/06/2018
float StateMachine::getFloatUserInput() {
    float float_conversion = 0.0;
    
    while(1) {
        bool valid_input = false;                   //flag for valid or invalid input
        
        xbee().printf("\n\rPlease enter your number below and press ENTER:\r\n");
        char user_string [80];                      //variable to store input as a character array
    
        xbee().scanf("%s", user_string);              //read formatted data from stdin
        xbee().printf("\n\n\ruser_string was <%s>\r\n", user_string);
        
        //check through the string for invalid characters (decimal values 43 through 57)
        for (int c = 0; c < strlen(user_string); c++) {
            //xbee().printf("character is [%c]\r\n", user_string[c]);   //debug
            if (user_string[c] >= 43 and user_string[c] <= 57) {
                //xbee().printf("VALID CHARACTER!\r\n"); //debug
                ;
            }
            else {
                xbee().printf("INVALID INPUT!\r\n");
                break;
            }
            
            if (c == (strlen(user_string) - 1)) {
                valid_input = true;
            }
        }
        
        if (valid_input) {
            float_conversion = atof(user_string);
            xbee().printf("VALID INPUT!  Your input was: %3.3f (PRESS \"S\" (shift + S) to save!)\r\n", float_conversion);
            break;
        }
    }
    
    return float_conversion;
}

float StateMachine::getTimerValue() {
    return _fsm_timer;    
}

void StateMachine::logFileMenu() {    
    char FILE_MENU_key;
    
    // print the menu
    xbee().printf("\n\r>>> LOG FILE MENU. Y = Yes, erase file (and exit).  N = No, keep file (and exit).  P = Print file size.<<<\n\r");
    
    // handle the key presses
    while(1) {
        // get the user's keystroke from either of the two inputs
        if (xbee().readable()) {
            xbee().printf("\n\r>>> LOG FILE MENU. Y = Yes, erase file (and exit).  N = No, keep file (and exit).  P = Print file size.<<<\n\r");
            FILE_MENU_key = xbee().getc();
        }
        else {
            continue; // didn't get a user input, so keep waiting for it
        }
    
        // handle the user's key input
        if (FILE_MENU_key == 'P') { // user wants to save these modified values
            xbee().printf("\n\r>> Printing log file size!\n\r");
            wait(2);
            mbedLogger().getFileSize("/local/LOG000.csv");
        }
        else if (FILE_MENU_key == 'Y') {
            xbee().printf("\n\r>> Erasing MBED LOG FILE!\n\r");
            wait(2);
            mbedLogger().eraseFile();
            break;
        }
        else if (FILE_MENU_key == 'N') {
            xbee().printf("\n\r>> EXITING MENU. Log file intact.\n\r");
            wait(2);
            break;
        }
        else {
            xbee().printf("\n\r[%c] This key does nothing here.                                  \r", FILE_MENU_key);
        }
    }
}