#include "mbed.h"
#include "mDot.h"
#include "mDotEvent.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

// This example will setup a peer to peer node, copy program to two dots
// Uplink packets will be sent every 5 seconds, however communication is half-duplex
// An mDot cannot receive while transmitting
// Received packets will be sent to the Serial port on pins 2 and 3
// baud rate for serial ports is set to 115200


Serial _serial(XBEE_DOUT, XBEE_DIN);
Serial debug(USBTX, USBRX);

static uint8_t config_network_addr[] = { 0x01, 0x02, 0x03, 0x04 };
static uint8_t config_network_nskey[] = { 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04 };
static uint8_t config_network_dskey[] = { 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04 };

// Custom event handler for receiving Class C packets

class RadioEvent : public mDotEvent
{


public:
    RadioEvent() {}

    virtual ~RadioEvent() {}

    /*!
     * MAC layer event callback prototype.
     *
     * \param [IN] flags Bit field indicating the MAC events occurred
     * \param [IN] info  Details about MAC events occurred
     */
    virtual void MacEvent(LoRaMacEventFlags* flags, LoRaMacEventInfo* info) {

        if (mts::MTSLog::getLogLevel() == mts::MTSLog::TRACE_LEVEL) {
            std::string msg = "OK";
            switch (info->Status) {
                case LORAMAC_EVENT_INFO_STATUS_ERROR:
                    msg = "ERROR";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_TX_TIMEOUT:
                    msg = "TX_TIMEOUT";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_RX_TIMEOUT:
                    msg = "RX_TIMEOUT";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_RX_ERROR:
                    msg = "RX_ERROR";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_JOIN_FAIL:
                    msg = "JOIN_FAIL";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_DOWNLINK_FAIL:
                    msg = "DOWNLINK_FAIL";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_ADDRESS_FAIL:
                    msg = "ADDRESS_FAIL";
                    break;
                case LORAMAC_EVENT_INFO_STATUS_MIC_FAIL:
                    msg = "MIC_FAIL";
                    break;
                default:
                    break;
            }
            logTrace("Event: %s", msg.c_str());

            logTrace("Flags Tx: %d Rx: %d RxData: %d RxSlot: %d LinkCheck: %d JoinAccept: %d",
                     flags->Bits.Tx, flags->Bits.Rx, flags->Bits.RxData, flags->Bits.RxSlot, flags->Bits.LinkCheck, flags->Bits.JoinAccept);
            logTrace("Info: Status: %d ACK: %d Retries: %d TxDR: %d RxPort: %d RxSize: %d RSSI: %d SNR: %d Energy: %d Margin: %d Gateways: %d",
                     info->Status, info->TxAckReceived, info->TxNbRetries, info->TxDatarate, info->RxPort, info->RxBufferSize,
                     info->RxRssi, info->RxSnr, info->Energy, info->DemodMargin, info->NbGateways);
        }

        if (flags->Bits.Rx) {

            logDebug("Rx %d bytes", info->RxBufferSize);
            if (info->RxBufferSize > 0) {

                for (int i = 0; i < info->RxBufferSize; i++) {
                    _serial.putc(info->RxBuffer[i]);
                }
            }
        }
    }
};


int main()
{
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;
    std::string data_str = "hello!";

    RadioEvent events;

    debug.baud(115200);
    _serial.baud(115200);

    // get a mDot handle
    dot = mDot::getInstance();

    // Set custom events handler for receiving Class C packets
    dot->setEvents(&events);

    // print library version information
    logInfo("version: %s", dot->getId().c_str());

    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetConfig();

    dot->setLogLevel(mts::MTSLog::TRACE_LEVEL);

    // set up the mDot with our network information: Network Address and Session Keys must match on each device
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    std::vector<uint8_t> temp;

    for (int i = 0; i < 4; i++) {
        temp.push_back(config_network_addr[i]);
    }

    logInfo("setting network addr");
    if ((ret = dot->setNetworkAddress(temp)) != mDot::MDOT_OK) {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(config_network_nskey[i]);
    }

    logInfo("setting network password");
    if ((ret = dot->setNetworkSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(config_network_dskey[i]);
    }

    logInfo("setting network password");
    if ((ret = dot->setDataSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("setting TX frequency");
    if ((ret = dot->setTxFrequency(915500000)) != mDot::MDOT_OK) {
        logError("failed to set TX frequency %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, DR8-DR13
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::DR13)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    // set join mode to AUTO_OTA so the mDot doesn't have to rejoin after sleeping
    logInfo("setting join mode to AUTO_OTA");
    if ((ret = dot->setJoinMode(mDot::PEER_TO_PEER)) != mDot::MDOT_OK) {
        logError("failed to set join mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }

    //*******************************************
    // end of configuration
    //*******************************************

    // format data for sending to the gateway
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);

    while(true) {
        // send the data
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to peer");
        }

        wait(5.0);
    }

    return 0;
}
