#include "mbed.h"
#if defined (TARGET_MTS_MDOT_F411RE)
#include "spiffs.h"
#include "SpiFlash25.h"
#else
#include "xdot_eeprom.h"
#endif /* TARGET_MTS_MDOT_F411RE */
#include "mDot.h"
#include "ChannelPlan.h"
#include <stdint.h>
#include <string>
#include <vector>

#ifndef __LORA_CONFIG_H__
#define __LORA_CONFIG_H__

#if defined (TARGET_MTS_MDOT_F411RE)
// this value represents the number of files you can have open at the same time
// adjust it according to your requirements
#define MAX_CONCURRENT_FDS      5

#define PAGE_SIZE               256
#define SECTOR_SIZE             64*1024
#define MEM_SIZE                2*1024*1024
#else
#define SETTINGS_ADDR       0x0000      // configuration is 1024 bytes (0x000-0x3FF)
#define PROTECTED_ADDR      0x0400      // protected configuration is 256 bytes (0x400-0x4FF)
#define SESSION_ADDR        0x0500      // session is 256 bytes (0x500-0x5FF)
#define USER_ADDR           0x0800      // user space is 6*1024 bytes (0x800 - 0x1FFF)
#endif /* TARGET_MTS_MDOT_F411RE */

#define MULTICAST_SESSIONS 3
#define EUI_LENGTH 8
#define KEY_LENGTH 16
#define PASSPHRASE_LENGTH 128

// DON'T CHANGE THIS UNLESS YOU REALLY WANT TO
#define PROTECTED_RFU_SIZE 223

// PROTECTED SETTINGS SHOULD ALWAYS HAVE SIZE OF 256 BYTES
typedef struct {
        uint8_t FrequencyBand;
        uint8_t DeviceEUI[EUI_LENGTH];
        uint8_t AppEUI[EUI_LENGTH];
        uint8_t AppKey[KEY_LENGTH];
        uint8_t RFU[PROTECTED_RFU_SIZE];
} ProtectedSettings_t;

// DON'T CHANGE THIS UNLESS YOU REALLY WANT TO
#define CONFIG_RFU_SIZE 361

// SETTINGS SHOULD ALWAYS HAVE SIZE OF 1024 BYTES
typedef struct {
        uint8_t ConfigVersion;

        uint32_t SerialBaudRate;
        uint32_t DebugBaudRate;
        uint8_t StartUpMode;

        /* Network Settings */
        uint8_t NetworkEUI[EUI_LENGTH];
        uint8_t NetworkEUIPassphrase[PASSPHRASE_LENGTH];
        uint8_t NetworkKey[KEY_LENGTH];
        uint8_t NetworkKeyPassphrase[PASSPHRASE_LENGTH];

        uint32_t NetworkAddress;
        uint8_t NetworkSessionKey[KEY_LENGTH];
        uint8_t DataSessionKey[KEY_LENGTH];

        uint8_t JoinMode;
        uint8_t JoinRetries;

        /* Radio Settings */

        uint8_t ForwardErrorCorrection;
        uint8_t ACKAttempts;
        bool EnableEncryption;
        bool EnableCRC;
        bool EnableADR;
        bool EnableEcho;
        bool EnableVerbose;

        uint32_t TxFrequency;
        uint8_t TxDataRate;
        bool TxInverted;
        uint8_t TxPower;
        bool TxWait;
        uint8_t FrequencySubBand;

        uint32_t RxFrequency;
        uint8_t RxDataRate;
        bool RxInverted;
        uint8_t RxOutput;

        /* Serial Settings */
        uint32_t WakeInterval;
        uint16_t WakeTimeout;
        uint32_t WakeDelay;

        uint8_t PublicNetwork;
        uint8_t LinkCheckCount;
        uint8_t LinkCheckThreshold;

        uint8_t LogLevel;
        uint8_t JoinByteOrder;

        uint8_t WakePin;
        uint8_t WakeMode;

        uint8_t PreserveSessionOverReset;

        uint8_t JoinDelay;
        uint8_t RxDelay;
        uint8_t Port;

        uint8_t Class;
        int8_t AntennaGain;

        bool FlowControl;
        uint8_t Repeat;

        bool SerialClearOnError;
        uint8_t Rx2Datarate;
        uint8_t JoinRx1DatarateOffset;  //!< Offset for datarate for first window
        uint8_t JoinRx2DatarateIndex;   //!< Datarate for second window

        uint32_t Channels[16];
        uint8_t ChannelRanges[16];

        uint32_t JoinRx2Frequency;      //!< Frequency used in second window

        uint8_t MaxEIRP;
        uint8_t UlDwellTime;
        uint8_t DlDwellTime;

        uint8_t padding;

        uint32_t DlChannels[16];

        uint8_t LastPlan;               //!< Channel plan from last saved configuration

        int8_t lbtThreshold;            //!< Listen before talk threshold
        uint16_t lbtTimeUs;             //!< Listen before talk time

        // Multicast session address and keys
        uint32_t MulticastAddress[MULTICAST_SESSIONS];
        uint8_t MulticastNetSessionKey[MULTICAST_SESSIONS][KEY_LENGTH];
        uint8_t MulticastAppSessionKey[MULTICAST_SESSIONS][KEY_LENGTH];

        uint8_t AutoSleep;

        uint8_t RFU[CONFIG_RFU_SIZE];
} LoRaSettings_t;

// DON'T CHANGE THIS UNLESS YOU REALLY WANT TO
#define SESSION_RFU_SIZE 2

// SESSION SETTINGS SHOULD ALWAYS HAVE SIZE OF 256 BYTES
typedef struct {
        bool Joined;
        uint8_t Rx1DatarateOffset;
        uint8_t Rx2Datarate;
        uint8_t ChannelMask500k;

        uint32_t NetworkAddress;

        uint8_t NetworkKey[KEY_LENGTH];
        uint8_t DataKey[KEY_LENGTH];

        uint64_t ChannelMask;

        uint32_t Channels[16];
        uint8_t ChannelRanges[16];

        uint32_t UplinkCounter;

        uint8_t Rx1Delay;
        uint8_t Datarate;
        uint8_t TxPower;
        uint8_t Repeat;

        uint32_t Rx2Frequency;

        uint32_t DownlinkCounter;

        uint8_t MaxDutyCycle;
        uint8_t AdrAckCounter;
        uint8_t LinkFailCount;
        uint8_t FrequencySubBand;

        uint32_t NetworkId;

        bool ServerAckRequested;
        uint8_t DeviceClass;

        uint8_t CommandBufferIndex;

        uint8_t CommandBuffer[15];

        uint8_t UlDwellTime;
        uint8_t DlDwellTime;

        uint32_t DlChannels[16];

        uint8_t MaxEIRP;

        // Multicast session counter values
        uint32_t MulticastCounters[MULTICAST_SESSIONS];

        uint8_t RFU[SESSION_RFU_SIZE];

        uint8_t LastPlan;               //!< Channel plan from last saved session

} NetworkSession_t;

class LoRaConfig {

    public:

        static const uint8_t EuiLength = EUI_LENGTH;
        static const uint8_t KeyLength = KEY_LENGTH;
        static const uint8_t PassPhraseLength = PASSPHRASE_LENGTH;

        enum JoinMode {
            MANUAL,
            OTA,
            AUTO_OTA,
            PEER_TO_PEER
        };

        enum Mode {
            COMMAND_MODE,
            SERIAL_MODE
        };

        enum RX_Output {
            HEXADECIMAL,
            BINARY
        };

        enum DataRates {
            DR0,
            DR1,
            DR2,
            DR3,
            DR4,
            DR5,
            DR6,
            DR7,
            DR8,
            DR9,
            DR10,
            DR11,
            DR12,
            DR13,
            DR14,
            DR15
        };

        enum FrequencySubBands {
            FSB_ALL,
            FSB_1,
            FSB_2,
            FSB_3,
            FSB_4,
            FSB_5,
            FSB_6,
            FSB_7,
            FSB_8
        };

        enum JoinByteOrder {
            LSB,
            MSB
        };

        static const uint32_t BaudRates[];
        static const std::string DRS[];
        static const uint32_t DefaultBaudRate;

        uint32_t ValidateBaudRate(uint32_t rate);

        LoRaConfig(lora::ChannelPlan *plan);
        ~LoRaConfig();

        bool Save();
        bool SaveSession();
        bool SaveProtected();

#if defined (TARGET_MTS_MDOT_F411RE)
        void EnablePVD();
        bool PVDO();
#endif /* TARGET_MTS_MDOT_F411RE */

        void Mount();
        void Load();
        void Default();
        void DefaultSession();
        void DefaultProtected();

        uint8_t PublicNetwork();
        void PublicNetwork(uint8_t val);

        uint8_t* DeviceEUI();
        void DeviceEUI(uint8_t* devEUI);

        const uint8_t* AppEUI();
        void AppEUI(const uint8_t* appEUI);

        const uint8_t* AppKey();
        void AppKey(const uint8_t* appKey);

        uint8_t* NetworkEUI();
        void NetworkEUI(const uint8_t* networkEUI);

        uint8_t* NetworkEUIPassphrase();
        void NetworkEUIPassphrase(uint8_t* networkEUIPassphrase, uint8_t length);

        uint8_t* NetworkKey();
        void NetworkKey(const uint8_t* networkKey);

        uint8_t* NetworkKeyPassphrase();
        void NetworkKeyPassphrase(uint8_t* networkKeyPassphrase, uint8_t length);

        uint32_t NetworkAddress();
        void NetworkAddress(uint32_t addr);

        uint8_t* NetworkSessionKey();
        void NetworkSessionKey(uint8_t* networkSessionKey);

        uint8_t* DataSessionKey();
        void DataSessionKey(uint8_t* dataSessionKey);

        uint8_t JoinMode();
        void JoinMode(uint8_t mode);

        uint8_t JoinRetries();
        void JoinRetries(uint8_t retries);

        uint8_t LinkCheckCount();
        void LinkCheckCount(uint8_t count);

        uint8_t LinkCheckThreshold();
        void LinkCheckThreshold(uint8_t count);

        uint8_t ForwardErrorCorrection();
        void ForwardErrorCorrection(uint8_t bytes);

        uint8_t ACKAttempts();
        void ACKAttempts(uint8_t attempts);

        bool EnableEncryption();
        void EnableEncryption(uint8_t enable);

        bool EnableCRC();
        void EnableCRC(uint8_t enable);

        bool EnableADR();
        void EnableADR(uint8_t enable);

        bool EnableEcho();
        void EnableEcho(uint8_t enable);

        bool EnableVerbose();
        void EnableVerbose(uint8_t enable);

        uint8_t TxDataRate();
        void TxDataRate(uint8_t rate);

        uint8_t TxPower();
        void TxPower(uint8_t power);

        uint8_t FrequencyBand();
        void FrequencyBand(uint8_t band);

        uint8_t FrequencySubBand();
        void FrequencySubBand(uint8_t band);

        uint32_t TxFrequency();
        void TxFrequency(uint32_t freq);

        bool TxInverted();
        void TxInverted(bool enable);

        bool TxWait();
        void TxWait(bool timeout);

        uint32_t RxFrequency();
        void RxFrequency(uint32_t freq);

        bool RxInverted();
        void RxInverted(bool enable);

        uint8_t RxOutput();
        void RxOutput(uint8_t type);

        uint8_t RxDataRate();
        void RxDataRate(uint8_t rate);

        uint8_t Rx2DataRate();
        void Rx2DataRate(uint8_t rate);

        uint8_t StartUpMode();
        void StartUpMode(uint8_t mode);

        /* Serial Settings */
        uint32_t SerialBaudRate();
        void SerialBaudRate(uint32_t rate);

        uint32_t WakeInterval();
        void WakeInterval(uint32_t interval);

        uint32_t WakeDelay();
        void WakeDelay(uint32_t delay);

        uint16_t WakeTimeout();
        void WakeTimeout(uint16_t timeout);

        uint8_t WakePin();
        void WakePin(uint8_t pin);

        uint8_t WakeMode();
        void WakeMode(uint8_t mode);

        uint32_t DebugBaudRate();
        void DebugBaudRate(uint32_t baud);

        bool Joined();
        void Joined(bool val);

        uint32_t* Channels();

        uint32_t* DownlinkChannels();

        uint8_t* ChannelRanges();

        uint8_t* SessionDataKey();
        void SessionDataKey(uint8_t* key);

        uint8_t* SessionNetworkKey();
        void SessionNetworkKey(uint8_t* key);

        uint32_t SessionNetworkAddress();
        void SessionNetworkAddress(uint32_t addr);

        uint8_t SessionDatarate();
        void SessionDatarate(uint8_t datarate);

        uint8_t SessionTxPower();
        void SessionTxPower(uint8_t power);

        uint8_t SessionRepeat();
        void SessionRepeat(uint8_t repeat);

        uint8_t SessionRxDelay();
        void SessionRxDelay(uint8_t delay);

        uint8_t SessionRx1DatarateOffset();
        void SessionRx1DatarateOffset(uint8_t offset);

        uint8_t SessionRx2Datarate();
        void SessionRx2Datarate(uint8_t datarate);

        uint32_t SessionRx2Frequency();
        void SessionRx2Frequency(uint32_t frequency);

        uint64_t SessionChannelMask();
        void SessionChannelMask(uint64_t mask);

        uint8_t Session500kMask();
        void Session500kMask(uint8_t mask);

        uint32_t* SessionChannels();

        uint32_t* SessionDownlinkChannels();

        uint8_t* SessionChannelRanges();

        uint32_t SessionUplinkCounter();
        void SessionUplinkCounter(uint32_t count);

        uint32_t SessionDownlinkCounter();
        void SessionDownlinkCounter(uint32_t count);

        uint8_t SessionAdrAckCounter();
        void SessionAdrAckCounter(uint8_t count);

        uint8_t SessionLinkFailCount();
        void SessionLinkFailCount(uint8_t count);

        uint8_t SessionFrequencySubBand();
        void SessionFrequencySubBand(uint8_t band);

        bool SessionServerAckRequested();
        void SessionServerAckRequested(bool val);

        uint32_t SessionNetworkId();
        void SessionNetworkId(uint32_t id);

        uint8_t SessionDeviceClass();
        void SessionDeviceClass(uint8_t cls);

        uint8_t* SessionCommandBuffer();
        void SessionCommandBuffer(uint8_t* buffer);

        uint8_t SessionCommandBufferIndex();
        void SessionCommandBufferIndex(uint8_t index);

        uint8_t SessionMaxDutyCycle();
        void SessionMaxDutyCycle(uint8_t duty);

        uint8_t SessionUplinkDwelltime();
        void SessionUplinkDwelltime(uint8_t dwell);

        uint8_t SessionDownlinkDwelltime();
        void SessionDownlinkDwelltime(uint8_t dwell);

        uint8_t SessionMaxEIRP();
        void SessionMaxEIRP(uint8_t pow);

        uint8_t LogLevel();
        void LogLevel(uint8_t level);

        uint8_t JoinByteOrder();
        void JoinByteOrder(uint8_t order);

        uint8_t PreserveSessionOverReset();
        void PreserveSessionOverReset(uint8_t);

        uint8_t JoinDelay();
        void JoinDelay(uint8_t delay);

        uint8_t JoinRx1DatarateOffset();
        void JoinRx1DatarateOffset(uint8_t offset);

        uint8_t JoinRx2DatarateIndex();
        void JoinRx2DatarateIndex(uint8_t datarate);

        uint32_t JoinRx2Frequency();
        void JoinRx2Frequency(uint32_t freq);

        uint8_t RxDelay();
        void RxDelay(uint8_t delay);

        uint8_t Port();
        void Port(uint8_t port);

        uint8_t Class();
        void Class(uint8_t cls);

        int8_t AntennaGain();
        void AntennaGain(int8_t gain);

        bool FlowControl();
        void FlowControl(bool enable);

        uint8_t Repeat();
        void Repeat(uint8_t repeat);

        bool SerialClearOnError();
        void SerialClearOnError(bool val);

        uint8_t LastPlan();
        uint8_t SessionLastPlan();

        void LbtTimeUs(uint16_t us);
        uint16_t LbtTimeUs();
        void LbtThreshold(int8_t rssi);
        int8_t LbtThreshold();

        uint32_t GetMulticastAddress(int i);
        uint8_t* GetMulticastNetSessionKey(int i);
        uint8_t* GetMulticastAppSessionKey(int i);
        uint32_t GetMulticastCounter(int i);


        void SetMulticastAddress(int i, uint32_t addr);
        void SetMulticastNetSessionKey(int i, const uint8_t* key);
        void SetMulticastAppSessionKey(int i, const uint8_t* key);
        void SetMulticastCounter(int i, uint32_t count);

        void AutoSleep(uint8_t enable);
        uint8_t AutoSleep();

        /*!
         * AES encryption/decryption cipher network app ieee eui
         */
        static const uint8_t defaultNetworkEUI[EUI_LENGTH];
        /*!
         * AES encryption/decryption cipher application key
         */
        static const uint8_t defaultNetworkKey[KEY_LENGTH];

        /*!
         * AES encryption/decryption cipher network session key
         */
        static const uint8_t defaultNetworkSessionKey[KEY_LENGTH];

        /*!
         * AES encryption/decryption cipher application session key
         */
        static const uint8_t defaultDataSessionKey[KEY_LENGTH];

        void Sleep();
        void Wakeup();

        void SetChannelPlan(lora::ChannelPlan *plan);

#if defined (TARGET_MTS_MDOT_F411RE)
        mDot::mdot_file OpenUserFile(const char* file, int mode);
        bool SeekUserFile(mDot::mdot_file& file, size_t offset, int whence);
        int ReadUserFile(mDot::mdot_file& file, void* data, size_t length);
        int WriteUserFile(mDot::mdot_file& file, void* data, size_t length);
        bool CloseUserFile(mDot::mdot_file& file);
        bool MoveUserFile(mDot::mdot_file& file, const char* new_name);

        bool MoveUserFile(const char* file, const char* new_name);
        bool SaveUserFile(const char* file, void* data, uint32_t size);
        bool AppendUserFile(const char* file, void* data, uint32_t size);
        bool ReadUserFile(const char* file, void* data, uint32_t size);
        bool DeleteUserFile(const char* file);
        bool DeleteFile(const char* file);

        bool MoveUserFileToFirwareUpgrade(const char* file);

        std::vector<mDot::mdot_file> ListUserFiles();
#endif /* TARGET_MTS_MDOT_F411RE */

    private:
        LoRaSettings_t _settings;
        ProtectedSettings_t _protected;
        NetworkSession_t _session;
        bool _dirty_settings;

        uint32_t _frequencyMin;
        uint32_t _frequencyMax;

        lora::ChannelPlan *_plan;

#if defined (TARGET_MTS_MDOT_F411RE)
        // SpiFlash25 flash(MOSI, MISO, SCK, CS, W, HOLD);
        static SpiFlash25 _flash;

        mDot::mdot_file OpenFile(spiffs *fs, const char* file, int mode);
        bool SeekFile(spiffs *fs, mDot::mdot_file& file, size_t offset, int whence);

        int ReadFile(spiffs *fs, mDot::mdot_file& file, void* data, size_t length);
        int WriteFile(spiffs *fs, mDot::mdot_file& file, void* data, size_t length);
        int MoveFile(spiffs* fs, mDot::mdot_file& file, const char* new_name);
        bool CloseFile(spiffs *fs, mDot::mdot_file& file);

        bool AppendFile(spiffs *fs, const char* file, void* data, uint32_t size);
        bool SaveFile(spiffs *fs, const char* file, void* data, uint32_t size);
        bool ReadFile(spiffs *fs, const char* file, void* dest, uint32_t size);
        bool MoveFile(spiffs *fs, const char* file, const char* new_name);

        // glue code between SPI driver and filesystem
        static int spi_read(unsigned int addr, unsigned int size, unsigned char* data);
        static int spi_write(unsigned int addr, unsigned int size, unsigned char* data);
        static int spi_erase(unsigned int addr, unsigned int size);

        static u8_t spiffs_work_buf[PAGE_SIZE * 2];
        static u8_t spiffs_fds[32 * MAX_CONCURRENT_FDS];
        static u8_t spiffs_cache_buf[(PAGE_SIZE + 32) * 4];

        u8_t _openFds;

        static spiffs _fs;

        static char file[];
        static char protected_file[];
        static char session_file[];
        static char user_dir[];
#endif /* TARGET_MTS_MDOT_F411RE */

};

#endif // __LORA_CONFIG_H__
