/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2013 Semtech

Description: Generic SX1272 driver implementation

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#ifndef __SXRADIO1272_H__
#define __SXRADIO1272_H__

#include "SxRadio.h"
#include "SxRadio1272Regs-Fsk.h"
#include "SxRadio1272Regs-LoRa.h"

/*!
 * Radio wakeup time from SLEEP mode
 */
#if defined(TARGET_XDOT_L151CC)
#define RADIO_WAKEUP_TIME                           4 // [ms]
#else
#define RADIO_WAKEUP_TIME                           2 // [ms]
#endif

/*!
 * SX1272 definitions
 */
#define XTAL_FREQ                                   32000000
#define FREQ_STEP                                   61.03515625

#define RX_BUFFER_SIZE                              256

/*!
 * ============================================================================
 * Public functions prototypes
 * ============================================================================
 */

class SxRadio1272 : public SxRadio
{
public:
    enum Bandwidth {
        BW_125, BW_250, BW_500
    };

    SxRadio1272(PinName reset, PinName DIO0, PinName DIO1, PinName DIO2,
                PinName DIO3, PinName DIO4, PinName mosi, PinName miso,
                PinName sclk, PinName nss, osPriority priority = osPriorityAboveNormal);
    virtual ~SxRadio1272() {}

    /*!
     * \brief Initializes the radio
     *
     * \param [IN] events Structure containing the driver callback functions
     */
    virtual void Init( SxRadioEvents *events );
    /*!
     * \brief Prepares the radio for destruction
     */
    virtual void Terminate( void );
    /*!
     * \brief Configures the radio with the given modem
     *
     * \param [IN] modem Modem to be used [0: FSK, 1: LoRa] 
     */
    virtual void SetModem( RadioModems_t modem );
    /*!
     * \brief Sets the channel frequency
     *
     * \param [IN] freq         Channel RF frequency
     */
    virtual void SetChannel( uint32_t freq );
    /*!
     * \brief Detect channel free using CAD
     *
     * \param [IN] modem      Radio modem to be used [0: FSK, 1: LoRa]
     * \param [IN] freq       Channel RF frequency
     * \param [IN] rssiThresh RSSI threshold
     * \param [IN] bandwidth 0:125k, 1:250k, 2:500k
     * \param [IN] rssiVal    pointer to variable to hold RSSI value if desired - ignored if NULL
     *
     * \retval isFree         [true: Channel is free, false: Channel is not free]
     */
    virtual bool IsChannelFree( RadioModems_t modem, uint32_t freq, uint8_t datarate, int16_t rssiThresh, uint8_t bandwidth, uint32_t timeout = 5000, int16_t *rssiVal = NULL );
    /*!
     * \brief Generates a 32 bits random value based on the RSSI readings
     *
     * \remark This function sets the radio in LoRa modem mode and disables 
     *         all interrupts.
     *         After calling this function either Radio.SetRxConfig or
     *         Radio.SetTxConfig functions must be called.
     *
     * \retval randomValue    32 bits random value
     */
    virtual uint32_t Random( void );
    /*!
     * \brief Sets the reception parameters
     *
     * \param [IN] modem        Radio modem to be used [0: FSK, 1: LoRa]
     * \param [IN] bandwidth    Sets the bandwidth
     *                          FSK : >= 2600 and <= 250000 Hz
     *                          LoRa: [0: 125 kHz, 1: 250 kHz,
     *                                 2: 500 kHz, 3: Reserved] 
     * \param [IN] datarate     Sets the Datarate
     *                          FSK : 600..300000 bits/s
     *                          LoRa: [6: 64, 7: 128, 8: 256, 9: 512,
     *                                10: 1024, 11: 2048, 12: 4096  chips]
     * \param [IN] coderate     Sets the coding rate (LoRa only)
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [1: 4/5, 2: 4/6, 3: 4/7, 4: 4/8] 
     * \param [IN] bandwidthAfc Sets the AFC Bandwidth (FSK only) 
     *                          FSK : >= 2600 and <= 250000 Hz
     *                          LoRa: N/A ( set to 0 ) 
     * \param [IN] preambleLen  Sets the Preamble length
     *                          FSK : Number of bytes 
     *                          LoRa: Length in symbols (the hardware adds 4 more symbols)
     * \param [IN] symbTimeout  Sets the RxSingle timeout value (LoRa only) 
     *                          FSK : N/A ( set to 0 ) 
     *                          LoRa: timeout in symbols
     * \param [IN] fixLen       Fixed length packets [0: variable, 1: fixed]
     * \param [IN] payloadLen   Sets payload length when fixed length is used
     * \param [IN] crcOn        Enables/Disables the CRC [0: OFF, 1: ON]
     * \param [IN] FreqHopOn    Enables disables the intra-packet frequency hopping
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [0: OFF, 1: ON]
     * \param [IN] HopPeriod    Number of symbols bewteen each hop
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: Number of symbols
     * \param [IN] iqInverted   Inverts IQ signals (LoRa only)
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [0: not inverted, 1: inverted]
     * \param [IN] rxContinuous Sets the reception in continuous mode
     *                          [false: single mode, true: continuous mode]
     */
    virtual void SetRxConfig( RadioModems_t modem, uint32_t bandwidth,
                              uint32_t datarate, uint8_t coderate,
                              uint32_t bandwidthAfc, uint16_t preambleLen,
                              uint16_t symbTimeout, bool fixLen,
                              uint8_t payloadLen,
                              bool crcOn, bool FreqHopOn, uint8_t HopPeriod,
                              bool iqInverted, bool rxContinuous );
    /*!
     * \brief Sets the transmission parameters
     *
     * \param [IN] modem        Radio modem to be used [0: FSK, 1: LoRa] 
     * \param [IN] power        Sets the output power [dBm]
     * \param [IN] fdev         Sets the frequency deviation (FSK only)
     *                          FSK : [Hz]
     *                          LoRa: 0
     * \param [IN] bandwidth    Sets the bandwidth (LoRa only)
     *                          FSK : 0
     *                          LoRa: [0: 125 kHz, 1: 250 kHz,
     *                                 2: 500 kHz, 3: Reserved] 
     * \param [IN] datarate     Sets the Datarate
     *                          FSK : 600..300000 bits/s
     *                          LoRa: [6: 64, 7: 128, 8: 256, 9: 512,
     *                                10: 1024, 11: 2048, 12: 4096  chips]
     * \param [IN] coderate     Sets the coding rate (LoRa only)
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [1: 4/5, 2: 4/6, 3: 4/7, 4: 4/8] 
     * \param [IN] preambleLen  Sets the preamble length
     *                          FSK : Number of bytes 
     *                          LoRa: Length in symbols (the hardware adds 4 more symbols)
     * \param [IN] fixLen       Fixed length packets [0: variable, 1: fixed]
     * \param [IN] crcOn        Enables disables the CRC [0: OFF, 1: ON]
     * \param [IN] FreqHopOn    Enables disables the intra-packet frequency hopping
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [0: OFF, 1: ON]
     * \param [IN] HopPeriod    Number of symbols bewteen each hop
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: Number of symbols
     * \param [IN] iqInverted   Inverts IQ signals (LoRa only)
     *                          FSK : N/A ( set to 0 )
     *                          LoRa: [0: not inverted, 1: inverted]
     * \param [IN] timeout      Transmission timeout [us]
     */
    virtual void SetTxConfig( RadioModems_t modem, int8_t power, uint32_t fdev, 
                              uint32_t bandwidth, uint32_t datarate,
                              uint8_t coderate, uint16_t preambleLen,
                              bool fixLen, bool crcOn, bool FreqHopOn,
                              uint8_t HopPeriod, bool iqInverted, uint32_t timeout );

    /**
     * Generate CW with current TX config
     */
    virtual void SetTxContinuous(bool enable);

    /*!
     * \brief Computes the packet time on air for the given payload
     *
     * \Remark Can only be called once SetRxConfig or SetTxConfig have been called
     *
     * \param [IN] modem      Radio modem to be used [0: FSK, 1: LoRa]
     * \param [IN] pktLen     Packet payload length
     *
     * \retval airTime        Computed airTime for the given packet payload length
     */
    virtual double TimeOnAir( RadioModems_t modem, uint8_t pktLen );
    /*!
     * \brief Sends the buffer of size. Prepares the packet to be sent and sets
     *        the radio in transmission
     *
     * \param [IN]: buffer     Buffer pointer
     * \param [IN]: size       Buffer size
     */
    virtual void Send( const uint8_t *buffer, uint8_t size );
    /*!
     * \brief Sets the radio in sleep mode
     */
    virtual void Sleep( void );
    /*!
     * \brief Sets the radio in standby mode
     */
    virtual void Standby( void );
    /*!
     * \brief Sets the radio in reception mode for the given time
     * \param [IN] timeout Reception timeout [ms]
     *                     [0: continuous, others timeout]
     */
    virtual void Rx( uint32_t timeout );
    /*!
     * \brief Start a Channel Activity Detection
     */
    virtual void StartCad( void );
    /*!
     * \brief Reads the current RSSI value
     *
     * \retval rssiValue Current RSSI value in [dBm]
     */
    virtual int16_t Rssi( RadioModems_t modem );
    /*!
     * \brief Writes the radio register at the specified address
     *
     * \param [IN]: addr Register address
     * \param [IN]: data New register value
     */
    virtual void Write( uint8_t addr, uint8_t data );
    /*!
     * \brief Reads the radio register at the specified address
     *
     * \param [IN]: addr Register address
     * \retval data Register value
     */
    virtual uint8_t Read ( uint8_t addr );
    /*!
     * \brief Writes multiple radio registers starting at address
     *
     * \param [IN] addr   First Radio register address
     * \param [IN] buffer Buffer containing the new register's values
     * \param [IN] size   Number of registers to be written
     */
    virtual void WriteBuffer( uint8_t addr, const uint8_t *buffer, uint8_t size );
    /*!
     * \brief Reads multiple radio registers starting at address
     *
     * \param [IN] addr First Radio register address
     * \param [OUT] buffer Buffer where to copy the registers data
     * \param [IN] size Number of registers to be read
     */
    virtual void ReadBuffer( uint8_t addr, uint8_t *buffer, uint8_t size );

    virtual void SignalMacEvent(void);

    virtual void CheckForReset(void);
    
    virtual void ResetRadio();

    virtual uint32_t GetTimeOnAir();
protected:
    /*!
     * \brief Set antenna switch to low power
     *
     * \param [IN] status [true: Antenna in low power, false: Antenna active]
     */
    virtual void SetAntSwLowPower( bool status ) { }

    /*!
     * \brief Set antenna direction
     *
     * \param [IN] rxTx [1: transmitter, 0: receiver]
     */
    virtual void SetAntSw( uint8_t rxTx ) { }

    /*!
     * \brief Get power amp setting
     *
     * \param [IN] channel Channel RF frequency
     *
     * \retval Power amp setting
     */
    virtual uint8_t GetPaSelect( uint32_t channel )
    {
        return RF_PACONFIG_PASELECT_PABOOST;
    }

private:
    /*!
     * Radio registers definition
     */
    typedef struct
    {
        RadioModems_t Modem;
        uint8_t       Addr;
        uint8_t       Value;
    }RadioRegisters_t;

    /*!
     * Radio FSK modem parameters
     */
    typedef struct
    {
        int8_t   Power;
        uint32_t Fdev;
        uint32_t Bandwidth;
        uint32_t BandwidthAfc;
        uint32_t Datarate;
        uint16_t PreambleLen;
        bool     FixLen;
        uint8_t  PayloadLen;
        bool     CrcOn;
        bool     IqInverted;
        bool     RxContinuous;
        uint32_t TxTimeout;
    }RadioFskSettings_t;

    /*!
     * Radio FSK packet handler state
     */
    typedef struct
    {
        bool  SyncWordDetected;
        int8_t   RssiValue;
        int32_t  AfcValue;
        uint8_t  RxGain;
        uint16_t Size;
        uint16_t NbBytes;
        uint8_t  FifoThresh;
        uint8_t  ChunkSize;
    }RadioFskPacketHandler_t;

    /*!
     * Radio LoRa modem parameters
     */
    typedef struct
    {
        int8_t   Power;
        uint32_t Bandwidth;
        uint32_t Datarate;
        bool     LowDatarateOptimize;
        uint8_t  Coderate;
        uint16_t PreambleLen;
        bool     FixLen;
        uint8_t  PayloadLen;
        bool     CrcOn;
        bool     FreqHopOn;
        uint8_t  HopPeriod;
        bool     IqInverted;
        bool     RxContinuous;
        uint32_t TxTimeout;
    }RadioLoRaSettings_t;

    /*!
     * Radio LoRa packet handler state
     */
    typedef struct
    {
        int8_t SnrValue;
        int16_t RssiValue;
        uint8_t Size;
    }RadioLoRaPacketHandler_t;

    /*!
     * Radio Settings
     */
    typedef struct
    {
        uint32_t                 Channel;
        RadioFskSettings_t       Fsk;
        RadioFskPacketHandler_t  FskPacketHandler;
        RadioLoRaSettings_t      LoRa;
        RadioLoRaPacketHandler_t LoRaPacketHandler;
    }RadioSettings_t;

    /*!
     * \brief Resets the SX1272
     */
    void Reset( void );

    /*!
     * \brief Sets the SX1272 operating mode
     *
     * \param [IN] opMode New operating mode
     */
    void SetOpMode( uint8_t opMode );

    /*!
     * \brief Sets the SX1272 in transmission mode for the given time
     * \param [IN] timeout Transmission timeout [ms] [0: continuous, others timeout]
     */
    void SetTx( uint32_t timeout );

    /*!
     * \brief Writes the buffer contents to the SX1272 FIFO
     *
     * \param [IN] buffer Buffer containing data to be put on the FIFO.
     * \param [IN] size Number of bytes to be written to the FIFO
     */
    void WriteFifo( const uint8_t *buffer, uint8_t size );

    /*!
     * \brief Reads the contents of the SX1272 FIFO
     *
     * \param [OUT] buffer Buffer where to copy the FIFO read data.
     * \param [IN] size Number of bytes to be read from the FIFO
     */
    void ReadFifo( uint8_t *buffer, uint8_t size );

    static void OnRxTxTimeoutIrq( void const *arg );
    static void OnSyncTimeoutIrq( void const *arg );
    void OnTimeoutIrqBottom( void );
    void SX1272OnDioIrq( void );
    void SX1272OnDio3Irq( void );
    void SX1272OnDio1Irq( void );

    void SX1272IrqTxDone();
    void SX1272IrqFskRxDone(uint8_t irqFlags2);
    void SX1272IrqFskFifoLevel(void);
    void SX1272IrqFskFifoEmpty(void);
    void SX1272IrqFskSyncMatch(void);
    void SX1272IrqFsk(void);
    void SX1272IrqLoraRxDone(uint8_t irqFlags);
    void SX1272IrqLoraRxTimeout();
    void SX1272IrqLoraCadDone(uint8_t irqFlags);
    void SX1272IrqLoraFhss(void);
    void SX1272IrqLora(void);
    void SX1272Irq(void);

    static void RadioThread(void const *args);

    /*!
     * Interrupt thread
     */
    Thread thread;
    uint64_t Stack[2560/8];        // 2.5kB stack for the radio thread
				   //align stack fix for ARM mbed 5.5.* custom stack error

    /*!
     * Reset IO line
     */
    DigitalInOut reset;

    InterruptIn DIO3;
    InterruptIn DIO4;
    InterruptIn DIO0;
    InterruptIn DIO1;
    InterruptIn DIO2;


    SPI spi;
    /*!
     * SPI slave select
     */
    DigitalOut nss;

    /*!
     * Event callback handler
     */
    SxRadioEvents *RadioEvents;

    /*!
     * Reception buffer
     */
    uint8_t RxTxBuffer[RX_BUFFER_SIZE];

    /*!
     * Radio hardware and global parameters
     */
    RadioSettings_t settings;

    RtosTimer TxTimeoutTimer;
    RtosTimer RxTimeoutTimer;
    RtosTimer RxTimeoutSyncWord;

    bool RxTxTimeoutPending;
    bool SyncTimeoutPending;

    uint8_t opMode;
    uint32_t timeOnAir;
    
    static const RadioRegisters_t RadioRegsInit[];
};

#endif // __SXRADIO1272_H__
