#include "mbed.h"
#include "mtsas.h"

/******************************** NOTE *******************************/
/******** Uncomment the hardware platform you are working on. ********/
#define DRAGONFLY
//#define NUCLEO_F401RE
//#define FREESCALE_KL46Z
//#define FREESCALE_K64F
/*********************************************************************/

int main(){

#if defined(DRAGONFLY)
    /** Dragonfly
    * To configure the serial pins for the Dragonfly board, use:
    * RADIO_TX = pin PC_7, RADIO_RX = pin PC_6
    * RADIO_RTS = pin PB_10,RADIO_CTS = pin PB_12
    */
    MTSSerialFlowControl* io = new MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
#elif defined(NUCLEO_F401RE)
    /** STMicro Nucleo F401RE
    * The supported jumper configurations of the MTSAS do not line up with
    * the pin mapping of the Nucleo F401RE. Therefore, the MTSAS serial TX
    * pin (JP8 Pin 2) must be manually jumped to Serial1 RX (Shield pin D2)
    * and the MTSAS serial RX pin (JP9 Pin 2) pin must be manually jumped to
    * Serial1 TX (Shield pin D8).
    */
    MTSSerialFlowControl* io = new MTSSerialFlowControl(D8, D2, D3, D6);
#elif defined(FREESCALE_KL46Z)
    /** Freescale KL46Z
    * To configure the serial pins for the Freescale KL46Z board, use MTSAS jumper
    * configuration B.
    */
    MTSSerialFlowControl* io = new MTSSerialFlowControl(D2, D9, D3, D6);
#elif defined(FREESCALE_K64F)    
    /** Freescale K64F
    * To configure the serial pins for the Freescale K64F board, use MTSAS jumper
    * configuration A.
    */
    MTSSerialFlowControl* io = new MTSSerialFlowControl(D1, D0, D3, D6);
#endif
    
    //Sets the log level to INFO, higher log levels produce more log output.
    //Possible levels: NONE, FATAL, ERROR, WARNING, INFO, DEBUG, TRACE
    MTSLog::setLogLevel(MTSLog::INFO_LEVEL);
    
    //Modify to match your apn if you are using an HSPA radio with a SIM card
    const char APN[] = "";
    
    //Phone number to send to and receive from. Must be in the form "1xxxxxxxxxx"
    string PHONE_NUMBER = "";
    
    Cellular::Sms txtmsg;
    txtmsg.phoneNumber = PHONE_NUMBER;
    txtmsg.message = "Hello World! MTSAS is up and running!";
    
    //Sets the baud rate for communicating with the radio
    io->baud(115200); 
    
    //Creates a radio object
    Cellular* radio = CellularFactory::create(io);
    
    if (! radio) {
        logFatal("Failed to initialize radio");
        return 1;
    }
    
    Transport::setTransport(radio);
    
    //Set radio APN
    for (int i = 0; i < 10; i++) {
        if (i >= 10) {
            logError("Failed to set APN\n");
        }
        if (radio->setApn(APN) == MTS_SUCCESS) {
            logInfo("Successfully set APN\n");
            break;
        } else {
            wait(1);
        }
    }
    
    //Delete any previously received SMS messages
    for (int i = 0; i < 10; i++) {
        if (i >= 10) {
            logError("Failed to delete SMS messages\n");
        }
        if (radio->deleteAllReceivedSms() == MTS_SUCCESS) {
            logInfo("Deleted all SMS messages\n");
            break;
        } else {
            wait(1);
        }
    }
    
    // Send SMS message to phone
    for (int i = 1; i < 10; i++) {
        if(radio->sendSMS(txtmsg) == MTS_SUCCESS) {
            logInfo("Sent SMS successfully:<%s>\n", txtmsg.message.c_str());
            break;
        } else {
            logError("Failed to send SMS<%s>\n", txtmsg.message.c_str());
        }
    }
    
    //Checking for received SMS messages
    while (true) {
        logInfo("Checking for received messages");
        vector<Cellular::Sms> recv = radio->getReceivedSms();
        if(recv.size() > 0) {
            int size = recv.size();
            for (int i = 0; i < size; i++) {
                logInfo("Message %d: [%s] [%s] [%s]", i, recv[i].phoneNumber.c_str(), recv[i].timestamp.c_str(), recv[i].message.c_str());
            }
        }
        
        if(radio->deleteOnlyReceivedReadSms() != MTS_SUCCESS) {
            logError("Failed to delete received and read SMS messages");
        }
        wait(10);
    }
    
    return 0;
}