/** Dragonfly Cellular HTTPS Example
 * Configures the cellular radio, brings up the cellular link, and does HTTPS GET and POST requests.
 * To do HTTPS requests with a certain server, the root certificate used to validate that server's certificate must be installed. See ssl_certificates.h for information on how to get the proper root certificate.
 *
 * NOTE: This example changes the baud rate of the debug port to 115200 baud!
 */

#include "mbed.h"
#include "mtsas.h"
#include "ssl_certificates.h"

// This line controls the regulator's battery charger.
// BC_NCE = 0 enables the battery charger
// BC_NCE = 1 disables the battery charger
DigitalOut bc_nce(PB_2);

bool init_mtsas();
char* httpResToStr(HTTPResult res);

// The MTSSerialFlowControl object represents the physical serial link between the processor and the cellular radio.
mts::MTSSerialFlowControl* io;
// The Cellular object represents the cellular radio.
mts::Cellular* radio;

// An APN is required for GSM radios.
static const char apn[] = "";

bool radio_ok = false;

int main() {
    // Disable the battery charger unless a battery is attached.
    bc_nce = 1;
    
    // Change the baud rate of the debug port from the default 9600 to 115200.
    Serial debug(USBTX, USBRX);
    debug.baud(115200);
    
    //Sets the log level to INFO, higher log levels produce more log output.
    //Possible levels: NONE, FATAL, ERROR, WARNING, INFO, DEBUG, TRACE
    mts::MTSLog::setLogLevel(mts::MTSLog::INFO_LEVEL);
    
    logInfo("initializing cellular radio");
    radio_ok = init_mtsas();
    if (! radio_ok) {
        while (true) {
            logError("failed to initialize cellular radio");
            wait(1);
        }
    }
    
    logInfo("setting APN");
    if (radio->setApn(apn) != MTS_SUCCESS)
        logError("failed to set APN to \"%s\"", apn);
        
    logInfo("bringing up the link");
    if (! radio->connect()) {
        logError("failed to bring up the link");
    } else {
        
        // HTTPClient object used for HTTP requests.
        HTTPClient http;
        
        // Enable strict certificate validation.
        http.setPeerVerification(VERIFY_PEER);
        
        // Load certificates defined in ssl_certificates.h.
        // See comments in ssl_certificates.h for information on how to get and format root certificates.
        if (http.addRootCACertificate(ssl_certificates) != HTTP_OK)
            logError("loading SSL certificates failed");
        
        // HTTP GET example - httpbin.org
        {
            char http_rx_buf[1024];
            HTTPResult res;
            
            // IHTTPDataIn object - will contain data received from server.
            HTTPText http_rx(http_rx_buf, sizeof(http_rx_buf));
            
            // Make a HTTP GET request to http://httpbin.org/
            res = http.get("https://httpbin.org/get", &http_rx);
            if (res != HTTP_OK)
                logError("HTTPS GET failed [%d][%s]", res, httpResToStr(res));
            else
                logInfo("HTTPS GET succeeded [%d]\r\n%s", http.getHTTPResponseCode(), http_rx_buf);
        }
        
        // HTTP POST example - httpbin.org
        {
            char http_rx_buf[1024];
            HTTPResult res;
            
            char http_tx_buf[] = "{ \"name\": \"temp_1\", \"temperature\": 75 }";
            
            // IHTTPDataIn object - will contain data received from server.
            HTTPText http_rx(http_rx_buf, sizeof(http_rx_buf));
            
            // IHTTPDataOut object - contains data to be posted to server.
            // HTTPJson automatically adds the JSON content-type header to the request.
            HTTPJson http_tx(http_tx_buf, sizeof(http_tx_buf));
            
            // Make a HTTP POST request to http://httpbin.org/
            res = http.post("https://httpbin.org/post", http_tx, &http_rx);
            if (res != HTTP_OK)
                logError("HTTPS POST failed [%d][%s]", res, httpResToStr(res));
            else
                logInfo("HTTPS POST succeeded [%d]\r\n%s", http.getHTTPResponseCode(), http_rx_buf);
        }
    }
    
    logInfo("finished - bringing down link");
    radio->disconnect();
    
    return 0;
}

bool init_mtsas() {
    io = new mts::MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
    if (! io)
        return false;
        
    // radio default baud rate is 115200
    io->baud(115200);
    radio = mts::CellularFactory::create(io);
    if (! radio)
        return false;
        
    // Transport must be set properly before any TCPSocketConnection or UDPSocket objects are created
    Transport::setTransport(radio);
    
    return true;
}

char* httpResToStr(HTTPResult res) {
    switch(res) {
        case HTTP_PROCESSING:
            return "HTTP_PROCESSING";
        case HTTP_PARSE:
            return "HTTP_PARSE";
        case HTTP_DNS:
            return "HTTP_DNS";
        case HTTP_PRTCL:
            return "HTTP_PRTCL";
        case HTTP_NOTFOUND:
            return "HTTP_NOTFOUND";
        case HTTP_REFUSED:
            return "HTTP_REFUSED";
        case HTTP_ERROR:
            return "HTTP_ERROR";
        case HTTP_TIMEOUT:
            return "HTTP_TIMEOUT";
        case HTTP_CONN:
            return "HTTP_CONN";
        case HTTP_CLOSED:
            return "HTTP_CLOSED";
        case HTTP_REDIRECT:
            return "HTTP_REDIRECT";
        case HTTP_OK:
            return "HTTP_OK";
        default:
            return "HTTP Result unknown";
    }
}
