// ---------------------------------------------
// MultiTech Thermostat & Fan Demo (fan unit)
//
// This program runs on a MultiTech mDot with:
//   - 1 MultiTech UDK
//   - 1 Arduino base shield
//   - 1 Grove LED05291P 4-digit display
//   - 1 relay for fan control
// ---------------------------------------------

#include "mbed.h"
#include "mDot.h"
#include "MTSText.h"
#include "DigitDisplay.h"

mDot* dot;
Ticker ledTick;
Ticker periodicPollTick;

DigitalOut led(LED1);
DigitalOut fan(PA_1);             //D6 on Arduino base shield
DigitDisplay display(PB_1, PB_0); //A0 & A1 on Arduino base shield

// Configuration variables
static std::string config_network_name = "your_name_here";
static std::string config_network_pass = "your_key_here";
static uint8_t config_frequency_sub_band = 1;

static volatile bool timeToPollConduit = true;

//Function prototypes
void printError(mDot* dot, int32_t returnCode);
void printVersion();
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setPower(uint8_t power);
bool setAck(uint8_t retries);
bool joinNetwork();
bool send(const std::string text);
std::string receive();
void ledTock();
void periodicPollTock();


int main()
{
    //Start LED startup sequence
    ledTick.attach(&ledTock, 0.1);

    printf("\r\n\r\n");
    printf("=====================================\r\n");
    printf("MTS Thermostat/Fan Demo (Fan unit)\r\n");
    printf("=====================================\r\n");
    printVersion();

    // get the mDot handle
    dot = mDot::getInstance();

    // reset to default config so we know what state we're in
    dot->resetNetworkSession();
    dot->resetConfig();

    // set up the mDot with our network information
    setNetworkName(config_network_name);
    setNetworkPassphrase(config_network_pass);
    setFrequencySubBand(config_frequency_sub_band);
    setPower(7);    // Reduce latency for 868 units
    setAck(0);      // Disable ack for less latency

    while (!joinNetwork()) { wait(2); dot->resetNetworkSession();  }

    // Stop LED startup sequence & configure them for operation
    ledTick.detach();
    led = 1;

    // Configure timers
    periodicPollTick.attach(&periodicPollTock, 2);

    static bool fanState = false;
    uint16_t setPoint = 0;
    uint16_t temperature = 0;

    while (1) {
        if (timeToPollConduit) {
            std::string rxData;

            led = 1;
            display.setColon(false);

            // Send packet (Only send payload if no data pending because Node-RED queues new packet if payload not empty)
            send(dot->getDataPending() ? "" : " ");
            // Receive a packet
            rxData = receive();

            if (rxData.size() > 0) {
                printf("Rx packet data: %s\r\n", rxData.c_str());
                
                // Parse receive data
                std::vector<std::string> bytes;
                bytes = mts::Text::split(rxData, " ");
                temperature = atoi(bytes[0].c_str());
                setPoint = atoi(bytes[1].c_str());
                fanState = atoi(bytes[2].c_str());
                
                // Update 4 digit display
                display.write(0, temperature / 10);
                display.write(1, temperature % 10);
                display.write(2, setPoint / 10);
                display.write(3, setPoint % 10);

                fan = fanState;
            }
            else {
                printf("Nothing!\r\n");
            }

            timeToPollConduit = false;
            display.setColon(true);
            led = 0;
        }
    }
}

void ledTock() {
    led = !led;
}

void periodicPollTock() {
    timeToPollConduit = true;
}


void printVersion()
{
    printf("%s\r\n\r\n", dot->getId().c_str());
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t returnCode;
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((returnCode = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t returnCode;
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((returnCode = dot->setNetworkName(name)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t returnCode;
    printf("Setting passphrase to '%s'...\r\n", passphrase.c_str());
    if ((returnCode = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setPower(uint8_t power)
{
    int32_t returnCode;
    printf("Setting tx power to '%d'...\r\n", power);
    if ((returnCode = dot->setTxPower(power)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}


bool joinNetwork()
{
    int32_t returnCode;
    printf("\r\nJoining network...\r\n");
    if ((returnCode = dot->joinNetworkOnce()) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t returnCode;
    printf("Setting ack to '%d'...\r\n", retries);
    if ((returnCode = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool send(const std::string text)
{
    int32_t returnCode;
    uint32_t timeTillSend = dot->getNextTxMs();
    if (timeTillSend != 0) {
        printf("waiting %lu ms to send\r\n", timeTillSend);
        return false;
    }

    printf("Sending data...  ");
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((returnCode = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

std::string receive()
{
    printf("Receiving packet...  ");
    std::string text;
    std::vector<uint8_t> data(text.begin(), text.end());
    data.clear();

    if (dot->recv(data) == mDot::MDOT_OK) {
        if (data.size() > 0) {
            return std::string((const char*) &data[0], data.size());
        }
    }
    return "";
}

void printError(mDot* dot, int32_t returnCode)
{
    std::string error = mDot::getReturnCodeString(returnCode) + " - " + dot->getLastError();
    printf("%s\r\n", error.c_str());
}
