/**
 * @file    GpsParser.h
 * @brief   NEMA String to Packet Parser - NEMA strings to compact packet data
 * @author  Tim Barr
 * @version 1.0
 * @see     http://www.kh-gps.de/nmea.faq
 * @see     http://www.catb.org/gpsd/NMEA.html
 * @see     http://www.skytraq.com.tw/products/Venus638LPx_PB_v3.pdf
 *
 * Copyright (c) 2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#ifndef GPSPARSER_H
#define GPSPARSER_H

#include "mbed.h"
#include "rtos.h"
#include "MTSSerial.h"
#include "NCP5623B.h"
#include <string>
#include <vector>
#include <ctime>
#define START_THREAD 1

using namespace mts;

class GPSPARSER
{
public:

    /**
     * @typedef struct latitude
     * @brief latitude degrees, minutes, seconds structure
     * range of degrees : 0 to 90 signed (negative degrees is South)
     * range of minutes : 0 to 60 unsigned
     * Range of seconds : 0-9999 unsigned or 1/10000 of a minute.
     * Multiply by 6 and divide or modulus by 1000 to get seconds and fractional seconds
     */
    typedef struct{int8_t degrees; uint8_t minutes; uint16_t seconds;} latitude;
    
    /**
     * @typedef struct longitude
     * @brief longitude degrees, minutes, seconds structure negative degrees is West
     * range of degrees : 0 to 180 signed (negative degrees is West)
     * range of minutes : 0 to 60 unsigned
     * Range of seconds : 0-9999 unsigned or 1/10000 of a minute.
     * Multiply by 6 and divide or modulus by 1000 to get seconds and fractional seconds
     */
    typedef struct{int16_t degrees; uint8_t minutes; uint16_t seconds;} longitude;
    
    /**
     * @typdef satellite_prn
     * @brief satellite prn list, up to 12 valid
     */
    typedef uint8_t satellite_prn[12];

    /** Create the GPSPARSER object
     *  @param uart - an MTSSerial object
     *  @param led - an NCP5623B led controller object
     */ 
    GPSPARSER(MTSSerial *uart, NCP5623B* led = NULL);

    /** destroy the GPSPARSER object
     */ 
    ~GPSPARSER(void);
    
    /** GPS device detected
     *  @return boolean whether GPS device detected
     */
    bool gpsDetected(void);

    /** get longitude structure
     *  @return last valid NEMA string longitude data
     */
    longitude getLongitude(void);

    /** get latitude structure
     *  @return last valid NEMA string latitude data
     */
    latitude getLatitude(void);

    /** get timestamp structure
     *  @return last valid NEMA string time and date data
     *  uses tm struc from ctime standard library
     */
    struct tm getTimestamp(void);

    /** get GPS Lock status
     *  @return boolean of last valid NEMA string lock status data
     */
    bool getLockStatus(void);

    /** get GPS Fix status
     *  @return last valid NEMA Fix status data
     *  1 = no fix
     *  2 = 2D fix - latitude, longitude, time only valid
     *  3 = 3D fix - all data valid
     */
    uint8_t getFixStatus(void);

    /** get GPS Fix qualty
     *  @return last valid NEMA string Fix Quality data
     *  0 = invalid
     *  1 = GPS fix (SPS)
     *  2 = DGPS fix
     *  3 = PPS fix
     *  4 = Real Time Kinematic
     *  5 = Float RTK
     *  6 = estimated (dead reckoning) (2.3 feature)
     *  7 = Manual input mode
     *  8 = Simulation mode
     */
    uint8_t getFixQuality(void);

    /** get number of visible satellites
     *  @return last valid NEMA string number of satellites visible
     */
    uint8_t getNumSatellites(void);

    /** get calculated altitude
     *  @return last valid NEMA string altitude data in meters 
     */
    int16_t getAltitude(void);

private:

    std::string _nema_buff;
    bool _gps_detected;
    latitude _gps_latitude;
    longitude _gps_longitude;
    struct tm _timestamp;
    bool _gps_status;
    uint8_t _fix_status;
    uint8_t _fix_quality;
    uint8_t _num_satellites;
    satellite_prn _satellite_prn;
    int16_t _msl_altitude;
    
    MTSSerial *_gps_uart;
    Thread _getSentenceThread;
    NCP5623B* _led;
    int8_t _led_state;
    int8_t _led_state_out;
    Ticker _tick;
    bool _tick_running;
    Mutex _mutex;
    
    /** Start sentence parser thread
     *  
     */
    static void startSentenceThread (void const *p);

    /** Read NEMA sentences from specified serial port and call specific sentence parser
     */
    void readNemaSentence (void);

    /** Parse GGA NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseGGA(char *nema_buf);

    /** Parse GSA NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseGSA(char *nema_buf);

    /** Parse GSV NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseGSV(char *nema_buf);

    /** Parse RMC NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseRMC(char *nema_buf);

    /** Parse VTG NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseVTG(char *nema_buf);

    /** Parse GLL NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseGLL(char *nema_buff);

    /** Parse ZDA NEMA sentence
     *  @return status of parser attempt
     */
    uint8_t parseZDA(char *nema_buff);

    void blinker();

};
#endif
