#ifndef AXEDAWRAPPER_H
#define AXEDAWRAPPER_H

#include "HTTPClient.h"
#include "MbedJSONValue.h"

namespace mts {
    
class AxedaWrapper {
public:
    /** Constructor
    * Creates an AxedaWrapper object
    *
    * @param serial A made up serial number for your device.  It needs to be unique!
    *               A good format to use is <your initials>-<a random string of numbers>,
    *               e.g. "JTA-862653"
    */
    AxedaWrapper(const std::string& serial, const std::string& model = "mbed", const std::string& host = "dev6-connect.axeda.com", int port = 52689);
    
    /** Destructor
    * Deletes the AxedaWrapper object and returns all allocated memory
    */
    ~AxedaWrapper();
    
    /** push
    * Adds a key-value pair to the internal map.
    *
    * @param key The "name" of the value.
    * @param value The value to be stored.
    */
    void push(const std::string& key, const std::string& value);
    void push(const std::string& key, const char* value);
    void push(const std::string& key, int value);
    void push(const std::string& key, double value);
    void push(const std::string& key, bool value);
    
    /** toString
    * Returns a std::string containing the serialized json contents of the internal map
    * If the internal map is empty, "{}" will be returned
    */
    std::string toString();
    
    /** clear
    * Removes all key-value pairs from the internal map.
    * If the internal map is empty, no action is taken.
    */
    void clear();
    
    /** size
    * @returns the current size of the map
    */
    int size();
    
    /** send
    * Sends a key-value pair to the Axeda platform
    *
    * @param key The "name" of the value.
    * @param value The value to be sent.
    * @return true if the value was sent successfully, false otherwise
    */
    bool send(const std::string& key, const std::string& value);
    bool send(const std::string& key, const char* value);
    bool send(const std::string& key, int value);
    bool send(const std::string& key, double value);
    bool send(const std::string& key, bool value);
    
    /** sendAll
    * Sends all the key-value pairs in the internal map to the Axeda platform
    *
    * @param clear If true, clear the internal map if the send is successful,
    *              if false, don't clear the internal map
    * @return true if successful, false otherwise
    */
    bool sendAll(bool clean = true);
    
private:
    bool sendBase(const std::string& data);
    
    HTTPClient* _http_client;
    std::string _url;
    MbedJSONValue* _data;
    int _timeout;
};

}

#endif