/* This is an example program which reads in the 3-axis acceleration, pressure, and temperature data from
 * a FRDM-FXS-MULTI sensor board. It then uses an MTSAS SocketModem Shield board to send the read data over
 * a cellular connection to the 2lemetry cloud using an MQTT client protocol.
 */

#include "mbed.h"
#include "mtsas.h"
#include "PubSubClient.h"
#include "FXLS8471Q.h"
#include "MPL3115A2.h"

/* PLEASE READ THIS!
 * The following fields must be populated in order to properly send data to the "Default ThingFabric Project" in your 2lemetry account using the MQTT client
 * You must have a hacker (or higher) account at http://app.thingfabric.com
 * After you register and login, follow the steps below to set up your client
 * Click on "Default ThingFabric Project"
 *      Set _2LEMETRY_DOMAIN to the string after "Project" at the top of the page
 * Click on "Credentials"
 * Click on "Default ThingFabric Credential"
 *      Set _2LEMETRY_USER_ID to the value in the "Key (Username)" field
 *      Set _2LEMETRY_TOKEN to the value in the "MD5 Secret" field
 * If you are running this code on multiple devices, you will want to make the _2LEMETRY_DEVICE_ID field unique for each device
 * Set the _APN field to the APN provided with your sim card
 * Build, flash, and run
 *      This code sends a random integer value approximately every 30 seconds
 * Click on "Default ThingFabric Project"
 * Click on "Analytics"
 * You should be able to see your test data (page needs to be refreshed periodically, it doesn't automatically refresh)
 */

char _2LEMETRY_USERID[] = "";
char _2LEMETRY_TOKEN[] = "";
char _2LEMETRY_DOMAIN[] = "";
char _2LEMETRY_STUFF[] = "mbed";
char _2LEMETRY_DEVICE_ID[] = "nucleo-0001";

char _APN[] = "";

char _host[] = "q.mq.tt";
int _port = 1883;

#define MPL3115A2_I2C_ADDRESS (0x60<<1)

#define DATA_INTERVAL 30

void callback(char* topic, char* payload, unsigned int len) {
    logInfo("topic: [%s]\r\npayload: [%s]", topic, payload);
}

int main() {
    MTSLog::setLogLevel(MTSLog::TRACE_LEVEL);
    
    /** STMicro Nucelo F401RE
    * The supported jumper configurations of the MTSAS do not line up with
    * the pin mapping of the Nucleo F401RE. Therefore, the MTSAS serial TX
    * pin (JP8 Pin 2) must be manually jumped to Serial1 RX (Shield pin D2)
    * and the MTSAS serial RX pin (JP9 Pin 2) pin must be manually jumped to
    * Serial1 TX (Shield pin D8).
    * Uncomment the following line to use the STMicro Nuceleo F401RE
    */
    MTSSerialFlowControl io(D8, D2, D3, D6);
    
    /** Freescale KL46Z
    * To configure the serial pins for the Freescale KL46Z board, use MTSAS jumper
    * configuration B. Uncomment the following line to use the Freescale KL46Z board
    */
    //MTSSerialFlowControl io(D2, D9, D3, D6);
    
    /** Freescale K64F
    * To configure the serial pins for the Freescale K64F board, use MTSAS jumper
    * configuration A. Uncomment the following line to use the Freescale K64F board
    */
    //MTSSerialFlowControl io(D1, D0, D3, D6);

    //Set baudrate between the board and the radio to 115200
    io.baud(115200);
    
    Cellular* radio = CellularFactory::create(&io);
    if (! radio) {
        logFatal("failed to create Cellular object - exiting");
        return 1;
    }
    
    Transport::setTransport(radio);
    
    while (radio->setApn(_APN) != MTS_SUCCESS) {
        logError("failed to set APN [%s]", _APN);
        wait(2);
    }
    
    while (! radio->connect()) {
        logError("failed to bring up PPP link");
        wait(2);
    }
    
    PubSubClient mqtt(_host, _port, callback);
    
    char topicStr[128];
    char buf[128];
    snprintf(topicStr, sizeof(topicStr), "%s/%s/%s", _2LEMETRY_DOMAIN, _2LEMETRY_STUFF, _2LEMETRY_DEVICE_ID);
    
    FXLS8471Q acc(D11, D12, D13, D10);
    MPL3115A2 alt(D14, D15, MPL3115A2_I2C_ADDRESS, D4, D3);
    alt.Barometric_Mode(); 
    
    float acc_data[3];
    
    while (true) {
        if (! mqtt.connect(_2LEMETRY_DEVICE_ID, _2LEMETRY_USERID, _2LEMETRY_TOKEN)) {
            logError("failed to connect to 2lemetry server");
            continue;
        }
        
        acc.ReadXYZ(acc_data);
        
        snprintf(buf, sizeof(buf), "{\"x\":%f,\"y\":%f,\"z\":%f,\"pressure\":%f,\"temperature\":%f}", acc_data[0],acc_data[1],acc_data[2], alt.getPressure(), alt.getTemperature());
        logInfo("publishing: [%s]", buf);
        if (! mqtt.publish(topicStr, buf)) {
            logError("failed to publish: [%s]", buf);
        }
        wait(1);
        mqtt.loop();
        mqtt.disconnect();
        wait(DATA_INTERVAL);
    }
    
}