/* Copyright (c) 2016 MtM Technology Corporation, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "AK09970.h"


AK09970::AK09970(I2C &i2c, PinName int1) : _i2c(i2c), _int1(int1) {

}

void AK09970::ConfigDevice(uint8_t mode) {
    char val[8];

    /* Soft reset */
    RegWrite(0x30, 0x01);

    /* Disable all interrupts  */
    val[0] = 0x00;
    val[1] = 0x00;
    RegWrite(0x20, val, 2);

    /* Mode */
    _sensitivity = (mode & SMR_WideRange) ? 3.1f : 1.1f;
    RegWrite(0x21, mode);
}

void AK09970::GetDeviceID(uint8_t *id) {
    char buf[2];
    RegRead(0x00, buf, 2);
    *id = buf[1];  
}

void AK09970::GetData(Data *data) {
    char buf[8];

    /* Read data */
    RegRead(0x17, buf, 8);
    
    /* Format data */
    uint16_t status = (((uint16_t)buf[0]) << 8) | buf[1];
    int16_t x_adc = (((int16_t)buf[6]) << 8) | buf[7];
    int16_t y_adc = (((int16_t)buf[4]) << 8) | buf[5];
    int16_t z_adc = (((int16_t)buf[2]) << 8) | buf[3];
    
    /* Convert data */
    data->x = ConvertAdcToMagnetic(x_adc);
    data->y = ConvertAdcToMagnetic(y_adc);
    data->z = ConvertAdcToMagnetic(z_adc);
}

float AK09970::ConvertAdcToMagnetic(int16_t adc) {
    float magnetic_flux_density = (float)adc * _sensitivity;
    return magnetic_flux_density;
}

void AK09970::RegWrite(char reg, char val) {
    char data[2];
    data[0] = reg;
    data[1] = val;
    _i2c.write(AK09970_SLAVE_ADDR, data, 2, 0);
}

void AK09970::RegWrite(char reg, char *val, int len) {
    char data[9];
    data[0] = reg;
    memcpy((data+1), val, len);
    _i2c.write(AK09970_SLAVE_ADDR, data, (len+1), 0);
}

void AK09970::RegRead(char reg, char *val, int len) {
    _i2c.write(AK09970_SLAVE_ADDR, &reg, 1, 0);
    _i2c.read (AK09970_SLAVE_ADDR, val, len);
}

void AK09970::RegReadModifyWrite(char reg, char clr_mask, char set_mask) {
    char val;
    RegRead (reg, &val, 1);             // Read
    val = (val & ~clr_mask) | set_mask; // Modify
    RegWrite(reg, &val, 1);             // Write
}
