/* Copyright (c) 2016 MtM Technology Corporation, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "AK09912.h"


AK09912::AK09912(I2C &i2c, PinName int1) : _i2c(i2c), _int1(int1) {

}

void AK09912::ConfigDevice() {
    /* Soft reset */
    RegWrite(0x32, 0x01);

    /* Get sensitivity adjustment values */
    RegWrite(0x31, 0x1F);   // MODE(Fuse ROM ACCESS)
    RegRead(0x60, (char *)_asa, 3); // ASAX, ASAY, ASAZ
    RegWrite(0x31, 0x00);   // MODE(Power down)

    /* Enable both temperature measurement and noise suppression filter  */
    RegWrite(0x30, 0xE0);   // TEM(Enable), NSF(High)

    /* Mode */
    RegWrite(0x31, 0x02);   // MODE(ContMode1)
}

void AK09912::GetDeviceID(uint8_t *id) {
    RegRead(0x01, (char *)id, 1);
}

void AK09912::GetData(Data *data){
    char status;
    char buf[10];

    /* Read data */
    RegRead(0x10, &status, 1);
    RegRead(0x11, buf, 7);
    RegRead(0x18, &status, 1);  // Dummy read

    /* Format data */
    int16_t x_adc = (((int16_t)buf[1]) << 8) | buf[0];
    int16_t y_adc = (((int16_t)buf[3]) << 8) | buf[2];
    int16_t z_adc = (((int16_t)buf[5]) << 8) | buf[4];
    uint8_t t_adc = buf[6];

    /* Convert data */
    data->x = ConvertAdcToMagnetic(x_adc, _asa[0]);
    data->y = ConvertAdcToMagnetic(y_adc, _asa[1]);
    data->z = ConvertAdcToMagnetic(z_adc, _asa[2]);
    data->t = ConvertAdcToTemperature(t_adc);
}

float AK09912::ConvertAdcToMagnetic(int16_t adc, uint8_t asa) {
    float magnetic_flux_density = (float)adc * 0.15f;   // Original
    return magnetic_flux_density * (((float)asa - 128) * 0.5f / 128 + 1);   // Adjustment
}

float AK09912::ConvertAdcToTemperature(uint8_t adc) {
    return (120.0f - (float)adc) / 1.6f + 35.0f;
}
    
void AK09912::RegWrite(char reg, char val) {
    char data[2];
    data[0] = reg;
    data[1] = val;
    _i2c.write(AK09912_SLAVE_ADDR, data, 2, 0);
}

void AK09912::RegRead(char reg, char *val, int len) {
    _i2c.write(AK09912_SLAVE_ADDR, &reg, 1, 0);
    _i2c.read (AK09912_SLAVE_ADDR, val, len);
}

void AK09912::RegReadModifyWrite(char reg, char clr_mask, char set_mask) {
    char val;
    RegRead (reg, &val, 1);             // Read
    val = (val & ~clr_mask) | set_mask; // Modify
    RegWrite(reg, val);                 // Write
}
