/* Copyright (c) 2016 MtM Technology Corporation, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "mbed.h"
#include "AK9750.h"
#include "AK09912.h"
#include "AK09970.h"
#include "config.h"

I2C i2c(p3, p2);

#ifdef NRF52
Serial pc(p18, p16);
DigitalOut reset_sensor(p24, 1);   // Reset pin for AK09970 (Low active)
AK9750 ak9750(i2c, p5);
#else
Serial pc(p15, p16);
DigitalOut reset_sensor(p4, 1);   // Reset pin for AK09970 (Low active)
AK9750 ak9750(i2c, p0);
#endif

AK09912 ak09912(i2c);
AK09970 ak09970(i2c);

volatile bool isIntEvent = false;
static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);
int32_t count = 0;

//==============================================================================
static void IntEventCallback() {
    isIntEvent = true;
}
void readSensorCallback() {
#if (CONFIG_RUN_MODE == RUN_ALL_SENSOR_COM)    
    AK9750::Data ak9750_data;
    AK09912::Data ak09912_data;
    AK09970::Data ak09970_data;
    int32_t triggered_area;
    ak9750.GetData(&ak9750_data);
    triggered_area = ak9750.GetTriggeredAreaNum(&ak9750_data);
    pc.printf("AK9750(%5.1fpA, %5.1fpA, %5.1fpA, %5.1fpA, %2.1f'C, Area%d)\n", 
    ak9750_data.ir1, ak9750_data.ir2, ak9750_data.ir3, ak9750_data.ir4, ak9750_data.tmp, triggered_area);

    ak09912.GetData(&ak09912_data);
    pc.printf("AK09912(%5.1fuT/LSB, %5.1fuT/LSB, %5.1fuT/LSB, %2.1f'C)\n",
    ak09912_data.x, ak09912_data.y, ak09912_data.z, ak09912_data.t);

    ak09970.GetData(&ak09970_data);
    pc.printf("AK09970(%6.1fuT/LSB, %6.1fuT/LSB, %6.1fuT/LSB)\n",
    ak09970_data.x, ak09970_data.y, ak09970_data.z);

    pc.printf("\n");
#elif (CONFIG_RUN_MODE == RUN_IR_AK9750_APP)
    AK9750::Data ak9750_data;
    ak9750.GetData(&ak9750_data);
    pc.printf(" 3:%d,%d,%d,%d,%d,\r", count, (int32_t)ak9750_data.ir1, (int32_t)ak9750_data.ir2, (int32_t)ak9750_data.ir3, (int32_t)ak9750_data.ir4);
    count++;
     
#elif (CONFIG_RUN_MODE == RUN_COMPASS_AK09912_APP)
    AK09912::Data ak09912_data;
    ak09912.GetData(&ak09912_data);
    pc.printf("C:%d,%d,%d,%d\r", count, (int32_t)ak09912_data.x, (int32_t)ak09912_data.y, (int32_t)ak09912_data.z);
    count++;
   
#elif (CONFIG_RUN_MODE == RUN_MAGNETIC_AK09970_APP)
    AK09970::Data ak09970_data;
    ak09970.GetData(&ak09970_data);
    pc.printf("C:%d,%d,%d,%d\r", count, (int32_t)ak09970_data.x, (int32_t)ak09970_data.y, (int32_t)ak09970_data.z);
    count++;
#endif  
}
//==============================================================================
int main() {
    /* Disable the hardware flow control of Serial, then show the mbed version */
    pc.set_flow_control(SerialBase::Disabled);
    pc.baud(CONFIG_BAUDRATE);
    pc.printf("\n");
    pc.printf("mbed version(%d.%d.%d)\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
    pc.printf("\n");

    /* Reset sensor */
    reset_sensor = 1;
    wait_ms(10);
    reset_sensor = 0;
    wait_ms(10);
    reset_sensor = 1;
    wait_ms(10);

    /* Config device and check device ID */
    uint8_t id;

    ak9750.ConfigDevice();
    ak9750.GetDeviceID(&id);
    pc.printf("AK9750_DEVICE_ID(0x%02X)\n", id);
    if(id != ak9750.DEVICE_ID){
        pc.printf("Read Device ID fail !\n");
        while(1);
    }
    
    ak09912.ConfigDevice();
    ak09912.GetDeviceID(&id);
    pc.printf("AK09912_DEVICE_ID(0x%02X)\n", id);
    if(id != ak09912.DEVICE_ID){
        pc.printf("Read Device ID fail !\n");
        while(1);
    }

    ak09970.ConfigDevice();
    ak09970.GetDeviceID(&id);
    pc.printf("AK09970_DEVICE_ID(0x%02X)\n", id);
    if(id != ak09970.DEVICE_ID){
        pc.printf("Read Device ID fail !\n");
        while(1);
    }


    /* Main loop */
#if (CONFIG_RUN_MODE == RUN_ALL_SENSOR_COM)
    eventQueue.call_every(200, readSensorCallback);
    eventQueue.dispatch_forever();
    
#elif (CONFIG_RUN_MODE == RUN_IR_AK9750_APP)
    eventQueue.call_every(200, readASensorCallback);
    eventQueue.dispatch_forever();
    
#elif (CONFIG_RUN_MODE == RUN_COMPASS_AK09912_APP)
    eventQueue.call_every(200, readASensorCallback);
    eventQueue.dispatch_forever();
    
#elif (CONFIG_RUN_MODE == RUN_MAGNETIC_AK09970_APP)
    eventQueue.call_every(200, readASensorCallback);
    eventQueue.dispatch_forever();
    
#elif (CONFIG_RUN_MODE == RUN_IR_AK9750_INT_COM)
    // Enable interrupt sources and set callback function
    ak9750.SetIntEvent(AK9750::INT_DR, &IntEventCallback);
    
    while (1) {
        /* If any interrupt occurred */
        if (isIntEvent) {
            /* Read interrupt status */
            uint8_t status = ak9750.GetIntStatus();
            if (status & AK9750::INT_DR) {
                pc.printf("INT_DR\n");
                /* Read data */
                AK9750::Data data;
                ak9750.GetData(&data);
                pc.printf("AK9750(%5.1fpA, %5.1fpA, %5.1fpA, %5.1fpA)\n", data.ir1, data.ir2, data.ir3, data.ir4);
            }
            if (status & AK9750::INT_IR24L) pc.printf("INT_IR24L\n");
            if (status & AK9750::INT_IR24H) pc.printf("INT_IR24H\n");
            if (status & AK9750::INT_IR13L) pc.printf("INT_IR13L\n");
            if (status & AK9750::INT_IR13H) pc.printf("INT_IR13H\n");
            isIntEvent = false;
        }
    }
#else
    #errror "Config run mode ...FAIL"
#endif
}
