/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "BMA250E.h"
#include "BMG160.h"


/* UART printf */
#ifdef NRF52
Serial pc(p20, p24);
#else
Serial pc(p5, p4);
#endif


/* LED blink */
Ticker ledBlinkTicker;
DigitalOut ledR(p16, 1);
DigitalOut ledG(p15, 1);
DigitalOut ledB(p6 , 1);

/* Sensor */
uint8_t acceRange = 0x0C;   // 2G(0x03), 4G(0x05), 8G(0x08), 16G(0x0C) 
uint8_t gyroRange = 0x00;   // 2000deg/s(0x00), 1000deg/s(0x01), 500deg/s(0x02), 250deg/s(0x03), 125deg/s(0x04)
int16_t acceXYZ[3];
int16_t gyroXYZ[3];
uint8_t accePayload[7];
uint8_t gyroPayload[7];

/* UART printf */
#ifdef NRF52
BMA250E acclerameter(p14, p28, NC, NC, acceRange, 0x0D);
BMG160 gyro(p14, p28, NC, NC, gyroRange, 0x00);
#else
BMA250E acclerameter(p14, p13, NC, NC, acceRange, 0x0D);
BMG160 gyro(p14, p13, NC, NC, gyroRange, 0x00);
#endif

static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);

/* UUID, Device name */
uint16_t sensServUUID = /*0xA000*/0x1811;
uint16_t acceCharUUID = /*0xA001*/0x2A56;
uint16_t gyroCharUUID = /*0xA002*/0x2A57;
static const char     DEVICE_NAME[] = "Mt5MtSense03";
static const uint16_t uuid16_list[] = { /*0xA000*/0x1811 };

/* Setup custom characteristics */
GattCharacteristic  acceChar(   acceCharUUID, accePayload,
                                sizeof(accePayload), sizeof(accePayload),
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | 
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

GattCharacteristic  gyroChar(   gyroCharUUID, gyroPayload,
                                sizeof(gyroPayload), sizeof(gyroPayload),
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | 
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

/* Setup custom service */
GattCharacteristic *characteristics[] = {&acceChar, &gyroChar};
GattService sensServ(sensServUUID, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));



void ledBlinkCallback(void)
{
    ledR = !ledR;
    ledG = !ledG;
    ledB = !ledB;
}

void updateSensorCallback(void)
{
    /* Get sensor data */
    acclerameter.ReadXYZ(acceXYZ);
    gyro.ReadXYZ(gyroXYZ);
//    pc.printf("aXYZ(%6d,%6d,%6d), gXYZ(%6d,%6d,%6d)\r\n", acceXYZ[0], acceXYZ[1], acceXYZ[2], gyroXYZ[0], gyroXYZ[1], gyroXYZ[2]);
    BLE &ble = BLE::Instance();
    if (ble.getGapState().connected) {
        /* Write data to client */
        accePayload[0] = acceRange;
        accePayload[1] = (uint8_t)(acceXYZ[0] >> 8);
        accePayload[2] = (uint8_t)(acceXYZ[0] >> 0);
        accePayload[3] = (uint8_t)(acceXYZ[1] >> 8);
        accePayload[4] = (uint8_t)(acceXYZ[1] >> 0);
        accePayload[5] = (uint8_t)(acceXYZ[2] >> 8);
        accePayload[6] = (uint8_t)(acceXYZ[2] >> 0);
        ble.gattServer().write(acceChar.getValueHandle(), accePayload, sizeof(accePayload));
        gyroPayload[0] = gyroRange;
        gyroPayload[1] = (uint8_t)(gyroXYZ[0] >> 8);
        gyroPayload[2] = (uint8_t)(gyroXYZ[0] >> 0);
        gyroPayload[3] = (uint8_t)(gyroXYZ[1] >> 8);
        gyroPayload[4] = (uint8_t)(gyroXYZ[1] >> 0);
        gyroPayload[5] = (uint8_t)(gyroXYZ[2] >> 8);
        gyroPayload[6] = (uint8_t)(gyroXYZ[2] >> 0);
        ble.gattServer().write(gyroChar.getValueHandle(), gyroPayload, sizeof(gyroPayload));
    }
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising(); // restart advertising
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble          = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    ble.addService(sensServ);

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_TAG);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(500); /* 500ms */
    ble.gap().startAdvertising();
}
void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}
int main(void)
{
    pc.set_flow_control(SerialBase::Disabled);
    pc.baud(115200);
    pc.printf("~ Hell World ~\n");

    /* LED blink timer */
    eventQueue.call_every(1000, ledBlinkCallback);

    /* Update Sensor timer */
    eventQueue.call_every(200, updateSensorCallback);

    /* Init BLE */
    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    eventQueue.dispatch_forever();

    return 0;
}
