/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "CCS811.h"
#include "AirQualityService.h"

//initial function pin
#ifdef NRF52
Serial pc(p20, p24);
#else
Serial pc(p5, p4);
#endif

I2C i2c(p3, p2);
DigitalOut ledR(p16, 1);
CCS811 ccs811(i2c, pc);

static AirQualityService *AQSPtr;
static uint16_t eco2, tvoc;

const static char     DEVICE_NAME[] = "Mt5MtSense02";
static const uint16_t uuid16_list[] = {GattService::UUID_BATTERY_SERVICE};


static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
}

void updateSensorValue() {
    
    ledR = 0;
    
    uint16_t revertECO2 = (eco2>>8) | (eco2<<8);
    uint16_t revertTVOC = (tvoc>>8) | (tvoc<<8);
    
    AQSPtr->updateCO2(revertECO2);
    AQSPtr->updateTVOC(revertTVOC);
    
}

void blinkCallback(void)
{
    ledR = !ledR; /* Do blinky on LED1 while we're waiting for BLE events */
}

void CCS811Callback(void) {
    
    ccs811.readData(&eco2, &tvoc);
    
    BLE &ble = BLE::Instance();
    if (ble.gap().getState().connected) {
        eventQueue.call(updateSensorValue);
    }
    
    pc.printf("eco2:%8d, tvoc:%8d\r\n", eco2, tvoc);
    
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;


    uint8_t co2CID[16] = {
            0x00,  'M',  'T', 'M',
            'C' ,  'O',  '2', 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF
    };
    
    uint8_t tvocCID[16] = {
            0x00,  'M',  'T', 'M' ,
            'T' ,  'V',  'O', 'C' ,
            0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF
    };
    
    UUID CO2CID (co2CID);
    UUID TVOCCID(tvocCID);
    
    AQSPtr = new AirQualityService(ble, CO2CID, TVOCCID);


    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
//    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *) uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *) DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(100); /* 1000ms */
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

int main()
{
    pc.set_flow_control(SerialBase::Disabled);
    pc.baud(115200);
    pc.printf("\r\n");
    pc.printf("Welcome MTM Node !\r\n");
    
    eventQueue.call_every(500, blinkCallback);

    ccs811.init();
//    ccs811.softRest();
    eventQueue.call_every(2000, CCS811Callback);

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    eventQueue.dispatch_forever();

    return 0;
}
