/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "ble/services/EnvironmentalService.h"
#include "MtSense01.h"

/* UART printf */
#ifdef NRF52
Serial pc(p20, p24);
#else
Serial pc(p5, p4);
#endif

I2C i2c(p3, p2);
DigitalOut led1(p16, 1);
MtSense01 mtsense1(i2c, pc);

const static char     DEVICE_NAME[] = "Mt5MtSense01";
static const uint16_t uuid16_list[] = {0x1822, 0x181A};

static EnvironmentalService *environmentalServicePtr;

static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
}

void initMtSense01() {
    mtsense1.Initial();
}

void readMtSense01() {
    int pressPa;
    float tempC, humidity;
    
    mtsense1.readData(&tempC, &pressPa, &humidity);
    BLE &ble = BLE::Instance();
    if (ble.gap().getState().connected) {
        environmentalServicePtr->updateHumidity(humidity);
        environmentalServicePtr->updatePressure(pressPa);
        environmentalServicePtr->updateTemperature(tempC);
    }
    
//    pc.printf("tempC: %8f, pressPa: %8d, humidity: %8f \r\n", tempC, pressPa, humidity);
        
}

void blinkCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    BLE &ble = BLE::Instance();
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service */
    environmentalServicePtr = new EnvironmentalService(ble);

    /* Setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *) uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *) DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(100); /* 100ms */
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

int main()
{
    pc.set_flow_control(SerialBase::Disabled);
    pc.baud(115200);
    pc.printf("\r\n");
    pc.printf("Welcome MTM Node !\r\n");
    
    
    eventQueue.call_every(500, blinkCallback);
    
    initMtSense01();
    eventQueue.call_every(1000, readMtSense01); // 1000 ms repeat

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    eventQueue.dispatch_forever();

    return 0;
}
