/* Copyright (c) 2016 MtM Technology Corporation, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "BMA250E.h"

/******************************* 
 * Public methods
 *******************************/
BMA250E::BMA250E(PinName i2c_sda, PinName i2c_scl, PinName interrupt1_pinname, PinName interrupt2_pinname,uint8_t range, uint8_t bandwith)
    : i2c_(i2c_sda, i2c_scl), 
      interrupt1_pinname_(interrupt1_pinname),
      interrupt1_(interrupt1_pinname),
      interrupt2_pinname_(interrupt2_pinname),  
      interrupt2_(interrupt2_pinname),
      range_(range), bandwith_(bandwith) {
          
    /* Basic */
    RegWrite(0x14, 0xB6);        // softreset
    RegWrite(0x0F, range_);      // range
    RegWrite(0x10, bandwith_);   // bandwidth
    
    /* Interrupt */
    RegWrite(0x16, 0x00);   // Disable all interrupts
    RegWrite(0x17, 0x00);   // 
    RegWrite(0x20, 0x00);   // int1_od(PP), int1_lvl(Low active), int2_od(PP), int2a_lvl(Low active)
    RegWrite(0x21, 0x80);   // reset_int, latch_int(non_latched)
}

void BMA250E::ReadXYZ(int16_t *xyz) {
    char val[6];

    /* Read raw data */
    RegRead(0x02, val, sizeof(val));
    xyz[0] = ((int16_t)val[1] << 8) | (val[0] & 0xC0);
    xyz[1] = ((int16_t)val[3] << 8) | (val[2] & 0xC0);
    xyz[2] = ((int16_t)val[5] << 8) | (val[4] & 0xC0);

    /* Align right */
    xyz[0] >>= 6;
    xyz[1] >>= 6;
    xyz[2] >>= 6;
    
}

void BMA250E::NewData(void(*fptr)(void)) {
    // TODO
}
void BMA250E::AnyMotion(void(*fptr)(void)) {
    // TODO
}
void BMA250E::TapSening(void(*fptr)(void), bool double_tap) {
    RegWrite(0x2A, 0x04);   // tap_quiet(30ms), tap_shock(50ms), tap_dur(250ms)
    RegWrite(0x2B, 0x0A);   // tap_samp(2samples), tap_th(0x0A)

    if (interrupt1_pinname_ != NC) {
        /* Interrupt 1 */
        if (double_tap) {
            /* Double tap */
            RegReadModifyWrite(0x19, 0x30, 0x10);   // int1_d_tap
            RegReadModifyWrite(0x16, 0x30, 0x10);   // d_tap_en
        } else {
            /* Single tap */
            RegReadModifyWrite(0x19, 0x30, 0x20);   // int1_s_tap
            RegReadModifyWrite(0x16, 0x30, 0x20);   // s_tap_en
        }
        interrupt1_.mode(PullUp);
        interrupt1_.fall(fptr);
    }
    if (interrupt2_pinname_ != NC) {
        /* Interrupt 2 */
        if (double_tap) {
            /* Double tap */
            RegReadModifyWrite(0x1B, 0x30, 0x10);   // int2_d_tap
            RegReadModifyWrite(0x16, 0x30, 0x10);   // d_tap_en
        } else {
            /* Single tap */
            RegReadModifyWrite(0x1B, 0x30, 0x20);   // int2_s_tap
            RegReadModifyWrite(0x16, 0x30, 0x20);   // s_tap_en
        }
        interrupt2_.mode(PullUp);
        interrupt2_.fall(fptr);
    }
}
void BMA250E::OrientationRecognition(void(*fptr)(void)) {
    // TODO
}
void BMA250E::FlatDetection(void(*fptr)(void)) {
    // TODO
}
void BMA250E::LowHighGDetection(void(*fptr)(void), bool high_g) {
    // TODO
}
void BMA250E::ShakeDetection(void(*fptr)(void)) {
    RegWrite(0x28, 0x64);   // slope_th(100)

    if (interrupt1_pinname_ != NC) {
        /* Interrupt 1 */
        RegReadModifyWrite(0x19, 0x04, 0x04);   // int1_slope
        RegReadModifyWrite(0x16, 0x07, 0x07);   // slope_en_z/y/x
        interrupt1_.mode(PullUp);
        interrupt1_.fall(fptr);
    }
    if (interrupt2_pinname_ != NC) {
        /* Interrupt 2 */
        RegReadModifyWrite(0x1B, 0x04, 0x04);   // int2_slope
        RegReadModifyWrite(0x16, 0x07, 0x07);   // slope_en_z/y/x
        interrupt2_.mode(PullUp);
        interrupt2_.fall(fptr);
    }
}

void BMA250E::EnterStandbyMode(void)
{
    RegReadModifyWrite(0x12, 0x40, 0x40);   // lowpower_mode(1)
    RegReadModifyWrite(0x11, 0x80, 0x80);   // suspend(1)
}

void BMA250E::LeaveStandbyMode(void)
{
    RegReadModifyWrite(0x12, 0x40, 0x40);   // lowpower_mode(1)
    RegReadModifyWrite(0x11, 0x80, 0x00);   // suspend(0)
}

/******************************* 
 * Private methods
 *******************************/
void BMA250E::RegWrite(char reg, char val) {
    char data[2];
    data[0] = reg;
    data[1] = val;
    i2c_.write(BMA250E_SLAVE_ADDR, data, 2, 0);
}

void BMA250E::RegRead(char reg, char *val, int len) {
    i2c_.write(BMA250E_SLAVE_ADDR, &reg, 1, 0);
    i2c_.read (BMA250E_SLAVE_ADDR, val, len);
}

void BMA250E::RegReadModifyWrite(char reg, char clr_mask, char set_mask) {
    char val;
    RegRead (reg, &val, 1);             // Read
    val = (val & ~clr_mask) | set_mask; // Modify
    RegWrite(reg, val);                 // Write
}