#include "mbed.h"
#include "WS2811.h"
#include "CubeUpdateParameters.h"

#pragma once

class LedCube
{
    public:
        LedCube();
        ~LedCube();
        
        /*Sets the initial size and position of the lighted cube*/ 
        void Init(int x, int y, int z);
        
        /* Returns the index of the LED given the cartesian 
         * coordinates of the LED on a given panel. The origin 
         * is the led at the bottom left of panel 1 when using 
         * a three panel cube.
             ________
            /       /|
           /   3   / |
          /_______/ 2|
          |       |  | 
          |   1   | /
          |       |
          --------      
          
          Z    Y
          |   /
          |  /
          | /
          |/
           -------X   */
        int getLedIndex(int panel, int x, int y);
        
        /* Lights up (if on) or turns off (if !on) the LEDs on the LED cube 
         * corresponding to the location of the square. All panels will show 
         * the cube, with brightness depending on the distance from the 
         * square to the panel.*/
        void updateLEDs(bool on, int size, int x, int y, int z);
        void updateLEDsOld(bool on, int size, int x, int y, int z);
        
        /* Updates the LED cube.*/
        void cubeUpdate();
        
        /* Updates the LED cube given the size, hue and offsets in the X, Y and Z axes.*/
        void UpdateCube(int size, int deltaX, int deltaY, int deltaZ, float hue); 
        
        /* Updates the LED cube given parameters in a CubeUpdateParameters struct.*/
        void UpdateCube2(CubeUpdateParameters cubeParams);
        
        /* Moves the square inside the cube by deltaX in the x-axis,
         * by deltaY in the y-axis, and deltaZ in the z-axis. Returns
         * 1 if movement occured, and -1 if no movement occured.*/
        int move(int deltaX, int deltaY, int deltaZ);
        
        /* Changes the color of the square in the LED cube to the given hue.*/
        void changeColor(float hue);
        
        /* Changes the size of the square in the LED cube to the given size.
         * The minimum size is 1, corresponding to a square of a single LED.
         * A size of 2 corresponds to a 2x2 LED square, 3 corresponds to 3x3 
         * and so forth. If square is on an edge, it moves accordingly in 
         * order to be able to increase size.*/
        void changeSize(int newSize);
        
    private:
        unsigned const X;
        unsigned const Y;
        unsigned const Z;
        int pos[3];
        int prevPos[3];
        int size;
        int prevSize;

        float saturation;
        float brightness;
        uint8_t r;
        uint8_t g;
        uint8_t b;        
        WS2811 ledStrip1;
        WS2811 ledStrip2;
    
};