#include "mbed.h"
#include "BLE.h"
#include "config.h"

// Compile time options
#define IS_SMART_BEACON     1 // Whether targeting nRF51822 Smart Beacon
#define BLINK_LED           1 // Whether to blink the LED
#define CYCLE_CHANNELS      0 // Whether to advertise cycling between
                              // single channels or normally
#define ADVERTISE_NAME      0 // Whether to advertise a name
                              // (increases packet size)
#define BAUD_RATE           115200 // baud

// Experiment parameters
#define DONGLE_NUM          1       // last byte of address, for identification
#define CYCLE_INTERVAL      5000    // ms; see CYCLE_CHANNELS
#define ADV_INTERVAL        100     // ms
#define BLINK_INTERVAL      1000    // ms
#define TX_POWER            -4      // dBm; see mbed API for valid values
#define PAYLOAD_SIZE        2       // bytes
#define TAG_NAME            "MobiusTag" // see ADVERTISE_NAME

BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);

unsigned char CH_37_OFF = 0;
unsigned char CH_38_OFF = 0;
unsigned char CH_39_OFF = 0;
uint8_t channel = 0;

const uint8_t TAG_ADDR[BLEProtocol::ADDR_LEN] =
    {DONGLE_NUM, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA};

Serial pc(p9, p11);

#if IS_SMART_BEACON
#undef LED1
#define LED1 p12
#endif

#if BLINK_LED
void blinkCallback(void)
{
    static DigitalOut led(LED1, 1);
    led = !led;
}
#endif

void setupAdvertising(uint8_t channel, uint16_t interval)
{
    ble.gap().stopAdvertising();
    switch (channel) {
        case 0:
            CH_37_OFF = 0;
            CH_38_OFF = 0;
            CH_39_OFF = 0;
            break;
        case 37:
            CH_37_OFF = 0;
            CH_38_OFF = 1;
            CH_39_OFF = 1;
            break;
        case 38:
            CH_37_OFF = 1;
            CH_38_OFF = 0;
            CH_39_OFF = 1;
            break;
        case 39:
            CH_37_OFF = 1;
            CH_38_OFF = 1;
            CH_39_OFF = 0;
            break;
        default:
            CH_37_OFF = 1;
            CH_38_OFF = 1;
            CH_39_OFF = 1;
            break;
    }
    
    ble.gap().setAdvertisingInterval(interval);
    ble.gap().startAdvertising();
}

#if CYCLE_CHANNELS
void cycleChannelCallback(void)
{
    switch (channel) {
        case 0:
            channel = 37;
            break;
        case 37:
            channel = 38;
            break;
        case 38:
            channel = 39;
            break;
        case 39:
            channel = 0;
            break;
        default:
            channel = 99;
            break;
    }
    
    setupAdvertising(channel, ADV_INTERVAL);
}
#endif

void incSeqNumCallback(void)
{
    static uint8_t packet_no = 0;
    static uint8_t payload[PAYLOAD_SIZE] = {0};
    
    ble.gap().stopAdvertising();
    ble.gap().clearAdvertisingPayload();

    payload[0] = packet_no;
    payload[1] = channel;
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, payload, PAYLOAD_SIZE);
//    ble.gap().accumulateAdvertisingPayload(
//        GapAdvertisingData::BREDR_NOT_SUPPORTED);
#if ADVERTISE_NAME
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::SHORTENED_LOCAL_NAME,
        (const uint8_t*)TAG_NAME, sizeof(TAG_NAME));
#endif
    
    ble.gap().startAdvertising();
    
    packet_no++;
}

void bleInitCallback(BLE::InitializationCompleteCallbackContext *params)
{
    static Ticker cycleChannelTicker;
    
    if (params->error != BLE_ERROR_NONE ||
        params->ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
    
    ble.gap().setAdvertisingType(
        GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    ble.gap().setAddress(BLEProtocol::AddressType::PUBLIC, TAG_ADDR);
    ble.gap().setTxPower(TX_POWER);

#if CYCLE_CHANNELS
    cycleChannelTicker.attach(cycleChannelCallback, CYCLE_INTERVAL/1000.0);
#else
    setupAdvertising(0, ADV_INTERVAL);
#endif
}

int main(void)
{   
    static Ticker ledTicker, incSeqNumTicker;
    
    pc.baud(BAUD_RATE);

#if BLINK_LED
    ledTicker.attach(blinkCallback, BLINK_INTERVAL/1000.0);
#endif

    ble.init(bleInitCallback);
    while (!ble.hasInitialized()) { }

    incSeqNumTicker.attach(incSeqNumCallback, ADV_INTERVAL/1000.0);

    while (true) {
        ble.waitForEvent();
    }
}
