//------------------------------------------------------------------//
//Supported MCU:   RZ/A1H
//File Contents:   Camera Module Test Program (GR-PEACH version on the Micon Car)
//Version number:  Ver.1.00
//Date:            2017.07.13
//Copyright:       Renesas Electronics Corporation
//------------------------------------------------------------------//
 
//This program supports the following boards:
//* GR-PEACH(E version)
//* Camera module (SC-310)
 
//Include
//------------------------------------------------------------------//
#include "mbed.h"
#include "iodefine.h"
#include "DisplayBace.h"
 
//Define
//------------------------------------------------------------------//

//Define(NTSC-Video)
//------------------------------------------------------------------//
#define VIDEO_INPUT_CH          (DisplayBase::VIDEO_INPUT_CHANNEL_0)
#define VIDEO_INT_TYPE          (DisplayBase::INT_TYPE_S0_VFIELD)
#define DATA_SIZE_PER_PIC       (2u)

/*! Frame buffer stride: Frame buffer stride should be set to a multiple of 32 or 128
    in accordance with the frame buffer burst transfer mode. */
#define PIXEL_HW                (320u)  /* QVGA */
#define PIXEL_VW                (240u)  /* QVGA */
#define VIDEO_BUFFER_STRIDE     (((PIXEL_HW * DATA_SIZE_PER_PIC) + 31u) & ~31u)
#define VIDEO_BUFFER_HEIGHT     (PIXEL_VW)

//Constructor
//------------------------------------------------------------------//
DigitalIn   push_sw(P2_13);             /* SW1 on the Motor Drive board */

//Prototype
//------------------------------------------------------------------//
unsigned int pushsw_get( void );

//Prototype(NTSC-video)
//------------------------------------------------------------------//
static void IntCallbackFunc_Vfield(DisplayBase::int_type_t int_type);
static void WaitVfield(const int32_t wait_count);
static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type);
static void WaitVsync(const int32_t wait_count);


//Globle(NTSC-video)
//------------------------------------------------------------------//
static uint8_t FrameBuffer_Video_A[VIDEO_BUFFER_STRIDE * VIDEO_BUFFER_HEIGHT]__attribute((section("NC_BSS"),aligned(16)));  //16 bytes aligned!;
static uint8_t FrameBuffer_Video_B[VIDEO_BUFFER_STRIDE * VIDEO_BUFFER_HEIGHT]__attribute((section("NC_BSS"),aligned(16)));  //16 bytes aligned!;
static volatile int32_t vsync_count;
static volatile int32_t vfield_count;

//Main
//------------------------------------------------------------------//
int main( void )
{
    /* NTSC-Video */
    DisplayBase::graphics_error_t error;
    uint8_t * write_buff_addr = FrameBuffer_Video_A;
    uint8_t * save_buff_addr  = FrameBuffer_Video_B;

    /* Create DisplayBase object */
    DisplayBase Display;

    /* Graphics initialization process */
    error = Display.Graphics_init(NULL);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    error = Display.Graphics_Video_init( DisplayBase::INPUT_SEL_VDEC, NULL);
    if( error != DisplayBase::GRAPHICS_OK ) {
        while(1);
    }

    /* Interrupt callback function setting (Vsync signal input to scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_VI_VSYNC, 0, IntCallbackFunc_Vsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }
    /* Video capture setting (progressive form fixed) */
    error = Display.Video_Write_Setting(
                VIDEO_INPUT_CH,
                DisplayBase::COL_SYS_NTSC_358,
                write_buff_addr,
                VIDEO_BUFFER_STRIDE,
                DisplayBase::VIDEO_FORMAT_YCBCR422,
                DisplayBase::WR_RD_WRSWA_32_16BIT,
                PIXEL_VW,
                PIXEL_HW
            );
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Interrupt callback function setting (Field end signal for recording function in scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(VIDEO_INT_TYPE, 0, IntCallbackFunc_Vfield);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process start */
    error = Display.Video_Start (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process stop */
    error = Display.Video_Stop (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process start */
    error = Display.Video_Start (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Wait vsync to update resister */
    WaitVsync(1);

    /* Wait 2 Vfield(Top or bottom field) */
    WaitVfield(2);

    while(1) {

        /* SW check */
        if ( pushsw_get() ) {
            WaitVfield( 2 );
            if (write_buff_addr == FrameBuffer_Video_A) {
                write_buff_addr = FrameBuffer_Video_B;
                save_buff_addr  = FrameBuffer_Video_A;
            } else {
                write_buff_addr = FrameBuffer_Video_A;
                save_buff_addr  = FrameBuffer_Video_B;
            }
            /* Change write buffer */
            error = Display.Video_Write_Change(
                        VIDEO_INPUT_CH,
                        write_buff_addr,
                        VIDEO_BUFFER_STRIDE);
            if (error != DisplayBase::GRAPHICS_OK) {
                while (1);
            }
        }
    }
}


//pushsw_get(on Motor drive board)
//------------------------------------------------------------------//
unsigned int pushsw_get( void )
{
    return (~push_sw) & 0x1;            /* Read ports with switches */
}


//******************************************************************//
// @brief       Interrupt callback function
// @param[in]   int_type    : VDC5 interrupt type
// @retval      None
//*******************************************************************/
static void IntCallbackFunc_Vfield(DisplayBase::int_type_t int_type)
{
    if (vfield_count > 0) {
        vfield_count--;
    }
}

//******************************************************************//
// @brief       Wait for the specified number of times Vsync occurs
// @param[in]   wait_count          : Wait count
// @retval      None
//*******************************************************************/
static void WaitVfield(const int32_t wait_count)
{
    vfield_count = wait_count;
    while (vfield_count > 0) {
        /* Do nothing */
    }
}

//******************************************************************//
// @brief       Interrupt callback function for Vsync interruption
// @param[in]   int_type    : VDC5 interrupt type
// @retval      None
//*******************************************************************/
static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type)
{
    if (vsync_count > 0) {
        vsync_count--;
    }
}

//******************************************************************//
// @brief       Wait for the specified number of times Vsync occurs
// @param[in]   wait_count          : Wait count
// @retval      None
//*******************************************************************/
static void WaitVsync(const int32_t wait_count)
{
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

//------------------------------------------------------------------//
// End of file
//------------------------------------------------------------------//
