//------------------------------------------------------------------//
//Supported MCU:   RZ/A1H
//File Contents:   Servo center adjustment from the PC
//                                 (GR-PEACH version on the Micon Car)
//Version number:  Ver.1.00
//Date:            2016.01.20
//Copyright:       Renesas Electronics Corporation
//------------------------------------------------------------------//
 
//This program supports the following boards:
//* GR-PEACH(E version)
//* Motor drive board Ver.5
//* Camera module (SC-310)
 
/*
This is a program to check the servo adjustment of the micon car using keys.
The action described takes place when one of the following keys is pressed.
 
Z key: servo offset value +1
X key: servo offset value -1
A key: servo offset value +10
S key: servo offset value -10
*/
 
//Include
//------------------------------------------------------------------//
#include "mbed.h"
#include "iodefine.h"
 
//Define
//------------------------------------------------------------------//
//Servo PWM cycle
#define     SERVO_PWM_CYCLE     33332   /* SERVO PWM period         */
                                        /* 16ms   P0φ/16 = 0.48us   */
#define     SERVO_CENTER        3124    /* 1.5ms / 0.48us - 1 = 3124*/
#define     HANDLE_STEP         18      /* 1 degree value           */
 
//Degrees
#define     DEGREEES_MAX        4791    /* Left  90° 2.3ms / 0.48us = 4791 */
#define     DEGREEES_MIN        1485    /* Right 90° 0.7ms / 0.48us = 1458 */
 
//LED Color on GR-PEACH
#define     LED_OFF             0x00
#define     LED_RED             0x01
#define     LED_GREEN           0x02
#define     LED_YELLOW          0x03
#define     LED_BLUE            0x04
#define     LED_PURPLE          0x05
#define     LED_SKYBLUE         0x06
#define     LED_WHITE           0x07
 
//Status
#define     RUN                 0x00
#define     STOP                0x01
#define     ERROR               0xff
 
//Constructor
//------------------------------------------------------------------//
Ticker      interrput;
Serial      pc(USBTX, USBRX);
DigitalOut  LED_R(P6_13);
DigitalOut  LED_G(P6_14);
DigitalOut  LED_B(P6_15);
 
//Prototype
//------------------------------------------------------------------//
void init_MTU2_PWM_Servo( void );       /* Initialize PWM functions */
void intTimer( void );                  /* Interrupt fanction       */
void timer( unsigned long timer_set );
void handle( int angle );
void led_status_process( int set );     /* Function for only interrupt */
void led_rgb(int led);
 
//Globle
//------------------------------------------------------------------//
volatile int            status;         /* Status                   */
volatile unsigned int   servo_offset;   /* servo offset             */
 
//Main
//------------------------------------------------------------------//
int main( void )
{
    int     i;
    char    c;
 
    /* Initialize MCU functions */
    init_MTU2_PWM_Servo();
    interrput.attach(&intTimer, 0.001);
    pc.baud(9600);
 
    /* Initialize Micon Car state */
    servo_offset = SERVO_CENTER;
 
    pc.printf(
        "Servo center adjustment from the PC\n\r"
        "(Micon Car on GR-PEACH version)\n\r"
        "\n\r"
        "'Z' key   : Center Value +1\n\r"
        "'X' key   : Center Value -1\n\r"
        "\n\r"
        "'A' key   : Center Value +10\n\r"
        "'S' key   : Center Value -10\n\r"
        "\n\r"
    );
 
    pc.printf( "%5d\r", servo_offset );
 
    while( 1 ) {
        status = RUN;
 
        MTU2TGRD_0 = servo_offset;
 
        i = pc.scanf( "%c", &c );
        if( i == 1 ) {
            switch( c ) {
            case 'Z':
            case 'z':
                servo_offset += 1;
                if( servo_offset > DEGREEES_MAX ) servo_offset = DEGREEES_MAX;
                pc.printf( "%5d\r", servo_offset );
                break;
 
            case 'A':
            case 'a':
                servo_offset += 10;
                if( servo_offset > DEGREEES_MAX ) servo_offset = DEGREEES_MAX;
                pc.printf( "%5d\r", servo_offset );
                break;
 
            case 'X':
            case 'x':
                servo_offset -= 1;
                if( servo_offset < DEGREEES_MIN ) servo_offset = DEGREEES_MIN;
                pc.printf( "%5d\r", servo_offset );
                break;
 
            case 'S':
            case 's':
                servo_offset -= 10;
                if( servo_offset < DEGREEES_MIN ) servo_offset = DEGREEES_MIN;
                pc.printf( "%5d\r", servo_offset );
                break;
 
            default:
                break;
            }
        }
    }
}
 
//Initialize MTU2 PWM functions
//------------------------------------------------------------------//
//MTU2_0
//PWM mode 1
//TIOC0A(P4_0) :Servo-motor
//------------------------------------------------------------------//
void init_MTU2_PWM_Servo( void )
{
    /* Port setting for S/W I/O Contorol */
    /* alternative mode     */
 
    /* MTU2_0 (P4_0)        */
    GPIOPBDC4   = 0x0000;               /* Bidirection mode disabled*/
    GPIOPFCAE4 &= 0xfffe;               /* The alternative function of a pin */
    GPIOPFCE4  &= 0xfffe;               /* The alternative function of a pin */
    GPIOPFC4   |= 0x0001;               /* The alternative function of a pin */
                                        /* 2nd alternative function/output   */
    GPIOP4     &= 0xfffe;               /*                          */
    GPIOPM4    &= 0xfffe;               /* p4_0:output              */
    GPIOPMC4   |= 0x0001;               /* P4_0:double              */
 
    /* Mosule stop 33(MTU2) canceling */
    CPGSTBCR3  &= 0xf7;
 
    /* MTU2_0 (Motor PWM) */
    MTU2TCR_0   = 0x22;                 /* TCNT Clear(TGRA), P0φ/16 */
    MTU2TIORH_0 = 0x52;                 /* TGRA L>H, TGRB H>L       */
    MTU2TMDR_0  = 0x32;                 /* TGRC and TGRD = Buff-mode*/
                                        /* PWM-mode1                */
    MTU2TCNT_0  = 0;                    /* TCNT0 Set 0              */
    MTU2TGRA_0  = MTU2TGRC_0 = SERVO_PWM_CYCLE;
                                        /* PWM-Cycle(16ms)          */
    MTU2TGRB_0  = MTU2TGRD_0 = 0;       /* Servo-motor(P4_0)        */
    MTU2TSTR   |= 0x01;                 /* TCNT_0 Start             */
}
 
//Interrupt Timer
//------------------------------------------------------------------//
void intTimer( void )
{
    led_status_process( status );
}
 
//LED_Status(on GR-PEACH board) Function for only interrupt
//------------------------------------------------------------------//
void led_status_process( int set )
{
    static unsigned long    led_timer;
    int                     led_set;
    int                     on_time;
    int                     off_time;
 
    /* setting */
    switch( set ){
    case RUN:
        led_set  = LED_GREEN;
        on_time  = 500;
        off_time = 500;
        break;
    case STOP:
        led_set  = LED_RED;
        on_time  = 500;
        off_time = 0;
        break;
    default:
        led_set  = LED_OFF;
        on_time  = 0;
        off_time = 1;
        break;
    }
 
    /* Display */
    led_timer++;
    if( led_timer < on_time ) led_rgb( led_set );
    else if( led_timer < ( on_time + off_time ) ) led_rgb( LED_OFF );
    else led_timer = 0;
}
 
//LED_RGB(on GR-PEACH board)
//------------------------------------------------------------------//
void led_rgb(int led)
{
    LED_R = led & 0x1;
    LED_G = (led >> 1 ) & 0x1;
    LED_B = (led >> 2 ) & 0x1;
}
 
//------------------------------------------------------------------//
// End of file
//------------------------------------------------------------------//