/** 10_rtos_mempool
 *
 * Simple example to use a Mempool object together with a Queue
 * object in order to send more complex messages from thread1 to thread2.
 * Thread1 generates transition between random RGB colors and sends
 * messages to Thread2. Thread2 displays the requested color on the
 * built in RGB LED.
 *
 * Data items for messages are stored in a MemoryPool object.
 * Thread1 allocates MemoryPool slots, Thread2 should free up
 * these slots after using up data stored in the slot.
 */

#include "mbed.h"
#include "rtos.h"
PwmOut rled(LED_RED);
PwmOut gled(LED_GREEN);
PwmOut bled(LED_BLUE);

typedef struct {
    float red;
    float green;
    float blue;
} message_t;

MemoryPool<message_t, 4> mpool;                     //Memory pool for data storage
Queue<message_t, 4> queue;                          //Message queue for 4 items

void led_thread(void const *argument)
{
    rled.period_ms(20);                             //Set period to 20 ms
    rled.write(1.0f);                               //Initialize to 0% duty cycle
    gled.period_ms(20);                             //Set period to 20 ms
    gled.write(1.0f);                               //Initialize to 0% duty cycle
    bled.period_ms(20);                             //Set period to 20 ms
    bled.write(1.0f);                               //Initialize to 0% duty cycle
    while (true) {
        osEvent evt = queue.get();                  //Wait for a message
        if(evt.status == osEventMessage) {
            message_t *mymessage = (message_t*)evt.value.p;
            rled = 1.0f - mymessage->red;
            gled = 1.0f - mymessage->green;
            bled = 1.0f - mymessage->blue;
            mpool.free(mymessage);                  //Free up memory
        }
    }
}

float frand(void)
{
    float rv = (float)(rand()&0xFFFF);
    return (rv/65536.0f);
}

int main (void)
{
    float RGB1[3];
    float RGB2[3];
    float INC[3];
    Thread thread2(led_thread);
//--- Create a random color ---------------------
    for (int x=0; x<3; x++) {
        RGB1[x] = frand();
    }

    while (true) {
//--- Create a new random color -----------------
        for (int x=0; x<3; x++) {
            RGB2[x] = frand();
        }
//--- Determine increments to go from color 1 to color 2 in 25 steps
        for (int x=0; x<3; x++) {
            INC[x] = (RGB1[x] - RGB2[x]) / 25;
        }       
//--- Send color codes to thread2 ---------------
        for (int s=0; s<25; s++) {
            message_t *message = mpool.alloc();     //Allocate memory
            message->red = RGB1[0];
            message->green = RGB1[1];
            message->blue = RGB1[2];
            queue.put(message);                     //Send data as message
            Thread::wait(100);
            for (int x=0; x<3; x++) {
                RGB1[x] -= INC[x];                  //Approach to second colour
            }
        }
    }
}
