#include "mbed.h"
#include "BMX055.h"
#include <math.h>

const float G = 9.80665;
const float PI = 3.14159;

static int num_of_points = 0;
static float points[1024][3];


BMX055::BMX055(PinName SDA, PinName SCL) :
    bmx055(SDA, SCL),
    accel(),
    gyroscope(),
    magnet()
{
    bmx055.frequency(100000);
    bmx_init();
}

void BMX055::bmx_init(void)
{
    char buf[2] = { 0 };
    printf("accel setting\n");
    buf[0] = 0x14;
    buf[1] = 0xB6;
    bmx055.write(ACC, buf, 2);
    wait_ms(2);
    buf[0] = 0x0F;
    buf[1] = 0x05;
    buf[0] = 0x10;
    buf[1] = 0x08;
    bmx055.write(ACC, buf, 2);
    buf[0] = 0x11;
    buf[1] = 0x00;
    bmx055.write(ACC, buf, 2);
    wait_ms(2);

    printf("gyroscope setting\n");
    buf[0] = 0x0F;
    buf[1] = 0x02;
    bmx055.write(GYRO, buf, 2);
    wait(0.1);
    buf[0] = 0x10;
    buf[1] = 0x00;
    bmx055.write(GYRO, buf, 2);
    wait(0.1);
    buf[0] = 0x11;
    buf[1] = 0x00;
    bmx055.write(GYRO, buf, 2);
    wait(0.1);

    printf("magnet setting\n");
    buf[0] = 0x4B;
    buf[1] = 0x82;
    bmx055.write(MAG, buf, 2);
    wait(0.1);
    buf[0] = 0x4B;
    buf[1] = 0x01;
    bmx055.write(MAG, buf, 2);
    wait(0.1);
    buf[0] = 0x4C;
    buf[1] = 0x00;
    bmx055.write(MAG, buf, 2);
    buf[0] = 0x4E;
    buf[1] = 0x84;
    bmx055.write(MAG, buf, 2);
    buf[0] = 0x51;
    buf[1] = 0x04;
    bmx055.write(MAG, buf, 2);
    buf[0] = 0x52;
    buf[1] = 0x16;
    bmx055.write(MAG, buf, 2);
    wait(0.1);

    buf[0] = 0x00;
    bmx055.write(MAG, buf, 1, 1);
    bmx055.read(MAG, buf, 1);
    printf("read:0x%02x\n", buf[0]);

    timer.start();
    before_time = -1;
}

void BMX055::Get_acc(void)
{
    uint8_t data[6] = { 0 };
    char send[1], get[1];
    char temp;

    send[0] = (char)(2);
    bmx055.write(ACC, send, 1, true);
    bmx055.read(ACC, (char*)data, 6);

    for (int i = 0; i < 3; i++) {
        accel[i] = (int16_t)(((int16_t)data[i * 2 + 1] << 8) | data[i * 2]) >> 4;
        if (accel[i] > 2047)accel[i] -= 4096;
        accel[i] = (accel[i] / 1024 * G - acc_biases[i]) * acc_gains[i];
    }
}

void BMX055::Get_gyro(void)
{
    int data[6] = { 0 };
    char send[1], get[1];
    char temp;

    for (int i = 0; i < 6; i++) {
        send[0] = (char)(2 + i);
        bmx055.write(GYRO, send, 1);
        bmx055.read(GYRO, get, 1);
        temp = get[0];
        data[i] = temp;
    }

    for (int i = 0; i < 3; i++) {
        gyroscope[i] = (int16_t)(((int16_t)data[i * 2 + 1] << 8) | data[i * 2]) >> 4;
        if (gyroscope[i] > 32767)gyroscope[i] -= 65536;
        gyroscope[i] = (gyroscope[i] * (8000.0 / 32768) * (PI / 180) - gyro_biases[i]) * gyro_gains[i];
    }
}

void BMX055::Get_mag(void)
{
    int data[8] = { 0 };
    char send[1], get[1];
    char temp;

    for (int i = 0; i < 8; i++) {
        send[0] = (char)(0x42 + i);
        bmx055.write(MAG, send, 1);
        bmx055.read(MAG, get, 1);
        temp = get[0];
        data[i] = temp;
    }

    for (int i = 0; i < 3; i++) {
        if (i != 2)magnet[i] = (int16_t)(((int16_t)data[i * 2 + 1] << 8) | data[i * 2]) >> 3;
        else magnet[i] = (int16_t)(((int16_t)data[i * 2 + 1] << 8) | data[i * 2]) >> 1;
        if (i == 2 && magnet[i] > 16383)magnet[i] -= 32768;
        else if (i != 2 && magnet[i] > 4095)magnet[i] -= 8192;
        magnet[i] -= mag_biases[i];
        magnet[i] /= mag_gains[i];
    }
}

void BMX055::Update_posture()
{
    double dt = 0;
    double after_time = 0;

    after_time = timer.read();

    Get_gyro();

    dt = after_time - before_time;

    before_time = after_time;

    Quaternion omega(0, gyroscope[0], gyroscope[1], gyroscope[2]);

    Quaternion dq = (posture * 0.5f) * omega;

    posture += (dq * dt);/////////////////////////////////////////////////////

    posture /= posture.norm();////////////////////////////////////////////////




    Quaternion acc_rotation_vector;

    Get_acc();
    float acc_weight=0.0;

    Quaternion acc_vector_observed(0, accel[0], accel[1], accel[2]);
    acc_weight = ( 0.41 - fabs((9.797 - acc_vector_observed.norm()) / 9.797) ) * 0.5;

    if(acc_weight<0) {
        acc_weight = 0.0;
    }

    acc_vector_observed /= acc_vector_observed.norm();

    Quaternion acc_vector_estimated;

    acc_vector_estimated = posture.conjugate() * initial_acc * posture;

    acc_vector_estimated /= acc_vector_estimated.norm();

    acc_rotation_vector.w = 0;
    acc_rotation_vector.x = acc_vector_observed.y * acc_vector_estimated.z - acc_vector_observed.z * acc_vector_estimated.y;
    acc_rotation_vector.y = acc_vector_observed.z * acc_vector_estimated.x - acc_vector_observed.x * acc_vector_estimated.z;
    acc_rotation_vector.z = acc_vector_observed.x * acc_vector_estimated.y - acc_vector_observed.y * acc_vector_estimated.x;

    acc_rotation_vector *= acos((acc_vector_estimated.x * acc_vector_observed.x) + (acc_vector_estimated.y * acc_vector_observed.y) + (acc_vector_estimated.z * acc_vector_observed.z));


    acc_rotation_vector *= acc_weight;

    Quaternion mag_rotation_vector;

    Get_mag();

    float copy=0.0f;
    copy = magnet[0];
    magnet[0] = -magnet[1];
    magnet[1] = copy;

    Quaternion mag_vector_observed(0, magnet[0], magnet[1], magnet[2]);

    mag_vector_observed /= mag_vector_observed.norm();

    Quaternion mag_vector_estimated;

    mag_vector_estimated = posture.conjugate() * initial_mag * posture;

    mag_vector_estimated /= mag_vector_estimated.norm();




    mag_rotation_vector.w = 0;
    mag_rotation_vector.x = mag_vector_observed.y * mag_vector_estimated.z - mag_vector_observed.z * mag_vector_estimated.y;
    mag_rotation_vector.y = mag_vector_observed.z * mag_vector_estimated.x - mag_vector_observed.x * mag_vector_estimated.z;
    mag_rotation_vector.z = mag_vector_observed.x * mag_vector_estimated.y - mag_vector_observed.y * mag_vector_estimated.x;

    mag_rotation_vector *= acos((mag_vector_estimated.x * mag_vector_observed.x) + (mag_vector_estimated.y * mag_vector_observed.y) + (mag_vector_estimated.z * mag_vector_observed.z));

    mag_rotation_vector *= 0.2f;///////////////////////////////////////////////////////////////////////////////








    Quaternion rotation_vector;
    Quaternion rotation_quaternion;

    rotation_vector=(acc_rotation_vector+mag_rotation_vector)/2.0;

    rotation_quaternion.w = cos(rotation_vector.norm() / 2);
    rotation_quaternion.x = (rotation_vector.x / rotation_vector.norm()) * sin(rotation_vector.norm() / 2);
    rotation_quaternion.y = (rotation_vector.y / rotation_vector.norm()) * sin(rotation_vector.norm() / 2);
    rotation_quaternion.z = (rotation_vector.z / rotation_vector.norm()) * sin(rotation_vector.norm() / 2);

    rotation_quaternion=posture*rotation_quaternion*posture.conjugate();

    posture=rotation_quaternion*posture;

    posture /= posture.norm();

}

void BMX055::mag_calibration(DigitalIn button)
{

    printf("たくさん回したらもう一度青いボタンを押せ\n");

    for (int i = 0; i < 1024; i++) {
        Get_mag();
        for (int j = 0; j < 3; j++) {
            points[i][j] = magnet[j];
        }
        num_of_points++;
        if (button == 0) {
            break;
        }
        wait(0.01);
    }

    float Mat[6][7];

    for (int i = 0; i < 3; i++) {
        for (int j = 0; j < 3; j++) {
            Mat[i][j] = 0;
            for (int k = 0; k < num_of_points; k++) {
                Mat[i][j] += 2.0f * (points[k][i] * points[k][i]) * (points[k][j] * points[k][j]);
            }
        }

        for (int j = 3; j < 6; j++) {
            Mat[i][j] = 0;
            for (int k = 0; k < num_of_points; k++) {
                Mat[i][j] += 2.0f * (points[k][i] * points[k][i]) * points[k][j - 3];
            }
        }
    }

    for (int i = 3; i < 6; i++) {
        for (int j = 0; j < 3; j++) {
            Mat[i][j] = 0;
            for (int k = 0; k < num_of_points; k++) {
                Mat[i][j] += 2.0f * points[k][i - 3] * (points[k][j] * points[k][j]);
            }
        }

        for (int j = 3; j < 6; j++) {
            Mat[i][j] = 0;
            for (int k = 0; k < num_of_points; k++) {
                Mat[i][j] += 2.0f * points[k][i - 3] * points[k][j - 3];
            }
        }
    }

    for (int i = 0; i < 3; i++) {
        Mat[i][6] = 0;
        for (int k = 0; k < num_of_points; k++) {
            Mat[i][6] += 2.0f * points[k][i] * points[k][i];
        }
    }

    for (int i = 3; i < 6; i++) {
        Mat[i][6] = 0;
        for (int k = 0; k < num_of_points; k++) {
            Mat[i][6] += 2.0f * points[k][i - 3];
        }
    }



    for (int i = 0; i < 5; i++) {
        int max = i;

        for (int j = i; j < 6; j++) {
            if (fabs(Mat[j][i]) > fabs(Mat[i][max])) {
                max = j;
            }
        }

        if (max != i) {
            float element;
            for (int j = 0; j < 7; j++) {
                element = Mat[i][j];
                Mat[i][j] = Mat[max][j];
                Mat[max][j] = element;
            }
        }



        float coeff = 0;

        for (int j = i + 1; j < 6; j++) {
            coeff = Mat[j][i] / Mat[i][i];
            for (int k = i; k < 7; k++) {
                Mat[j][k] -= Mat[i][k] * coeff;
            }
        }
    }


    for (int i = 5; i > 0; i--) {
        float coeff = 0;
        for (int j = i - 1; j > -1; j--) {
            coeff = Mat[j][i] / Mat[i][i];
            for (int k = i; k < 7; k++) {
                Mat[j][k] -= Mat[i][k] * coeff;
            }
        }
    }

    for (int i = 0; i < 6; i++) {
        Mat[i][6] /= Mat[i][i];
        Mat[i][i] /= Mat[i][i];
    }





    float r = 1;
    for (int i = 0; i < 3; i++) {
        r += (Mat[i + 3][6] * Mat[i + 3][6]) / (4 * Mat[i][6]);
    }

    for (int i = 0; i < 3; i++) {
        mag_gains[i] = sqrt(r / Mat[i][6]);
    }

    for (int i = 0; i < 3; i++) {
        mag_biases[i] = -Mat[i + 3][6] / (2 * Mat[i][6]);
    }

    for (int a = 0; a < 3; a++) {
        printf("%f,%f\n", mag_gains[a], mag_biases[a]);
    }


}

void BMX055::set_initial_mag()
{
    Get_mag();

    float copy=0.0f;
    copy = magnet[0];
    magnet[0] = -magnet[1];
    magnet[1] = copy;

    initial_mag.w = 0;
    initial_mag.x = magnet[0];
    initial_mag.y = magnet[1];
    initial_mag.z = magnet[2];

    initial_mag /= initial_mag.norm();
}

void BMX055::set_initial_acc()
{
    Get_acc();

    initial_acc.w = 0;
    initial_acc.x = accel[0];
    initial_acc.y = accel[1];
    initial_acc.z = accel[2];

    initial_acc /= initial_acc.norm();
}

float BMX055::get_azimuth_machineframe()
{

    float sin_azimuth = -2.0 * posture.z * posture.w;
    float cos_azimuth = (posture.w * posture.w) - (posture.z * posture.z);

    if (sin_azimuth > 0) {
        return acos(cos_azimuth);
    } else {
        return 2.0f * PI - acos(cos_azimuth);
    }
}