/*******************************************************************************
* Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#ifndef __MAX77650_H_
#define __MAX77650_H_

// Include
#include "mbed.h"



/**
 * @brief Library for the MAX77650
 * The MAX77650/MAX77651 provide highly-integrated battery charging and
 * power supply solutions for low-power wearable applications where size and
 * efficiency are critical. Both devices feature a SIMO buck-boost regulator
 * that provides three independently programmable power rails from a single
 * inductor to minimize total solution size. A 150mA LDO provides ripple
 * rejection for audio and other noise-sensitive applications. A highly
 * configurable linear charger supports a wide range of Li+ battery capacities
 * and includes battery temperature monitoring for additional safety (JEITA).
 *
 * @code
 * #include "mbed.h"
 * #include "MAX77650"
 *
 *
 * // Hardware serial port
 * Serial serial(USBTX, USBRX);
 *
 * //SPI communications
 * I2C i2c(SCL, SDA);
 *
 * //Fuel Gauge
 * MAX77650 max77650(i2C, Sutff );//To be completed
 *
 *
 * int main(void)
 * {
 *     CODE CODE
 *      while(true)
 *      {
 *          CODE CODE
 *      }
 * }
 * @endcode
 */

// Definitions

#define DEVICE_ID       0x00
#define NUM_INTERRUPTS  12



class MAX77650
{

public:

    ///8-bit write address
    static const int I2C_W_ADRS = 0x90;
    ///8-bit read address
    static const int I2C_R_ADRS = 0x91;

    /**
     * @brief       Register Addresses
     * @details     Enumerated MAX77650 register addresses
     *
     */ 
    enum registers_t {
        INT_GLBL,                       ///< Global Interrupt
        INT_CHG,                        ///< Charger Interrupt
        STAT_CHG_A,                     ///< Status of charging station A
        STAT_CHG_B,                     ///< Status of charging station B
        ERCFLAG,                        ///< Flag Status
        STAT_GLBL,                      ///< Status
        INTM_GLBL,                      ///< Interrupt Mask
        INT_M_CHG,                      ///< Interrupt Mask for Charger

        CNFG_GLBL = 0x10,               ///< Global Resources
        CID,                            ///< Data Register
        CNFG_GPIO,                      ///< GPIO Config Register
        CNFG_CHG_A = 0x18,                     ///< Configuration Charging Station A
        CNFG_CHG_B,                     ///< Configuration Charging Station B
        CNFG_CHG_C,                     ///< Configuration Charging Station C
        CNFG_CHG_D,                     ///< Configuration Charging Station D
        CNFG_CHG_E,                     ///< Configuration Charging Station E
        CNFG_CHG_F,                     ///< Configuration Charging Station F
        CNFG_CHG_G,                     ///< Configuration Charging Station G
        CNFG_CHG_H,                     ///< Configuration Charging Station H
        CNFG_CHG_I,                     ///< Configuration Charging Station I

        CNFG_SBB_TOP = 0x28,            ///< Configuration SIMO Buck Boost Register Top
        CNFG_SBB0_A,                    ///< Configuration SIMO Buck Boost Register A, CN0
        CNFG_SBB0_B,                    ///< Configuration SIMO Buck Boost Register B, CN0
        CNFG_SBB1_A,                    ///< Configuration SIMO Buck Boost Register A, CN1
        CNFG_SBB1_B,                    ///< Configuration SIMO Buck Boost Register B, CN1
        CNFG_SBB2_A,                    ///< Configuration SIMO Buck Boost Register A, CN2
        CNFG_SBB2_B,                    ///< Configuration SIMO Buck Boost Register B, CN2

        CNFG_LDO_A = 0x38,              ///< LDO Configuration Register A
        CNFG_LDO_B,                     ///< LDO Configuration Register B

        CNFG_LED0_A = 0x40,             ///< LED Configuration A for LED number 0
        CNFG_LED1_A,                    ///< LED Configuration A for LED number 1
        CNFG_LED2_A,                    ///< LED Configuration A for LED number 2

        CNFG_LED0_B = 0x43,             ///< LED Configuration B for LED number 0
        CNFG_LED1_B,                    ///< LED Configuration B for LED number 1
        CNFG_LED2_B,                    ///< LED Configuration B for LED number 2
        CNFG_LED_TOP,                   ///< LED Configuration Master Control
    };

    /**
     * @brief MAX77650 constructor.
     * @details Default Constructor
     *
     */
    MAX77650(I2C &i2c);

    /**
     * @brief MAX77650 Destructor
     */
    ~MAX77650();

    /**
     * @brief       Write Register
     * @details     Writes data to MAX77650 Register
     *
     * @parameters  reg_addr Registers to write
     *              reg_data Data to write
     */
    int writeReg(registers_t reg_addr, uint8_t reg_data);

    /**
    * @brief       Read Register
    * @details     Reads data from MAX77650 register
    *
    * @parameters  reg_addr Register to read
    */
    int readReg(registers_t reg_addr, uint8_t &value);


    /**
     * @brief       init Charger Function
     * @details     Sets the parameters for the charger.
     *
     * @parameters  reg_addr Register to read
     */

    int initCharger();

    /**
     * @brief       Enable Charger Function
     * @details     Sets the parameters for the charger.
     *
     * @parameters  reg_addr Register to read
     */

    int enCharger();

    /**
     * @brief       Disable Charger Function
     * @details     Sets the parameters for the charger.
     *
     * @parameters  reg_addr Register to read
     */
    int disblCharger();




private:
    I2C &m_i2cBus;
    int interrupt_pin_num;
};

#endif  /* _MAX77650_H_ */