/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef RTCBASE_H_
#define RTCBASE_H_

#include "mbed.h"
#include <ctime>


/**
 * @brief	RTC base driver class for Maxim Max3134x RTC series.
 */
class RtcBase
{
public:
	/**
	* @brief	Mode of the comparator
	*/
	typedef enum {
		POW_MGMT_MODE_COMPARATOR,		/**< Comparator */
		POW_MGMT_MODE_POWER_MANAGEMENT,	/**< Power Management / Trickle Charger Mode */
	} power_mgmt_mode_t;

	/**
	* @brief	Analog comparator threshold voltage
	*/
	typedef enum {
		COMP_THRESH_1V4,	/**< 1.4V */
		COMP_THRESH_1V6,	/**< 1.6V */
		COMP_THRESH_1V8,	/**< 1.8V */
		COMP_THRESH_2V0,	/**< 2.0V */
	} comp_thresh_t;

	/**
	* @brief	Supply voltage select.
	*/
	typedef enum {
		POW_MGMT_SUPPLY_SEL_AUTO,	/**< Circuit decides whether to use VCC or VBACKUP */
		POW_MGMT_SUPPLY_SEL_VCC,	/**< Use VCC as supply */
		POW_MGMT_SUPPLY_SEL_AIN,	/**< Use AIN as supply */
	} power_mgmt_supply_t;

	/**
	* @brief	Selection of charging path's resistor value
	*/
	typedef enum {
		TRICKLE_CHARGER_3K5,	/**< 3500 Ohm */
		TRICKLE_CHARGER_3K5_2,	/**< 3500 Ohm */
		TRICKLE_CHARGER_7K,		/**< 7000 Ohm */
		TRICKLE_CHARGER_13K,	/**< 13000 Ohm */
	} trickle_charger_ohm_t;

	/**
	* @brief	Timer frequency selection
	*/
	typedef enum {
		TIMER_FREQ_1024HZ,	/**< 1024Hz */
		TIMER_FREQ_256HZ,	/**< 256Hz */
		TIMER_FREQ_64HZ,	/**< 64Hz */
		TIMER_FREQ_16HZ,	/**< 16Hz */
	} timer_freq_t;

	/**
	* @brief	CLKIN frequency selection
	*/
	typedef enum {
		CLKIN_FREQ_1HZ,		/**< 1Hz*/
		CLKIN_FREQ_50HZ,	/**< 50Hz */
		CLKIN_FREQ_60HZ,	/**< 60Hz */
		CLKIN_FREQ_32HZ768,	/**< 32.768Hz */
	} clkin_freq_t;

	/**
	* @brief	Square wave output frequency selection on CLKOUT pin
	*/
	typedef enum {
		SQUARE_WAVE_OUT_FREQ_1HZ,		/**< 1Hz */
		SQUARE_WAVE_OUT_FREQ_4096HZ,	/**< 4.098kHz */
		SQUARE_WAVE_OUT_FREQ_8192HZ,	/**< 8.192kHz */
		SQUARE_WAVE_OUT_FREQ_32768HZ,	/**< 32.768kHz */
	} square_wave_out_freq_t;

	/**
	* @brief	Selection of interrupt ids
	*/
	typedef enum {
		INTR_ID_ALARM1,		/**< Alarm1 flag */
		INTR_ID_ALARM2,		/**< Alarm2 flag */
		INTR_ID_TIMER,		/**< Timer interrupt flag */
		INTR_ID_RESERVED,
		INTR_ID_EXTERNAL,	/**< External interrupt flag for DIN1 */
		INTR_ID_ANALOG,		/**< Analog Interrupt flag / Power fail flag  */
		INTR_ID_OSF,		/**< Oscillator stop flag */
		INTR_ID_LOS,		/**< Loss of signal */
		INTR_ID_END,
	} intr_id_t;


	/**
	* @brief	Alarm number selection
	*/
	typedef enum {
		ALARM1,	/**< Alarm number 1 */
		ALARM2,	/**< Alarm number 2 */
	} alarm_no_t;

	/**
	* @brief	Alarm periodicity selection
	*/
	typedef enum {
		ALARM_PERIOD_EVERYSECOND,	/**< Once per second */
		ALARM_PERIOD_EVERYMINUTE,	/**< Second match / Once per minute */
		ALARM_PERIOD_HOURLY,		/**< Second and Minute match */
		ALARM_PERIOD_DAILY,			/**< Hour, Minute and Second match*/
		ALARM_PERIOD_WEEKLY,		/**< Day and Time match */
		ALARM_PERIOD_MONTHLY,		/**< Date and Time match */
		ALARM_PERIOD_YEARLY,		/**< Month, Date and Time match (Max31342 only) */
		ALARM_PERIOD_ONETIME,		/**< Year, Month, Date and Time match (Max31342 only) */
	} alarm_period_t;

	/**
	* @brief	Selection of INTA/CLKIN pin function
	*/
	typedef enum {
		CONFIGURE_PIN_AS_INTA,	/**< Configure pin as interrupt out */
		CONFIGURE_PIN_AS_CLKIN,	/**< Configure pin as external clock in  */
	} config_inta_clkin_pin_t;

	/**
	* @brief	Selection of INTB/CLKOUT pin function
	*/
	typedef enum {
		CONFIGURE_PIN_AS_CLKOUT,	/**< Output is square wave */
		CONFIGURE_PIN_AS_INTB,		/**< Output is interrupt */
	} config_intb_clkout_pin_t;

	/**
	 * @brief	Selection of sync delay
	 */
	typedef enum {
		SYNC_DLY_LESS_THAN_1SEC = 0,	/**<  Sync delay less than 1 second, recommended for external 1Hz clock */
		SYNC_DLY_LESS_THAN_100MS,		/**<  Sync delay less than 100 msec, recommended for external 50Hz/60Hz/32KHz clock */
		SYNC_DLY_LESS_THAN_20MS,		/**<  Sync delay less than 20 msec, recommended for internal clock */
	} sync_delay_t;

	typedef enum {
		TTS_INTERNAL_1SEC = 0,
		TTS_INTERNAL_2SEC,
		TTS_INTERNAL_4SEC,
		TTS_INTERNAL_16SEC,
		TTS_INTERNAL_32SEC,
		TTS_INTERNAL_64SEC,
		TTS_INTERNAL_128SEC,
	} ttsint_t;

	/**
	* @brief	Function pointer type to interrupt handler function
	*/
	typedef void (*interrupt_handler_function)(void *);

	/**
	* @brief		Read from a register.
	*
	* @param[in]	reg Address of a register to be read.
	* @param[out]	value Pointer to save result value.
	* @param[in]	len Size of result to be read.
	*
	* @returns		0 on success, negative error code on failure.
	*/
	int read_register(uint8_t reg, uint8_t *value, uint8_t len);

	/**
	* @brief		Write to a register.
	*
	* @param[in]	reg Address of a register to be written.
	* @param[out]	value Pointer of value to be written to register.
	* @param[in]	len Size of result to be written.
	*
	* @returns		0 on success, negative error code on failure.
	*/
	int write_register(uint8_t reg, const uint8_t *value, uint8_t len);

	/**
	* @brief		Read time info from RTC.
	*
	* @param[out]	rtc_time Time info from RTC.
	*
	* @returns		0 on success, negative error code on failure.
	*/
	int get_time(struct tm *rtc_ctime);

	/**
	* @brief		Set time info to RTC.
	*
	* @param[in]	rtc_time Time info to be written to RTC.
	*
	* @returns		0 on success, negative error code on failure.
	*/
	int set_time(const struct tm *rtc_ctime);

	/**
	* @brief		Non-volatile memory write
	*
	* @param[out]	buffer Pointer to the data to be written
	* @param[in]	offset Offset of location in NVRAM
	* @param[in]	length Number of bytes to write
	*
	* @return		0 on success, error code on failure
	*/
	int nvram_write(const uint8_t *buffer, int offset, int length);

	/**
	* @brief		Non-volatile memory read
	*
	* @param[in]	buffer Buffer to read in to
	* @param[in]	offset Offset of location in NVRAM
	* @param[in]	length Number of bytes to read
	*
	* @return		0 on success, error code on failure
	*/
	int nvram_read(uint8_t *buffer, int offset, int length);

	/**
	* @brief		NVRAM size of the part
	*
	* @return		0 if part does not have a NVRAM, otherwise returns size
	*/
	int nvram_size();

	/**
	* @brief		Set an alarm condition
	*
	* @param[in]	alarm_no Alarm number, ALARM1 or ALARM2
	* @param[in]	alarm_time Pointer to alarm time to be set
	* @param[in]	period Alarm periodicity, one of ALARM_PERIOD_*
	*
	* @return		0 on success, error code on failure
	*/
	int set_alarm(alarm_no_t alarm_no, const struct tm *alarm_time, alarm_period_t period);

	/**
	* @brief		Get alarm data & time
	*
	* @param[in]	alarm_no Alarm number, ALARM1 or ALARM2
	* @param[out]	alarm_time Pointer to alarm time to be filled in
	* @param[out]	period Pointer to the period of alarm, one of ALARM_PERIOD_*
	* @param[out]	is_enabled Pointer to the state of alarm
	*
	* @return		0 on success, error code on failure
	*/
	int get_alarm(alarm_no_t alarm_no, struct tm *alarm_time, alarm_period_t *period, bool *is_enabled);

	/**
	* @brief		Select power management mode of operation
	*
	* @param[in]	mode Mode selection, one of COMP_MODE_*
	*
	* @return		0 on success, error code on failure
	*/
	int set_power_mgmt_mode(power_mgmt_mode_t mode);

	/**
	* @brief		Set comparator threshold
	*
	* @param[in]	th Set Analog Comparator Threshold level, one of COMP_THRESH_*
	*
	* @return		0 on success, error code on failure
	*/
	int comparator_threshold_level(comp_thresh_t th);

	/**
	* @brief		Select device power source
	*
	* @param[in]	supply Supply selection, one of POW_MGMT_SUPPLY_SEL_*
	*
	* @return		0 on success, error code on failure
	*/
	int supply_select(power_mgmt_supply_t supply);

	/**
	* @brief		Configure trickle charger charging path, also enable it
	*
	* @param[in]	res Value of resister
	* @param[in]	diode Enable diode
	*
	* @return		0 on success, error code on failure
	*/
	int trickle_charger_enable(trickle_charger_ohm_t res, bool diode);

	/**
	* @brief		Disable trickle charger
	*
	* @return		0 on success, error code on failure
	*/
	int trickle_charger_disable();

	/**
	* @brief		Select square wave output frequency selection
	*
	* @param[in]	freq Clock frequency, one of CLKOUT_FREQ_*
	*
	* @return		0 on success, error code on failure
	*/
	int set_output_square_wave_frequency(square_wave_out_freq_t freq);

	/**
	* @brief		Select external clock input frequency
	*
	* @param[in]	freq Clock frequency, one of CLKIN_FREQ_*
	*
	* @return		0 on success, error code on failure
	*/
	int set_clkin_frequency(clkin_freq_t freq);

	/**
	* @brief		Select direction of INTB/CLKOUT pin
	*
	* @param[in]	sel Pin function, one of CONFIGURE_PIN_B3_AS_INTB or CONFIGURE_PIN_B3_AS_CLKOUT
	*
	* @return		0 on success, error code on failure
	*/
	int configure_intb_clkout_pin(config_intb_clkout_pin_t sel);

	/**
	* @brief		Select direction of INTA/CLKIN pin
	*
	* @param[in]	sel Pin function, one of CONFIGURE_PIN_B3_AS_INTA or CONFIGURE_PIN_B3_AS_CLKIN
	*
	* @return		0 on success, error code on failure
	*/
	int configure_inta_clkin_pin(config_inta_clkin_pin_t sel);

	/**
	* @brief		Initialize timer
	*
	* @param[in]	value Timer initial value
	* @param[in]	repeat Timer repeat mode enable/disable
	* @param[in]	freq Timer frequency, one of TIMER_FREQ_*
	* @param[in]	mode Timer mode, 0 or 1
	*
	* @return		0 on success, error code on failure
	*
	* @note			\p mode controls the countdown timer interrupt function
	* along with \p repeat.
	* 	Pulse interrupt when \p mode = 0, irrespective of \p repeat (true or false)
	* 	Pulse interrupt when \p mode = 1 and \p repeat = true
	* 	Level interrupt when \p mode = 1 and \p repeat = false
	*/
	int timer_init(uint8_t value, bool repeat, timer_freq_t freq);

	/**
	* @brief	Read timer value
	*
	* @return	0 on success, error code on failure
	*/
	uint8_t timer_get();

	/**
	* @brief	Enable timer
	*
	* @return	0 on success, error code on failure
	*/
	int timer_start();

	/**
	* @brief	Pause timer, timer value is preserved
	*
	* @return	0 on success, error code on failure
	*/
	int timer_pause();

	/**
	* @brief	Start timer from the paused value
	*
	* @return	0 on success, error code on failure
	*/
	int timer_continue();

	/**
	* @brief	Disable timer
	*
	* @return	0 on success, error code on failure
	*/
	int timer_stop();

	/**
	* @brief	Put device into data retention mode
	*
	* @return	0 on success, error code on failure
	*/
	int data_retention_mode_enter();

	/**
	* @brief	Remove device from data retention mode
	*
	* @return	0 on success, error code on failure
	*/
	int data_retention_mode_exit();

	/**
	* @brief	Enable I2C bus timeout mechanism
	*
	* @return	0 on success, error code on failure
	*/
	int i2c_timeout_enable();

	/**
	* @brief	Disable I2C bus timeout mechanism
	*
	* @return	0 on success, error code on failure
	*/
	int i2c_timeout_disable();

	/**
	* @brief		Enable interrupt
	*
	* @param[in]	id Interrupt id, one of INTR_ID_*
	*
	* @return		0 on success, error code on failure
	*/
	int irq_enable(intr_id_t id);

	/**
	* @brief		Disable interrupt
	*
	* @param[in]	id Interrupt id, one of INTR_ID_*
	*
	* @return		0 on success, error code on failure
	*/
	int irq_disable(intr_id_t id);

	/**
	* @brief	Disable all interrupts
	*
	* @return	0 on success, error code on failure
	*/
	int irq_disable_all();

	/**
	* @brief		Set interrupt handler for a specific interrupt id
	*
	* @param[in]	id Interrupt id, one of INTR_ID_*
	* @param[in]	func Interrupt handler function
	* @param[in]	cb Interrupt handler data
	*/
	void set_intr_handler(intr_id_t id, interrupt_handler_function func, void *cb);

	/**
	* @brief	Put device into reset state
	*
	* @return 	0 on success, error code on failure
	*/
	int sw_reset_assert();

	/**
	* @brief	Release device from state state
	*
	* @return	0 on success, error code on failure
	*/
	int sw_reset_release();

	/**
	* @brief	Enable the RTC oscillator
	*
	* @return	0 on success, error code on failure
	*/
	int rtc_start();

	/**
	* @brief	Disable the RTC oscillator
	*
	* @return	0 on success, error code on failure
	*/
	int rtc_stop();

	/**
	* @brief	Base class destructor.
	*/
	~RtcBase();

protected:
	typedef struct {
		uint8_t config_reg1;
		uint8_t config_reg2;
		uint8_t int_ploarity_config;
		uint8_t timer_config;
		uint8_t int_en_reg;
		uint8_t int_status_reg;
		uint8_t seconds;
		uint8_t minutes;
		uint8_t hours;
		uint8_t day;
		uint8_t date;
		uint8_t month;
		uint8_t year;
		uint8_t alm1_sec;
		uint8_t alm1_min;
		uint8_t alm1_hrs;
		uint8_t alm1day_date;
		uint8_t alm1_mon;
		uint8_t alm1_year;
		uint8_t alm2_min;
		uint8_t alm2_hrs;
		uint8_t alm2day_date;
		uint8_t timer_count;
		uint8_t timer_init;
		uint8_t ram_start;
		uint8_t ram_end;
		uint8_t pwr_mgmt_reg;
		uint8_t trickle_reg;
		uint8_t clock_sync_delay;
		uint8_t temp_msb;
		uint8_t temp_lsb;
		uint8_t ts_config;
	} regmap_t;

	typedef struct {
		union {
			unsigned char raw;
			struct {
				unsigned char seconds : 4;	/**< RTC seconds value. */
				unsigned char sec_10  : 3;	/**< RTC seconds in multiples of 10 */
				unsigned char         : 1;
			} bits;
			struct {
				unsigned char value   : 7;
				unsigned char         : 1;
			} bcd;
		} seconds;

		union {
			unsigned char raw;
			struct {
				unsigned char minutes : 4;	/**< RTC minutes value */
				unsigned char min_10  : 3;	/**< RTC minutes in multiples of 10 */
				unsigned char         : 1;
			} bits;
			struct {
				unsigned char value   : 7;
				unsigned char         : 1;
			} bcd;
		} minutes;

		union {
			unsigned char raw;
			struct {
				unsigned char hour       : 4;	/**< RTC hours value */
				unsigned char hr_10      : 2;	/**< RTC hours in multiples of 10 */
				unsigned char            : 2;
			} bits;
			struct {
				unsigned char value      : 6;
				unsigned char            : 2;
			} bcd;
		} hours;

		union {
			unsigned char raw;
			struct {
				unsigned char day : 3;	/**< RTC days */
				unsigned char     : 5;
			} bits;
			struct {
				unsigned char value : 3;
				unsigned char       : 5;
			} bcd;
		} day;

		union {
			unsigned char raw;
			struct {
				unsigned char date    : 4;	/**< RTC date */
				unsigned char date_10 : 2;	/**< RTC date in multiples of 10 */
				unsigned char         : 2;
			} bits;
			struct {
				unsigned char value   : 6;
				unsigned char         : 2;
			} bcd;
		} date;

		union {
			unsigned char raw;
			struct {
				unsigned char month    : 4;	/**< RTC months */
				unsigned char month_10 : 1;	/**< RTC month in multiples of 10 */
				unsigned char          : 2;
				unsigned char century  : 1;	/**< Century bit */
			} bits;
			struct {
				unsigned char value   : 5;
				unsigned char         : 3;
			} bcd;
		} month;

		union {
			unsigned char raw;
			struct {
				unsigned char year    : 4;	/**< RTC years */
				unsigned char year_10 : 4;	/**< RTC year multiples of 10 */
			} bits;
			struct {
				unsigned char value   : 8;
			} bcd;
		} year;
	} rtc_time_regs_t;

	typedef struct {
		union {
			unsigned char raw;
			struct {
				unsigned char seconds : 4;	/**< Alarm1 seconds */
				unsigned char sec_10  : 3;	/**< Alarm1 seconds in multiples of 10 */
				unsigned char a1m1    : 1;	/**< Alarm1 mask bit for minutes */
			} bits;
			struct {
				unsigned char value   : 7;
				unsigned char         : 1;
			} bcd;
		} sec;

		union {
			unsigned char raw;
			struct {
				unsigned char minutes : 4;	/**< Alarm1 minutes */
				unsigned char min_10  : 3;	/**< Alarm1 minutes in multiples of 10 */
				unsigned char a1m2    : 1;	/**< Alarm1 mask bit for minutes */
			} bits;
			struct {
				unsigned char value   : 7;
				unsigned char         : 1;
			} bcd;
		} min;

		union {
			unsigned char raw;
			struct {
				unsigned char hour       : 4;	/**< Alarm1 hours */
				unsigned char hr_10      : 2;	/**< Alarm1 hours in multiples of 10 */
				unsigned char            : 1;
				unsigned char a1m3       : 1;	/**< Alarm1 mask bit for hours */
			} bits;
			struct {
				unsigned char value      : 6;
				unsigned char            : 2;
			} bcd;
		} hrs;

		union {
			unsigned char raw;
			struct {
				unsigned char day_date : 4;	/**< Alarm1 day/date */
				unsigned char date_10  : 2;	/**< Alarm1 date in multiples of 10 */
				unsigned char dy_dt    : 1;
				unsigned char a1m4     : 1;	/**< Alarm1 mask bit for day/date */
			} bits;
			struct {
				unsigned char value   : 3;
				unsigned char         : 5;
			} bcd_day;
			struct {
				unsigned char value   : 6;
				unsigned char         : 2;
			} bcd_date;
		} day_date;

		union {
			unsigned char raw;
			struct {
				unsigned char month    : 4;	/**< Alarm1 months */
				unsigned char month_10 : 1;	/**< Alarm1 months in multiples of 10 */
				unsigned char          : 1;
				unsigned char a1m6     : 1;	/**< Alarm1 mask bit for year */
				unsigned char a1m5     : 1;	/**< Alarm1 mask bit for month */
			} bits;
			struct {
				unsigned char value   : 5;
				unsigned char         : 3;
			} bcd;
		} mon;

		union {
			unsigned char raw;
			struct {
				unsigned char year    : 4;	/* Alarm1 years */
				unsigned char year_10 : 4;	/* Alarm1 multiples of 10 */
			} bits;
			struct {
				unsigned char value   : 8;
			} bcd;
		} year;
	} alarm_regs_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char swrstn : 1;	/**< Software reset */
			unsigned char rs     : 2;	/**< Square wave output frequency selection on CLKOUT pin */
			unsigned char osconz : 1;	/**< Oscillator is on when set to 0. Oscillator is off when set to 1. */
			unsigned char clksel : 2;	/**< Selects the CLKIN frequency */
			unsigned char intcn  : 1;	/**< Interrupt control bit. Selects the direction of INTB/CLKOUT */
			unsigned char eclk   : 1;	/**< Enable external clock input */
		} bits;
	} config_reg1_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char             : 1;
			unsigned char set_rtc     : 1;	/**< Set RTC */
			unsigned char rd_rtc      : 1;	/**< Read RTC. */
			unsigned char i2c_timeout : 1;	/**< I2C timeout Enable */
			unsigned char bref        : 2;	/**< BREF sets the analog comparator threshold voltage. */
			unsigned char data_reten  : 1;	/**< Sets the circuit into data retention mode. */
			unsigned char             : 1;
		} bits;
	} config_reg2_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char tfs    : 2;	/**< Timer frequency selection */
			unsigned char trpt   : 1;	/**< Timer repeat mode. It controls the timer interrupt function along with TM. */
			unsigned char        : 1;
			unsigned char te     : 1;	/**< Timer enable */
			unsigned char tpause : 1;	/**< Timer Pause.*/
			unsigned char        : 2;
		} bits;
	} timer_config_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char d_mode      : 2;	/**< Sets the mode of the comparator to one of the two following: comparator mode, and power management mode/trickle charger mode. */
			unsigned char d_man_sel   : 1;	/**< Default low. When this bit is low, input control block decides which supply to use. When this bit is high, user can manually select whether to use V<sub>CC</sub> or VBACKUP as supply. */
			unsigned char d_vback_sel : 1;	/**< : Default low. When this bit is low, and D_MANUAL_SEL is high, V<sub>CC</sub> is switched to supply. When this bit is high, and D_MANUAL_SEL is high, V<sub>BACKUP</sub> is switched to supply. */
			unsigned char             : 4;
		} bits;
	} pwr_mgmt_reg_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char sync_delay : 2; /* Sync delay to take for the internal countdown chain to reset after the rising edge of Set_RTC */
			unsigned char            : 6;
		} bits;
	} clock_sync_reg_t;

	typedef union {
		unsigned char raw;
		struct {
			unsigned char automode    : 1;	/**< Automatic mode of temperature measurement. This mode is valid only when ONESHOTMODE=0. */
			unsigned char oneshotmode : 1;	/**< One-shot user requested temp measurement in real-time. AUTOMODE must be 0 in one-shot measurement mode. */
			unsigned char ttint       : 3;	/**< Set temp measurement interval to specified time for auto mode */
			unsigned char             : 3;
		} bits;
	} ts_config_t;

	static const uint8_t REG_NOT_AVAILABLE = 0xFF;

	/**
	* @brief	Base class constructor.
	*
	* @param[in]	regmap Pointer to device register mappings.
	* @param[in]	i2c Pointer to I2C bus object for this device.
	* @param[in]	inta_pin MCU's pin number that device's INTA pin connected
	* @param[in]	intb_pin MCU's pin number that device's INTB pin connected
	*/
	RtcBase(const regmap_t *regmap, I2C *i2c, PinName inta_pin, PinName intb_pin);

private:
	/* PRIVATE TYPE DECLARATIONS */

	/* PRIVATE VARIABLE DECLARATIONS */
	I2C *i2c_handler;
	InterruptIn *inta_pin;
	InterruptIn *intb_pin;
	const regmap_t *regmap;

	/* PRIVATE CONSTANT VARIABLE DECLARATIONS */
	static const uint8_t I2C_WRITE = 0;
	static const uint8_t I2C_READ = 1;
	static const uint8_t MAX3134X_I2C_ADDRESS = 0x69;
	static const uint8_t MAX3134X_I2C_W = ((MAX3134X_I2C_ADDRESS << 1) | I2C_WRITE);
	static const uint8_t MAX3134X_I2C_R = ((MAX3134X_I2C_ADDRESS << 1) | I2C_READ);

	enum config_reg2_set_rtc {
		CONFIG_REG2_SET_RTC_RTCRUN = 0,	/**< Setting this bit to zero doesn't allow to write into the RTC */
		CONFIG_REG2_SET_RTC_RTCPRGM,	/**< This bit must be set to one, before writing into the RTC. i.e to set the initial time for the RTC this bit must be high. */
	};

	/* PRIVATE FUNCTION DECLARATIONS */
	void interrupt_handler();

	void (RtcBase::*funcptr)(void);

	void post_interrupt_work();

	Thread *post_intr_work_thread;

	struct handler {
		void (*func)(void *);
		void *cb;
	};

	handler interrupt_handler_list[INTR_ID_END];

	int tm_hour_to_rtc_hr12(int hours, bool *is_am);

	int rtc_hr12_to_tm_hour(int hours, bool is_am);

	int rtc_regs_to_time(struct tm *time, const rtc_time_regs_t *regs);

	int time_to_rtc_regs(rtc_time_regs_t *regs, const struct tm *time);

	int set_alarm_regs(alarm_no_t alarm_no, const alarm_regs_t *regs);

	int get_alarm_regs(alarm_no_t alarm_no, alarm_regs_t *regs);

	int time_to_alarm_regs(alarm_regs_t &regs, const struct tm *alarm_time);

	int alarm_regs_to_time(struct tm *alarm_time, const alarm_regs_t *regs);

	int set_alarm_period(alarm_no_t alarm_no, alarm_regs_t &regs, alarm_period_t period);

	int set_rtc_time();

	int data_retention_mode_config(int state);

	int i2c_timeout_config(int enable);

	int set_clock_sync_delay(sync_delay_t delay);
};

#endif /* RTCBASE_H_ */
