/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
 
#ifndef __LOG_CHARACTERISTIC_H__
#define __LOG_CHARACTERISTIC_H__

#ifndef FIFO_SIZE
#define FIFO_SIZE               768
#endif

// 92e53a24-c294-406c-b64b-b401ae2e9982
static const uint8_t LogCharUUID[] = { 0x82,0x99,0x2e,0xae,0x01,0xb4,0x4b,0xb6,0x6c,0x40,0x94,0xc2,0x24,0x3a,0xe5,0x92 };

class LogCharacteristic
{
public:

    typedef struct {
        float value;
        time_t time;
    } qElement_t;

    LogCharacteristic(void) :
        numValues(0),
        gattChar(LogCharUUID, (uint8_t*)&numValues, sizeof(numValues), 
            sizeof(numValues), GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ)
    {
        clear();
    }

    GattCharacteristic* getChar(void)
    {
        return &gattChar;
    }

    /**
     * @brief   Clear the log
     */
    void clear(void) {
        // atomic FIFO access
        __disable_irq();

        rindex = 0;
        windex = 0;
        numValues = 0;

        __enable_irq();
    }

    /**
     * @brief   Add a new element to the log
     * @param   element to add to the log
     */
    void put(qElement_t element) {

        // Check if FIFO is full
        if (numValues == FIFO_SIZE) {
            
            fifoReduce();
        }

        // atomic FIFO access
        __disable_irq();

        // Put data into FIFO
        element_fifo[windex] = element;

        // Increment pointer
        windex++;
        if (windex == FIFO_SIZE) {
            windex = 0;
        }

        numValues++;

        __enable_irq();
    }

    /**
     * @brief   Get the oldest element from the log
     * @return  element from the log
     */
    qElement_t get(void) {
        qElement_t element = {0};

        // Check if empty
        if (numValues == 0) {
            return element;
        }

        // atomic FIFO access
        __disable_irq();

        // Get data from FIFO
        element = element_fifo[rindex];

        // Increment pointer
        rindex++;
        if (rindex == FIFO_SIZE) {
            rindex = 0;
        }

        numValues--;

        __enable_irq();

        return element;
    }

    /**
     * @brief   Get the length of the log
     * @return  length of the log
     */
    int len(void) {
        return numValues;
    }

private:

    // Remove every other element from the FIFO
    void fifoReduce(void)
    {   
        // atomic FIFO access
        __disable_irq();

        // i is the number of reductions
        int i = 0;
        while(i < (FIFO_SIZE/2)){

            // Position of new value has not rolled over
            if(rindex+i < FIFO_SIZE) {

                // Position of reduced value has not rolled over
                if(rindex+2*i < FIFO_SIZE) {
                    element_fifo[rindex+i] = element_fifo[rindex+2*i];
                } 

                // Position of reduced value has rolled over
                else {
                    element_fifo[rindex+i] = element_fifo[rindex+2*i-FIFO_SIZE];
                }
            } 

            // Position of new value has rolled over
            else {
                element_fifo[rindex+i-FIFO_SIZE] = element_fifo[rindex+2*i-FIFO_SIZE];
            }

            i++;
        }

        if(rindex < FIFO_SIZE/2) {
            windex = rindex + FIFO_SIZE/2;
        } else {
            windex = rindex - FIFO_SIZE/2;
        }

        numValues = FIFO_SIZE/2;

        __enable_irq();
    }

    uint16_t            numValues;
    GattCharacteristic  gattChar;
    
    // FIFO variables
    unsigned int rindex;
    unsigned int windex;
    qElement_t element_fifo[FIFO_SIZE];
};

#endif /* __LOG_CHARACTERISTIC_H__ */
