/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef __BLE_HUMIDITYCHAR_H__
#define __BLE_HUMIDITYCHAR_H__

#include "Characteristic.h"
#include "UUID.h"

/**
 * @class   HumidityChar
 * @brief   HumidityChar characteristic class.
 * @details Sub class from the characteristic class.
 */
class HumidityChar : public Characteristic
{
public:
    static const unsigned OFFSET_OF_FLAGS = 0;
    static const unsigned OFFSET_OF_VALUE = OFFSET_OF_FLAGS + 1;
    static const unsigned OFFSET_OF_TIME  = OFFSET_OF_VALUE + 4;
    static const unsigned CHAR_BYTES = (1 + 4 + TIMESTAMP_BYTES);

    static const unsigned TIMESTAMP_FLAG_POS    = 0;
    static const unsigned TIMESTAMP_FLAG_MSK    = (0x1 << TIMESTAMP_FLAG_POS);

    HumidityChar() : 
        gattChar(charUUID, bytes,CHAR_BYTES, CHAR_BYTES,  
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | 
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
    {
        // Default no timestamp and zero for the value
        memset(bytes, 0x0, CHAR_BYTES);
    }

    ~HumidityChar() {}

    virtual GattCharacteristic *getChar(void)
    {
        return &gattChar;
    }

    virtual uint8_t getNumBytes(void)
    {
        return CHAR_BYTES;
    }

    virtual uint8_t *getBytes(void)
    {
        return bytes;
    }

    // Update the characteristic value
    void update(float value, time_t time = 0)
    {
        // Convert value to bytes
        uint32_t value_bytes = float_bytes(value, 2);
        memcpy(&bytes[OFFSET_OF_VALUE], &value_bytes, sizeof(uint32_t));

        if (time != 0) {
            // Set timestamp flag
            bytes[OFFSET_OF_FLAGS] |= TIMESTAMP_FLAG_MSK;

            // Convert time_t
            date_time(time, &bytes[OFFSET_OF_TIME]);
        }
        else {
            // Clear timestamp flag
            bytes[OFFSET_OF_FLAGS] &= ~TIMESTAMP_FLAG_MSK;
        }
    }

private:
    GattCharacteristic      gattChar;
    uint8_t                 bytes[CHAR_BYTES];
    static const uint8_t    charUUID[];
};

// cec09431-a425-4c67-a1ab-230f18022708
const uint8_t HumidityChar::charUUID[] = {0x08,0x27,0x02,0x18,0x0f,0x23,0xab,0xa1,0x67,0x4c,0x25,0xa4,0x31,0x94,0xc0,0xce};

#endif /* #ifndef __BLE_HUMIDITYCHAR_H__*/
