/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef __BLE_CURRENTTIMECHAR_H__
#define __BLE_CURRENTTIMECHAR_H__

#include "mbed.h"
#include "Characteristic.h"

/**
 * @class   CurrentTimeChar
 * @brief   CurrentTimeChar characteristic class.
 * @details Sub class from the characteristic class.
 */

class CurrentTimeChar : public Characteristic
{
public:

    typedef enum {
        MON = 1,
        TUE = 2,
        WED = 3,
        THU = 4,
        FRI = 5,
        SAT = 6,
        SUN = 7
    } day_of_week_t;

    static const unsigned OFFSET_OF_YEAR = 0;
    static const unsigned OFFSET_OF_MONTH = OFFSET_OF_YEAR + 2;
    static const unsigned OFFSET_OF_DAY = OFFSET_OF_MONTH + 1;
    static const unsigned OFFSET_OF_HOUR = OFFSET_OF_DAY + 1;
    static const unsigned OFFSET_OF_MIN = OFFSET_OF_HOUR + 1;
    static const unsigned OFFSET_OF_SEC = OFFSET_OF_MIN + 1;
    static const unsigned OFFSET_OF_DOW = OFFSET_OF_SEC + 1;
    static const unsigned OFFSET_OF_SSEC = OFFSET_OF_DOW + 1;
    static const unsigned OFFSET_OF_REASON = OFFSET_OF_SSEC + 1;

    static const unsigned CHAR_BYTES = OFFSET_OF_REASON + 1;

    CurrentTimeChar() : 
        gattChar(GattCharacteristic::UUID_CURRENT_TIME_CHAR, bytes, CHAR_BYTES, CHAR_BYTES,  
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | 
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE)
    {
        // Default no timestamp and zero for the value
        memset(bytes, 0x0, CHAR_BYTES);
    }

    ~CurrentTimeChar() {}

    virtual GattCharacteristic *getChar(void)
    {
        return &gattChar;
    }

    virtual uint8_t getNumBytes(void)
    {
        return CHAR_BYTES;
    }

    virtual uint8_t *getBytes(void)
    {
        return bytes;
    }

    // Update the characteristic value
    void update(time_t current_time)
    {
        // Convert time_t to date_time
        date_time(current_time, &bytes[OFFSET_OF_YEAR]);

        // Set SSec to 0
        bytes[OFFSET_OF_SSEC] = 0;

        // Set day of the week
        struct tm ts = *localtime(&current_time);
        if(ts.tm_wday == 0) {
            bytes[OFFSET_OF_DOW] = 7;
        } else {
            bytes[OFFSET_OF_DOW] = ts.tm_wday;
        }
    }

    // Update the characteristic value with the RTC
    void update(void)
    {
        update(time(NULL));
    }

    // Get the characteristic value
    time_t get_value(void)
    {
        struct tm ts;
        ts.tm_year = (bytes[OFFSET_OF_YEAR + 1] << 8) + 
            bytes[OFFSET_OF_YEAR] - 1900;

        ts.tm_mon = bytes[OFFSET_OF_MONTH] - 1;
        ts.tm_mday = bytes[OFFSET_OF_DAY];
        ts.tm_hour = bytes[OFFSET_OF_HOUR];
        ts.tm_min = bytes[OFFSET_OF_MIN];
        ts.tm_sec = bytes[OFFSET_OF_SEC];

        return mktime(&ts);
    }

    // Update the RTC with the characteristic value
    void update_rtc()
    {
        set_time(this->get_value());
    }

private:
    GattCharacteristic      gattChar;
    uint8_t                 bytes[CHAR_BYTES];
};

#endif /* #ifndef __BLE_CURRENTTIMECHAR_H__*/
