/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef __BLE_REFERENCETIMEINFOCHAR_H__
#define __BLE_REFERENCETIMEINFOCHAR_H__

#include "mbed.h"
#include "Characteristic.h"

/**
 * @class   ReferenceTimeInfoChar
 * @brief   ReferenceTimeInfoChar characteristic class.
 * @details Sub class from the characteristic class.
 */

class ReferenceTimeInfoChar : public Characteristic
{
public:

    typedef enum {
        unknown = 0,
        network_time,
        gps,
        radio_time,
        manual,
        atomic_clock,
        cellular_network
    } source_t;

    static const unsigned OFFSET_OF_SOURCE = 0;
    static const unsigned OFFSET_OF_ACCURACY = OFFSET_OF_SOURCE + 1;
    static const unsigned OFFSET_OF_DAYS = OFFSET_OF_ACCURACY + 1;
    static const unsigned OFFSET_OF_HOURS = OFFSET_OF_DAYS + 1;

    static const unsigned CHAR_BYTES = OFFSET_OF_HOURS + 1;

    static const unsigned ACCURACY_OUT_OF_RANGE = 254;
    static const unsigned ACCURACY_UNKNOWN = 255;

    ReferenceTimeInfoChar() : 
        gattChar(GattCharacteristic::UUID_REFERENCE_TIME_INFORMATION_CHAR, bytes, 
            CHAR_BYTES, CHAR_BYTES,  
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ)
    {
        memset(bytes, 0x0, CHAR_BYTES);
        bytes[OFFSET_OF_ACCURACY] = ACCURACY_UNKNOWN;
        last_update = time(NULL);
    }

    ~ReferenceTimeInfoChar() {}

    virtual GattCharacteristic *getChar(void)
    {
        return &gattChar;
    }

    virtual uint8_t getNumBytes(void)
    {
        return CHAR_BYTES;
    }

    virtual uint8_t *getBytes(void)
    {
        return bytes;
    }

    // Update the characteristic value for reference source
    void updateSource(source_t source)
    {
        bytes[OFFSET_OF_SOURCE] = source;
    }

    // Update the characteristic value for time accuracy
    void updateAccuracy(uint8_t accuracy)
    {
        bytes[OFFSET_OF_ACCURACY] = accuracy;
    }

    // Update the characteristic value for time zone
    void updateUpdateTime()
    {
        // Calculate how many hours since last update
        uint32_t hours = (time(NULL)-last_update)/360;

        // Days is zero
        if(hours < 24) {
            bytes[OFFSET_OF_HOURS] = hours;
            bytes[OFFSET_OF_DAYS] = 0;
            return;
        }

        uint16_t days = hours/24;

        // Days overflow
        if(days > 255) {
            bytes[OFFSET_OF_DAYS] = 255;
            bytes[OFFSET_OF_HOURS] = 255;
            return;
        }

        // Days and hours since update
        hours -= (days*24);
        bytes[OFFSET_OF_DAYS] = days;
        bytes[OFFSET_OF_HOURS] = hours;
    }

    void resetUpdateTime()
    {
        last_update = time(NULL);
        bytes[OFFSET_OF_DAYS] = 0;
        bytes[OFFSET_OF_HOURS] = 0;
    }

private:
    GattCharacteristic      gattChar;
    uint8_t                 bytes[CHAR_BYTES];
    time_t                  last_update;
};

#endif /* #ifndef __BLE_REFERENCETIMEINFOCHAR_H__*/
