/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef __BLE_ENVIRONMENTAL_SERVICE_H__
#define __BLE_ENVIRONMENTAL_SERVICE_H__

#include <time.h>
#include "BLE.h"
#include "Characteristic.h"
#include "HumidityChar.h"
#include "PressureChar.h"
#include "TemperatureChar.h"
#include "LogCharacteristic.h"

// db8d3576-de54-4b92-8754-8a99e7a18d6a
static const uint8_t envServiceUUID[] = {0x6a,0x8d,0xa1,0xe7,0x99,0x8a,0x54,
    0x87,0x92,0x4b,0x54,0xde,0x76,0x35,0x8d,0xdb};

/**
 * @class   EnvironmentalService
 * @brief   BLE Environmental sensor data service.
 * @details Custom service for communicating environmental sensor data.
 */
class EnvironmentalService
{
public:

    /**
     * @brief   EnvironmentalService constructor.
     * @param   ble Reference to BLE device.
     * @param   pres_unit Unit for the pressure measurements.
     * @param   temp_unit Unit for the temperature measurements.
     */
    EnvironmentalService(BLEDevice &_ble, PressureChar::pres_unit_t pres_unit, 
            TemperatureChar::temp_unit_t temp_unit) :
        ble(_ble),
        humidity(), 
        logHumidity(),
        pressure(pres_unit),
        logPressure(),
        temperature(temp_unit),
        logTemperature()
    {
        GattCharacteristic *envChars[] = {  humidity.getChar(),
                                            logHumidity.getChar(),
                                            pressure.getChar(),
                                            logPressure.getChar(),
                                            temperature.getChar(),
                                            logTemperature.getChar() };

        GattService envService(envServiceUUID, envChars, 
            sizeof(envChars) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(envService);
        ble.gattServer().onDataRead(this, &EnvironmentalService::dataReadCallback);
    }

    /**
     * @brief   Update humidity characteristic.
     * @param   value New humidity measurement.
     * @param   time Timestamp to send with temperature measurement.
     */
    void updateHumidity(float value, time_t time = 0) 
    {
        bool enabledP;

        if (ble.gattServer().areUpdatesEnabled(*humidity.getChar(), &enabledP) == BLE_ERROR_NONE) {
            if (enabledP) {
                // Notifications are enabled. Write the value to the attribute and send the notification
                humidity.update(value, time);
                ble.gattServer().write(humidity.getChar()->getValueHandle(),
                                       humidity.getBytes(), humidity.getNumBytes());

                /* A log is not kept while notifications are enabled. However, the latest value will
                 * be kept in case notifications are disabled before the value was sent.
                 */
                logHumidity.clear();
            }
        }

        // Write the value to the log
        LogCharacteristic::qElement_t element = {value, time};
        logHumidity.put(element);
    }

    /**
     * @brief   Update pressure characteristic.
     * @param   value New pressure measurement.
     * @param   time Timestamp to send with temperature measurement.
     */
    void updatePressure(float value, time_t time = 0) 
    {
        bool enabledP;

        if (ble.gattServer().areUpdatesEnabled(*pressure.getChar(), &enabledP) == BLE_ERROR_NONE) {
            if (enabledP) {
                // Notifications are enabled. Write the value to the attribute and send the notification
                pressure.update(value, time);
                ble.gattServer().write(pressure.getChar()->getValueHandle(),
                                       pressure.getBytes(), pressure.getNumBytes());

                /* A log is not kept while notifications are enabled. However, the latest value will
                 * be kept in case notifications are disabled before the value was sent.
                 */
                logPressure.clear();
            }
        }

        // Write the value to the log
        LogCharacteristic::qElement_t element = {value, time};
        logPressure.put(element);
    }

    /**
     * @brief   Update temperature characteristic.
     * @param   temperature New temperature measurement.
     * @param   time Timestamp to send with temperature measurement.
     */
    void updateTemperature(float value, time_t time = 0) 
    {
        bool enabledP;

        if (ble.gattServer().areUpdatesEnabled(*temperature.getChar(), &enabledP) == BLE_ERROR_NONE) {
            if (enabledP) {
                // Notifications are enabled. Write the value to the attribute and send the notification
                temperature.update(value, time);
                ble.gattServer().write(temperature.getChar()->getValueHandle(),
                                       temperature.getBytes(), temperature.getNumBytes());

                /* A log is not kept while notifications are enabled. However, the latest value will
                 * be kept in case notifications are disabled before the value was sent.
                 */
                logTemperature.clear();
            }
        }

        // Write the value to the log
        LogCharacteristic::qElement_t element = {value, time};
        logTemperature.put(element);
    }

private:

    /**
     * @brief   Called when a characteristics value is being read
     */
    void dataReadCallback(const GattReadCallbackParams *eventDataP)
    {
        if (eventDataP->handle == humidity.getChar()->getValueHandle()) {
            if (logHumidity.len() > 0) {
                // Get the next value from the log and update the characteristic value
                LogCharacteristic::qElement_t element;
                element = logHumidity.get();
                humidity.update(element.value, element.time);
            }
            // The log is empty, leave the last characteristic value in place
        } else if (eventDataP->handle == pressure.getChar()->getValueHandle()) {
            if (logPressure.len() > 0) {
                // Get the next value from the log and update the characteristic value
                LogCharacteristic::qElement_t element;
                element = logPressure.get();
                pressure.update(element.value, element.time);
            }
            // The log is empty, leave the last characteristic value in place
        } else if (eventDataP->handle == temperature.getChar()->getValueHandle()) {
            if (logTemperature.len() > 0) {
                // Get the next value from the log and update the characteristic value
                LogCharacteristic::qElement_t element;
                element = logTemperature.get();
                temperature.update(element.value, element.time);
            }
            // The log is empty, leave the last characteristic value in place
        }
    }

private:

    BLEDevice           &ble;
    HumidityChar        humidity;
    LogCharacteristic   logHumidity;
    PressureChar        pressure;
    LogCharacteristic   logPressure;
    TemperatureChar     temperature;
    LogCharacteristic   logTemperature;
};

#endif /* #ifndef __BLE_ENVIRONMENTAL_SERVICE_H__*/
