/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
 
#ifndef __BLE_CURRENTTIMESERVICE_H__
#define __BLE_CURRENTTIMESERVICE_H__

#include <time.h>
#include "Characteristic.h"
#include "CurrentTimeChar.h"
#include "ReferenceTimeInfoChar.h"
#include "LocalTimeInfoChar.h"

/**
 * @class   CurrentTimeService
 * @brief   BLE Current Time service.
 * @details Standard service for communicating time information.
 */
class CurrentTimeService
{
public:

    /**
     * @brief   CurrentTimeSerivce constructor.
     * @param   ble Reference to BLE device.
     */
    CurrentTimeService(BLEDevice &_ble) :
        ble(_ble),
        currentTime(),
        localTimeInfo(),
        referenceTimeInfo()
    {

        GattCharacteristic *currentTimeChars[] = {currentTime.getChar(), localTimeInfo.getChar(),
            referenceTimeInfo.getChar()};

        GattService currentTimeService(GattService::UUID_CURRENT_TIME_SERVICE, currentTimeChars, 
            sizeof(currentTimeChars) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(currentTimeService);
        ble.gattServer().onDataWritten(this, &CurrentTimeService::onDataWritten);
        ble.gattServer().onDataRead(this, &CurrentTimeService::onDataRead);
    }

    // Update the value in the current time characteristic from RTC
    void updateCurrentTimeValue()
    {
        currentTime.update();
        ble.gattServer().write(currentTime.getChar()->getValueAttribute().getHandle(),
                               currentTime.getBytes(), currentTime.getNumBytes());
    }

private:

    void onDataWritten(const GattWriteCallbackParams *params)
    {

        // Check to see which value is being written
        if (params->handle == currentTime.getChar()->getValueAttribute().
            getHandle()) {

            // Update RTC from CurrentTimeChar
            currentTime.update_rtc();

            // Reset the time since update
            referenceTimeInfo.resetUpdateTime();
        }
    }

    void onDataRead(const GattReadCallbackParams *eventDataP)
    {
        if (eventDataP->handle == currentTime.getChar()->getValueAttribute().getHandle()) {
            // Update the characteristic value with the rtc
            currentTime.update();

        } else if (eventDataP->handle == referenceTimeInfo.getChar()->getValueAttribute().getHandle()) {
            // Update time since update
            referenceTimeInfo.updateUpdateTime();
        }
    }

    BLEDevice               &ble;
    CurrentTimeChar         currentTime;
    LocalTimeInfoChar       localTimeInfo;
    ReferenceTimeInfoChar   referenceTimeInfo;
};

#endif /* #ifndef __BLE_CURRENTTIMESERVICE_H__*/
