/*******************************************************************************
* Author: Ismail Kose, Ismail.Kose@maximintegrated.com
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

/*
 * TODO:
 * Add a function to enqueue data block instead of one by one.
 * Write function definitions in the header file as doxygen format
 * Init function will also allocate memory for queue buffer, providing the buffer will not necessary
 *
 * */

#define enter_critical_section()
#define exit_critical_section()

#include "queue.h"
int queue_reset(struct queue_t *q)
{
	if (!q)
		return -1;

	q->wr = q->base;
	q->rd = q->base;
	q->num_item = 0;
	q->ovf_item = 0;

	return 0;
}

int queue_init(struct queue_t *q, void *buf, int item_size, int buffer_size)
{
	if (!q || !buf)
		return -1;

	if (buffer_size % item_size != 0)
		return -1; // Padding problem

	q->num_item = 0;
	q->base = buf;
	q->wr = buf;
	q->rd = buf;
	q->item_size = item_size;
	q->buffer_size = buffer_size;

	return 0;
}

void queue_destroy(struct queue_t *q)
{
/* TODO: This is placeholder function, double check the implementation */
	free((void *)q->base);
	free((void *)q);
}

int enqueue(struct queue_t *q, void *data)
{
	int ret = 0;

	if (!q || !data)
		return -1; // Invalid pointer

	enter_critical_section();
	if (q->wr == q->rd)
		ret = (q->num_item != 0) ? -2 : 0; // Is FIFO Full or Empty?

	if (((uint32_t)q->wr) >= ((uint32_t)q->base + q->buffer_size))
		q->wr = q->base;

	memcpy((void *)q->wr, data, q->item_size);
	q->wr = (void *)((uint32_t)q->wr + q->item_size);
	q->num_item++;
	exit_critical_section();
	return ret;
}

int dequeue(struct queue_t *q, void *data)
{
	int fifo_size;

	if (!q || !data)
		return -1;

	fifo_size = q->buffer_size / q->item_size;

	enter_critical_section();
	if (q->num_item <= 0) {
		exit_critical_section();
		return -2;
	}

	if (q->num_item > fifo_size) {
		uint32_t curr_rd_off = (((uint32_t)q->rd - (uint32_t)q->base) + q->num_item * q->item_size);
		q->ovf_item = q->num_item - fifo_size;
		q->rd = (void *)((uint32_t)q->base + (curr_rd_off % q->buffer_size));
		q->num_item = fifo_size; // OVF number samples are already gone.
		printf("%s:%d - %d samples lost\n", __func__, __LINE__, q->ovf_item);
	} else
		q->ovf_item = 0;

	if (((uint32_t)q->rd) >= ((uint32_t)q->base + q->buffer_size))
		q->rd = q->base;

	memcpy(data, (void *)q->rd, q->item_size);
	q->rd = (void *)((uint32_t)q->rd + q->item_size);
	q->num_item--;
	exit_critical_section();

#if defined(QUEUE_DEBUG)
	do {
		static int cnt;

		if (cnt++ % 100 == 0)
			printf("$ Fifo size: %d, usage: %d\n", fifo_size, q->num_item);
	} while(0);
#endif

	return 0;
}

